/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.core;

import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.util.Assert;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.IRuntimeDefinition;
import org.eclipse.stp.soas.deploy.core.IServerDefinition;
import org.eclipse.stp.soas.deploy.core.ISupportedTechnologyType;
import org.eclipse.stp.soas.deploy.core.ITechnologyMap;
import org.eclipse.stp.soas.deploy.core.IVersion;
import org.eclipse.stp.soas.deploy.core.MatchRule;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.Version;


/**
 * @author rcernich
 * 
 * Created on Apr 8, 2004
 */
public class TechnologyMap implements ITechnologyMap {

	public static final String ATTR_TYPE = "type"; //$NON-NLS-1$
	public static final String ATTR_VERSION = "version"; //$NON-NLS-1$
	public static final String ATTR_MATCH_RULE = "matchRule"; //$NON-NLS-1$

	private IConfigurationElement mElement;
	private String mServerDef;
	private IVersion mServerVersion;
	
	private String mRuntimeDef;
	private IVersion mRuntimeVersion;
	
	private MatchRule mMatchRule;
	private SortedMap mSupportedTechnologiesByDef;

	public TechnologyMap(IConfigurationElement element)
			throws PackageExtensionException {
		super();
		init(element);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.ITechnologyMap#getSupportedTechnologyTypes()
	 */
	public ISupportedTechnologyType[] getSupportedTechnologyTypes() {
		return (ISupportedTechnologyType[]) mSupportedTechnologiesByDef
				.values()
				.toArray(
						new ISupportedTechnologyType[mSupportedTechnologiesByDef
								.size()]);
	}

	public IServerDefinition getServerDefinition() {
		return DeploymentExtensionManager.getInstance().getServerDefinition(
				mServerDef);
	}

	public IVersion getServerVersion() {
		return mServerVersion;
	}
	
	public IRuntimeDefinition getRuntimeDefinition() {
		return DeploymentExtensionManager.getInstance().getRuntimeDefinition(
				mRuntimeDef);
	}

	public IVersion getRuntimeVersion() {
		return mRuntimeVersion;
	}
	

	public MatchRule getVersionMatchRule() {
		return mMatchRule;
	}
	
	

	private void init(IConfigurationElement element)
			throws PackageExtensionException {
		Assert.isTrue(DeploymentExtensionManager.EXT_ELEM_TECHNOLOGY_MAP
				.equals(element.getName()));

		if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
			System.out.println(DeployCorePlugin.getDefault()
					.getResourceString(
							"TechnologyMap.trace.processingTechnologyMap", //$NON-NLS-1$
							new Object[] { element.getDeclaringExtension()
									.getContributor().getName()}));
			System.out.flush();
		}

		mElement = element;

		processAttributes();
		processSupportedTechnologyElements();

		((ServerDefinition) getServerDefinition()).addTechnologyMap(this);
	}

	private void processAttributes() throws PackageExtensionException {
		mServerDef = mElement.getAttribute(ATTR_TYPE);
		mServerVersion = Version.valueOf(mElement.getAttribute(ATTR_VERSION));
		String matchRuleAttr = mElement.getAttribute(ATTR_MATCH_RULE);
		if (matchRuleAttr == null
				|| MatchRule.valueOf(matchRuleAttr.trim()) == null) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.out.println(DeployCorePlugin.getDefault()
						.getResourceString(
								"TechnologyMap.trace.matchRuleNotSpecified")); //$NON-NLS-1$
				System.out.flush();
			}
			mMatchRule = MatchRule.EQUIVALENT;
		}
		else {
			mMatchRule = MatchRule.valueOf(matchRuleAttr.trim());
		}

		if (mServerDef == null) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"TechnologyMap.trace.error.typeAttributeNotSpecified", //$NON-NLS-1$
										new Object[] { mElement
												.getDeclaringExtension()
												.getContributor().getName()}));
				System.err.flush();
			}
			throw new PackageExtensionException(DeployCorePlugin
					.getDefault().getResourceString(
							"TechnologyMap.exception.technologyMapParseError")); //$NON-NLS-1$
		}
		else if (DeploymentExtensionManager.getInstance().getServerDefinition(
				mServerDef) == null) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"TechnologyMap.trace.error.invalidTypeSpecified", //$NON-NLS-1$
										new Object[] { mElement
												.getDeclaringExtension()
												.getContributor().getName()}));
				System.err.flush();
			}
			throw new PackageExtensionException(DeployCorePlugin
					.getDefault().getResourceString(
							"TechnologyMap.exception.invalidTypeSpecified")); //$NON-NLS-1$
		}
	}

	private void processSupportedTechnologyElements() {
		IConfigurationElement[] supportedTechnologies = mElement
				.getChildren(DeploymentExtensionManager.EXT_ELEM_SUPPORTED_TECHNOLOGY);

		mSupportedTechnologiesByDef = new TreeMap();
		for (int index = 0, count = supportedTechnologies.length; index < count; ++index) {
			try {
				ISupportedTechnologyType istt = new SupportedTechnologyType(
						supportedTechnologies[index]);
				ISupportedTechnologyType existingIstt = (ISupportedTechnologyType) mSupportedTechnologiesByDef
						.get(istt.getTechnologyDefinition());
				if (existingIstt != null) {
					istt = new MergedSupportedTechnologyType(existingIstt, istt);
				}
				mSupportedTechnologiesByDef.put(istt.getTechnologyDefinition(),
						istt);
			}
			catch (PackageExtensionException e) {
				if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
					System.err
							.println(DeployCorePlugin
									.getDefault()
									.getResourceString(
											"TechnologyMap.trace.error.supportedTechnologyParseError", //$NON-NLS-1$
											new Object[] {
													mServerDef,
													mServerVersion,
													mElement
															.getDeclaringExtension()
															.getContributor().getName()}));
					e.printStackTrace();
					System.err.flush();
				}
			}
		}
	}

}