/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.infrastructure.emf;

import org.eclipse.core.commands.operations.IOperationHistory;
import org.eclipse.emf.ecore.resource.Resource;

/**
 * IEditModel manages a set of Resources as an atomic unit, with context
 * established through the use of one or more
 * {@link org.eclipse.stp.core.infrastructure.emf.IScribblerDomain}s which
 * declare interesting resources.
 * 
 * <p>
 * IEditModels create
 * {@link org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler} s, which
 * can be used by editors to manage IUndoContexts within the IEditModel command
 * stack, to be notified of changes in interesting resources, and to save,
 * revert or close the resources as an atomic unit.
 * </p>
 * <p>
 * A project may have one or more IEditModels associated with it. Each
 * IEditModel will track resources that its IEditModelScribblers are interested
 * in; including resources that are in use by multiple IEditModelScribblers
 * concurrently. The set of resources that the IEditModelScribbler is interested
 * in is specified when the IEditModelScribbler is created.
 * {@link IEditModel#createScribbler(IScribblerDomain[], boolean)}.
 * </p>
 * <p>
 * The intended guidance for IEditModels is that each IEditModel will track a
 * set of resources associated with a particular editor or set of editors. For
 * instance, an editor for modifying an Enterprise Java Bean Deployment
 * Descriptor and an editor for modifying mapping data from the EJBs to the
 * database schema could share the same IEditModel (within the same project),
 * but each editor would have its own IEditModelScribbler. The IEditingDomains
 * of each IEditModelScribbler would be different, but would reference some of
 * the same resources (e.g. ejb-jar.xml). The single IEditModel would then
 * notify each IEditModelScribbler (one for the EJB DD editor, one for the EJB
 * Mapping Editor) whenever one of the resources for either IEditModelScribbler
 * had been modified (again, the resources are indicated by the IEditingDomains
 * contained by the IEditModelScribblers). The same project could then define
 * another IEditModel to track resources associated with "Technology XYZ" for
 * one or more (other) editors.
 * </p>
 * 
 * <p>
 * The IEditModel manages a single command stack (IOperationHistory) for all of
 * the IEditModelScribblers that are created from it. Each of the
 * IEditModelScribblers define an IUndoContext which is used when executing
 * operations. One or more IEditModelScribblers may share the same or equivalent
 * sets of IEditingDomains, in which case operations may be undone or redone
 * across the matching IEditModelScribblers.
 * </p>
 * 
 * @see org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler
 * @see org.eclipse.stp.core.infrastructure.emf.IScribblerDomain
 */
public interface IEditModel {

   /**
    * Track changes to the IEditModel including the addition or removal of
    * Resources and the loading or unloading of Resources.
    * 
    * @param aListener
    * @see EditModelEvent
    */
   void addListener(IEditModelListener aListener) throws EditModelException;

   /**
    * Remove an existing {@link IEditModelListener}.
    * 
    * @param aListener
    * @return
    */
   void removeListener(IEditModelListener aListener);

   /**
    * Respond to an errors that may occur during the IEditModelLifecycle.
    * 
    * @param errorHandler
    */
   void setErrorHandler(IEditModelErrorHandler errorHandler);

   /**
    * The IEditModel creates {@link IEditModelScribbler}s which then exposes
    * artifact lifecylce methods like <code>save()</code>,
    * <code>close()</code>, or <code>revert()</code>.
    * 
    * <p>
    * Several IEditModelScribblers may be created at the same time with either
    * mutually exclusive or intersecting {@link IScribblerDomain}s. The
    * IEditModel will track changes to all resources announced by the
    * {@link IScribblerDomain}s, and notify the {@link IEditModelScribbler}s
    * when changes occur to resources they are interested in.
    * </p>
    * 
    * <specific-request-for-input>
    * 
    * Should the IEditModelScribbler be able to contain a set of Resources that
    * have mixed read-only status? Where some are readonly and others are not?
    * 
    * </specific-request-for-input>
    * 
    * @param theDomains
    * @param makeReadOnly
    *           Indicates that the IEditModelScribbler will not be able to save
    *           or persist changes.
    * @return
    * @throws EditModelException
    * @see IEditModelScribbler
    * @see IScribblerDomain For more information on creating these objects.
    */
   IEditModelScribbler createScribbler(IScribblerDomain[] theDomains,
         boolean makeReadOnly) throws EditModelException;

   /**
    * Once an <code>IEditModel</code> has been disposed, it may no longer be
    * used; a new <code>IEditModel</code> must be loaded.
    * 
    * @return true if the Edit Model has been disposed
    */
   boolean isDisposed();

   /**
    * <i>Expected to use the IEditingDomain.isContained() API underneath; </i>
    * 
    * @param aResource
    * @return
    */
   boolean isInterrestedInResource(Resource aResource);

   /**
    * Return the command stack in use by this IEditModel. The current operation
    * history is shared by all existing IEditModelScribblers. Each
    * IEditModelScribbler becomes its own IUndoContext, so the history will
    * remember all commands executed, but the undo/redo of these operations can
    * be coordinated from equivalent IUndoContexts. The IUndoContext is
    * determined by the IEditModelScribbler, where the IEditingDomain(s) is(are)
    * the basis for matching IUndoContexts.
    * 
    * 
    * @return an IOperationHistory which is shared among all
    *         IEditModelScribblers created by this IEditModel
    * @see org.eclipse.core.commands.operations.IUndoContext
    * @see IScribblerDomain
    * @see IEditModelScribbler
    */
   IOperationHistory getOperationHistory() throws EditModelException;

   /**
    * The editModelLabel functions as a label so that clients can coordinate
    * access to a specific Edit Model across editors, operations, and views.
    * <p>
    * The editModelLabel does not imply any varied function, but merely serves
    * as a unique identifier for a specific Edit Model. In general, these ids
    * are meant to be coordinate across a particular set of related resources.
    * For instance, an EJB DD Editor might share the same underlying
    * editModelLabel that an EJB Mapping Editor would use.
    * </p>
    * 
    * @return The editModelLabel used to create or locate this Edit Model
    */
   String getEditModelLabel();

}
