/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.infrastructure.emf;

import org.eclipse.core.commands.operations.IUndoContext;
import org.eclipse.emf.ecore.resource.Resource;

/**
 * The IEditingDomain defines a set of ResourceDescriptors that it is interested
 * in. A ResourceDescriptor defines enough information to recognize a given EMF
 * Resource. When used together, IEditingDomains can create overlapping scopes
 * within a single
 * {@link org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler}.
 * 
 * <p>
 * If multiple IEditModelScribblers are created with equivalent sets of
 * IEditingDomains, then the operations will share the same IUndoContext against
 * the command stack on the IEditModel. The IUndoContext determines the ability
 * for a an operation to be undone or redone. Calls to the undo/redo methods on
 * the IOperationHistory (see
 * {@link org.eclipse.stp.core.infrastructure.emf.IEditModel#getOperationHistory()})
 * require an IUndoContext. If the IEditModelScribbler used for the call
 * (remember IEditModelScribbler is also an IUndoContext) matches another
 * instance of IEditModelScribbler with equivalent IEditingDomains, then the
 * operation from the other IEditModelScribbler will be undone.
 * </p>
 * 
 * <p>
 * The way in which IEditingDomains may be created is left open to the client of
 * the IEditModel Framework. A default implementation of IEditingDomain will be
 * provided that can be instantiated by clients. Clients can then create these
 * objects as needed (say from a user click action) or based on other extensions
 * (for instance in cases where an editor might provide extension points for
 * extension sections or pages to specify other interesting resources like in
 * the case of an EJB Editor that then picks up vendor-specific EJB-binding
 * files).
 * </p>
 * 
 * <specific-request-for-input>
 * 
 * Should IEditingDomains be allowed to change dynamically? For instance the
 * IEditModelScribbler could listen for the IEditingDomain to change its set of
 * interesting ResourceDescriptors. Adding new resources should not be an issue,
 * but removing interesting Resources from the IEditModelScribblers set of
 * IEditingDomain could present an issue.
 * 
 * IEditingDomains could change the resources they are interested in based on
 * the addition or removal of Server Features (via The generic Function Groups
 * Framework). An IEditModelScribbler would get notifications of these changes.
 * Then if a resource is removed from the "interesting set" the
 * IEditModelScribbler would use its User Interface mechanism (some form of
 * listener that can trigger dialogs for a user) to prompt the user for advice.
 * In this case, if an editor had a section or page added by a server feature
 * that was removed, the editor may prompt the user to close and restart the
 * editor (effectively a call to
 * {@link org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler#close(IProgressMonitor)})
 * which is triggered by the change in interest of the IEditingDomain.
 * 
 * 
 * </specific-request-for-input>
 */
public interface IScribblerDomain extends IUndoContext {

   /**
    * The edit model label will be used to identify the relevant edit model that
    * should be used when creating an IEditModelScribbler from this
    * IScribblerDomain.
    * 
    * The edit model label should be specific to a logical collection of
    * resources corresponding to a logical type of edit model.
    * 
    * For example, assume we have a project named "FooBar" with three logical
    * modules named "EjbM1", "EjbM2", and "WebM3" of types "ejb", "ejb", and
    * "web". Then we might choose "FooBar.EjbM1.ejb" as the edit model label for
    * the EJB edit model for FooBar/EjbM1. The WebM3 module might define two
    * relevant edit model labels, one for the "web" stuff and one for the
    * "webservice" stuff. We might label these two edit models as
    * "FooBar.WebM3.web" and "FooBar.WebM3.webservice" respectively.
    * 
    * Clients may rely on this method for establishing the specific IEditModel
    * of interest when using
    * {@link IEditModelFactory#createScribblerForRead(IProject, IScribblerDomain)}.
    * Clients must still explicitly supply an IEditModel label when using any
    * API that takes an array of IScribblerDomains.
    * 
    * @return A label for the edit model that should be used when creating
    *         IEditModelScribblers from this IScribblerDomain.
    */
   String getEditModelLabel();

   /**
    * Defines the interesting set of ResourceDescriptors for this
    * IEditingDomain.
    * 
    * @return
    */
   IResourceDescriptor[] getResourceDescriptors();

   /**
    * Returns true if the aResourceDescriptor is contained by this
    * IEditingDomain.
    * <p>
    * Replaces EditModel.isInterrestedInResource().
    * </p>
    * 
    * @param aResourceDescriptor
    * @return
    */
   boolean isContained(IResourceDescriptor aResourceDescriptor);

   /**
    * Returns true if the aResource is contained by this IEditingDomain.
    * <p>
    * Replaces EditModel.isInterrestedInResource().
    * </p>
    * 
    * @param aResourceDescriptor
    * @return
    */
   boolean isContained(Resource aResource);

   /**
    * The Technology Flavor Lifecycle will receive callbacks from the Scribbler
    * to save(), discard(), or revert() its modfications in step with the
    * scribbler's management of EMF resources.
    * 
    * @param flavorManager
    * @see ITechnologyFlavorLifecycle
    */
   void addTechnologyFlavorLifecycle(ITechnologyFlavorLifecycle flavorManager);

   /**
    * The following method should not be invoked by any client other than the
    * IEditModelScribbler.
    * 
    * @param technologyFlavorId
    * @return
    */
   ITechnologyFlavorLifecycle getTechnologyFlavorLifecycle(
         String technologyFlavorId);

   /**
    * The following method should not be invoked by any client other than the
    * IEditModelScribbler.
    * 
    * @param technologyFlavorId
    * @return
    */
   ITechnologyFlavorLifecycle[] getTechnologyFlavorLifecycles();

}
