/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.resources;

import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.infrastructure.emf.EclipseResourceDescriptor;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModelFactory;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IResourceDescriptor;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;
import org.eclipse.stp.core.internal.STPCorePlugin;
import org.eclipse.stp.core.internal.introspection.ComponentTypeIntrospectorRegistry;
import org.eclipse.stp.core.sca.Component;
import org.eclipse.stp.core.sca.Composite;
import org.eclipse.stp.core.sca.SCACoreRoot;

public class CompositeScribblerDomain extends SOACoreScribblerDomain implements
      IScribblerDomain {
   private final IFile                 composite;

   private final IResourceDescriptor[] descriptors;

   private IResourceDescriptor         compositeDescriptor;

   public CompositeScribblerDomain(IFile aComposite) {
      super(aComposite.getProject());
      Assert.isTrue(SOAConstants.COMPOSITE_EXT.equals(aComposite
            .getFileExtension())
            || SOAConstants.MODULE_EXT.equals(aComposite.getFileExtension()));
      composite = aComposite;
      compositeDescriptor = new EclipseResourceDescriptor(composite);
      descriptors = new IResourceDescriptor[] { compositeDescriptor };
   }

   public boolean isContained(Resource aResource) {
      if (super.isContained(aResource))
         return true;

      return isInterestedInURI(aResource.getURI());
   }

   public boolean isContained(IResourceDescriptor aDescriptor) {
      if (super.isContained(aDescriptor))
         return true;
      return isInterestedInURI(URI.createPlatformResourceURI(aDescriptor
            .getLocalPath().toString()));
   }

   private boolean isInterestedInURI(URI uri) {
      IEditModelScribbler scribbler = null;
      try {
         scribbler = createScribblerForRead();
         Resource resource = scribbler.getResource(getCompositeDescriptor());
         if (resource.getContents().size() > 0) {
            SCACoreRoot root = (SCACoreRoot) resource.getContents().get(0);
            Composite composite = root.getComposite();
            if (composite != null) {
               List components = composite.getComponents();
               Component component = null;
               for (int i = 0; i < components.size(); i++) {
                  component = (Component) components.get(i);
                  URI shareableURI = ComponentTypeIntrospectorRegistry
                        .getInstance().getShareableURI(
                              component.getImplementation());
                  if (uri.equals(shareableURI))
                     return true;
               }
            }
         }

      } catch (EditModelException e) {
         STPCorePlugin.logError(0, e.getMessage(), e);
      } finally {
         if (scribbler != null)
            try {
               scribbler.close(null);
            } catch (EditModelException e) {
               STPCorePlugin.logError(0, e.getMessage(), e);
            }
      }

      return false;
   }

   public IEditModelScribbler createScribblerForRead()
         throws EditModelException {
      return IEditModelFactory.eINSTANCE.createScribblerForRead(composite
            .getProject(), this);
   }

   public IEditModelScribbler createScribblerForWrite()
         throws EditModelException {
      return IEditModelFactory.eINSTANCE.createScribblerForWrite(composite
            .getProject(), this);
   }

   public IResourceDescriptor[] getResourceDescriptors() {
      return descriptors;
   }

   public IResourceDescriptor getCompositeDescriptor() {
      return compositeDescriptor;
   }

   public IProject getProject() {
      return composite.getProject();
   }

   public static String generateEditModelLabel(IProject project) {
      // TODO Auto-generated method stub
      return null;
   }
}
