/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.util;

import java.util.Iterator;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.stp.core.create.operations.ComponentCreationDataModelProvider;
import org.eclipse.stp.core.create.operations.ComponentTypeCreationDataModelProvider;
import org.eclipse.stp.core.create.operations.CompositeCreationDataModelProvider;
import org.eclipse.stp.core.create.operations.ModuleCreationDataModelProvider;
import org.eclipse.stp.core.create.operations.PropertyCreationDataModelProvider;
import org.eclipse.stp.core.create.operations.ReferenceCreationDataModelProvider;
import org.eclipse.stp.core.create.operations.ServiceCreationDataModelProvider;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModelFactory;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;
import org.eclipse.stp.core.internal.Messages;
import org.eclipse.stp.core.resources.ComponentTypeScribblerDomain;
import org.eclipse.stp.core.resources.CompositeScribblerDomain;
import org.eclipse.stp.core.resources.SOAConstants;
import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.Composite;
import org.eclipse.stp.core.sca.SCAPackage;
import org.eclipse.wst.common.frameworks.datamodel.DataModelFactory;
import org.eclipse.wst.common.frameworks.datamodel.IDataModel;

/**
 * This is the model access utility class. Clients should use this class to
 * create any model element or get handle to it.
 * 
 * For creating new model elements, clients can use createEObjectDataModel API.
 * They will create a new scribbler and pass it on to the API which will return
 * a DataModel. Client will fill in the required properties and execute the
 * default operation by callng scribbler.execute().
 */
public class SCAArtifactEdit implements IModelLifecycle {
   private static final String          EDIT_MODEL_SUFFIX = ".soaeditmodel"; //$NON-NLS-1$ 

   private IEditModelScribbler          scribbler         = null;

   private boolean                      isArtifactEditModelSelfManaged;

   private CompositeScribblerDomain     compositeDomain;

   private ComponentTypeScribblerDomain componentTypeDomain;

   /**
    * @param aScribbler
    */
   public SCAArtifactEdit(IEditModelScribbler aScribbler) {
      this.scribbler = aScribbler;
      this.isArtifactEditModelSelfManaged = false;
   }

   protected SCAArtifactEdit(IProject aProject,
         CompositeScribblerDomain aCompositeDomain,
         ComponentTypeScribblerDomain aComponentTypeDomain,
         boolean toAccessAsReadOnly) {
      Assert.isNotNull(aCompositeDomain);
      compositeDomain = aCompositeDomain;
      componentTypeDomain = aComponentTypeDomain;
      String editModelLabel = aProject.getName() + EDIT_MODEL_SUFFIX;
      try {
         if (toAccessAsReadOnly)
            scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
                  aProject, editModelLabel, new IScribblerDomain[] {
                        aCompositeDomain, aComponentTypeDomain });
         else
            scribbler = IEditModelFactory.eINSTANCE.createScribblerForWrite(
                  aProject, editModelLabel, new IScribblerDomain[] {
                        aCompositeDomain, aComponentTypeDomain });
      } catch (EditModelException e) {
         // Do something here
      }

      isArtifactEditModelSelfManaged = true;
   }

   /**
    * @param aProject
    * @param aCompositeDomain
    * @param aComponentTypeDomain
    * @return SCAArtifactEdit
    */
   public static SCAArtifactEdit createArtifactEditForRead(IProject aProject,
         CompositeScribblerDomain aCompositeDomain,
         ComponentTypeScribblerDomain aComponentTypeDomain) {
      SCAArtifactEdit scaArtifactEdit = null;
      try {
         scaArtifactEdit = new SCAArtifactEdit(aProject, aCompositeDomain,
               aComponentTypeDomain, true);
      } catch (IllegalArgumentException iae) {
         scaArtifactEdit = null;
      }

      return scaArtifactEdit;
   }

   /**
    * @param aProject
    * @param aCompositeDomain
    * @param aComponentTypeDomain
    * @return SCAArtifactEdit
    */
   public static SCAArtifactEdit createArtifactEditForWrite(IProject aProject,
         CompositeScribblerDomain aCompositeDomain,
         ComponentTypeScribblerDomain aComponentTypeDomain) {
      SCAArtifactEdit scaArtifactEdit = null;
      try {
         scaArtifactEdit = new SCAArtifactEdit(aProject, aCompositeDomain,
               aComponentTypeDomain, false);
      } catch (IllegalArgumentException iae) {
         scaArtifactEdit = null;
      }

      return scaArtifactEdit;
   }

   public void save(int options, IProgressMonitor aMonitor)
         throws ModelLifecycleException {
      if (isReadOnly())
         throwAttemptedReadOnlyModification();

      try {
         scribbler.save(true, aMonitor);
      } catch (EditModelException e) {
         throw new ModelLifecycleException(e);
      }
   }

   public void close(int options, IProgressMonitor aMonitor)
         throws ModelLifecycleException {
      try {
         if (isArtifactEditModelSelfManaged)
            scribbler.close(aMonitor);
      } catch (EditModelException e) {
         e.printStackTrace();
      }
   }

   public void discard(IProgressMonitor aMonitor)
         throws ModelLifecycleException {
      if (isReadOnly())
         throwAttemptedReadOnlyModification();

      try {
         scribbler.revert(aMonitor);
      } catch (EditModelException e) {
         throw new ModelLifecycleException(e);
      }
   }

   public void revert(int options, IProgressMonitor aMonitor)
         throws ModelLifecycleException {
      if (isReadOnly())
         throwAttemptedReadOnlyModification();

      try {
         scribbler.revert(aMonitor);
      } catch (EditModelException e) {
         throw new ModelLifecycleException(e);
      }

   }

   /**
    * @return boolean
    */
   public boolean isReadOnly() {
      return scribbler.isReadOnly();
   }

   private void throwAttemptedReadOnlyModification() {
      throw new IllegalStateException(Messages.Attempt_to_modify_an_SCA_);
   }

   /**
    * This API can be used to create an SCAObject of a specific
    * {@link org.eclipse.stp.core.util.SCAType} type. Clients can call
    * getDataModelForCreation() API passing in a type string and aProject. This
    * will return the appropriate IDataModel which needs to be further
    * populated.
    * 
    * @param type
    *           String
    * @param aProject
    *           IProject
    * @return IDataModel
    */
   public static IDataModel getDataModelForCreation(String type,
         IProject aProject) {
      Class dataModelClass = null;
      if (type.equals(SCAType.COMPONENT_TYPE)) {
         dataModelClass = ComponentTypeCreationDataModelProvider.class;
      } else if (type.equals(SCAType.COMPOSITE)) {
         dataModelClass = CompositeCreationDataModelProvider.class;
      } else if (type.equals(SCAType.MODULE)) {
         dataModelClass = ModuleCreationDataModelProvider.class;
      } else if (type.equals(SCAType.COMPONENT)) {
         dataModelClass = ComponentCreationDataModelProvider.class;
      } else if (type.equals(SCAType.REFERENCE)) {
         dataModelClass = ReferenceCreationDataModelProvider.class;
      } else if (type.equals(SCAType.SERVICE)) {
         dataModelClass = ServiceCreationDataModelProvider.class;
      } else if (type.equals(SCAType.PROPERTY)) {
         dataModelClass = PropertyCreationDataModelProvider.class;
      }

      if (dataModelClass != null) {
         IDataModel model = DataModelFactory.createDataModel(dataModelClass);
         Assert.isNotNull(model, Messages.DataModel_could_not_be_c_);

         // Set properties for scribbler creation
         // model.setStringProperty(IAbstractScribblerDataModelProperties.EDIT_MODEL_LABEL,
         // CompositeScribblerDomain.generateEditModelLabel(aProject));
         // model.setProperty(IAbstractScribblerDataModelProperties.SCRIBBLER_DOMAINS,
         // new IScribblerDomain [] {new CompositeScribblerDomain (aProject)});
         // model.setProperty(IAbstractScribblerDataModelProperties.PROJECT_NAME,
         // aProject.getName());

         return model;
      }

      return null;
   }

   public ComponentType getComponentType() {
      Resource resource = scribbler.getResource(componentTypeDomain
            .getComponentTypeDescriptor());
      return (ComponentType) getFirstInstanceOfEObjectFromResource(
            SCAPackage.eINSTANCE.getComponentType(), resource);
   }

   public Composite getComposite() {
      Resource resource = scribbler.getResource(compositeDomain
            .getCompositeDescriptor());
      return (Composite) getFirstInstanceOfEObjectFromResource(
            SCAPackage.eINSTANCE.getComposite(), resource);
   }

   public EObject getRootSCAObject(URI scaObjectURI) {
      Assert.isNotNull(scribbler);

      // get the resource from the scribbler
      Resource resource = scribbler.getResource(scaObjectURI);
      return getRootSCAObject(resource);
   }

   public EObject getRootSCAObject(Resource rootResource) {
      SCAPackage scdlPkg = SCAPackage.eINSTANCE;
      EClass eClass = null;
      String ext = rootResource.getURI().fileExtension();

      Assert.isNotNull(ext);

      if (ext.equals(SOAConstants.COMPONENT_TYPE_EXT))
         eClass = scdlPkg.getComponent();
      else if (ext.equals(SOAConstants.COMPOSITE_EXT))
         eClass = scdlPkg.getComposite();

      Assert.isNotNull(scribbler);

      return getFirstInstanceOfEObjectFromResource(eClass, rootResource);
   }

   /**
    * @param eClass
    *           for the object type we are looking for
    * @param resource
    *           to search for the given object type
    * @return the first <code>EObject</code> which is an instance of the given
    *         <code>EClass</code>
    */
   public EObject getFirstInstanceOfEObjectFromResource(EClass eClass,
         Resource resource) {
      if (resource == null)
         return null;

      for (Iterator iter = resource.getAllContents(); iter.hasNext();) {
         EObject element = (EObject) iter.next();
         if (element.eClass().equals(eClass))
            return element;
      }

      return null;
   }
}
