/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.core.bpel;

import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.Switches;
import org.eclipse.stp.b2j.core.jengine.internal.core.Runner;
import org.eclipse.stp.b2j.core.jengine.internal.core.RunnerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedVariable;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedMutex;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;

/**
 * 
 * @author amiguel
 * 
 * Part of the generic runtime engine API, used to help implement BPEL variables.
 */
public class BPELVariable {
	
	private static final String MAPNAME = "BPEL_VARIABLES";
	public static final String PREFIX = "BPV:";

	public static final String LOCK_PREFIX = "BPV_LOCK:";
	
//	private static HashMap map = new HashMap();
//	private static Object map_sync = new Object();
	
	public static void lockVariable(RunnerInterface runner, String name) throws Exception {
		SharedMutex mutex = runner.newMutex(LOCK_PREFIX+name);
		mutex.lock();
		if (name.indexOf("loggedOn") != -1 && name.indexOf("SLOCK") != -1) {
			System.out.println("LOCKED VARIABLE "+name+" via MUTEX "+mutex.sem.id);
		}
	}
	public static void releaseVariable(RunnerInterface runner, String name) throws Exception {
		SharedMutex mutex = runner.newMutex(LOCK_PREFIX+name);
		if (name.indexOf("loggedOn") != -1 && name.indexOf("SLOCK") != -1) {
			System.out.println("RELEASING VARIABLE "+name+" via MUTEX "+mutex.sem.id);
		}
		mutex.release();
	}
	
	public static void createVariable(RunnerInterface runner, String name, Message msg, boolean local) throws Exception {
		if (local) {
			HashMap map = runner.getEngineLocalStorageMap(MAPNAME);
			
			synchronized(map) {
				if (!map.containsKey(name)) {
					//dont want to overwrite existing variables with some new value
					SharedVariable var = new SharedVariable(null, name, -1, SharedVariable.MESSAGE, msg);
					map.put(name,var);
				}
			}
		} else {
			createVariable(runner,name,msg);
		}
	}

	public static void createVariable(RunnerInterface runner, String name, Message msg) throws Exception {
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipushStack("BPELVariable createVariable");
		}
		SharedVariable var = runner.newVariable(PREFIX+name,SharedVariable.MESSAGE,true);
		var.setValue(msg);
		var.varStore();
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipopStack();
		}
	}
	
	public static void setMessage(RunnerInterface runner, String varname, Message msg) throws Exception {
		HashMap map = runner.getEngineLocalStorageMap(MAPNAME);

		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipushStack("BPELVariable setMessage");
		}
		SharedVariable var;
		synchronized(map) {
			var = (SharedVariable)map.get(varname);
		}
		if (var != null) {
			var.setValue(msg);
		} else {
			var = runner.getVariable(PREFIX+varname);
			var.setValue(msg);
			var.varStore();
		}
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipopStack();
		}
	}
	
	public static Message getMessage(RunnerInterface runner, String varname) throws Exception {
		HashMap map = runner.getEngineLocalStorageMap(MAPNAME);

		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipushStack("BPELVariable getMessage");
		}
		SharedVariable var;
		synchronized(map) {
			var = (SharedVariable)map.get(varname);
		}
		if (var != null) {

			if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
				runner.ipopStack();
			}
			return (Message)var.getValue();
		} else {
			var = runner.getVariable(PREFIX+varname);
			var.varFetch();

			if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
				runner.ipopStack();
			}
			return (Message)var.getValue();
		}
	}
	
	/////////////////////////////
	// These methods help to optimise the XPath expressions
	/////////////////////////////
	
	public static SharedVariable getReference(RunnerInterface runner, String varname) throws Exception {
		HashMap map = runner.getEngineLocalStorageMap(MAPNAME);

		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipushStack("BPELVariable getReference");
		}
		SharedVariable var;
		synchronized(map) {
			var = (SharedVariable)map.get(varname);
		}
		if (var == null) {
			var = runner.getVariable(PREFIX+varname);
		}
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipopStack();
		}
		return var;
	}
	
	public static Message getMessageFromRef(RunnerInterface runner, SharedVariable var) throws Exception {
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipushStack("BPELVariable getReference");
		}
		var.varFetch();
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			runner.ipopStack();
		}
		return (Message)var.getValue();
	}
}