/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http;

import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.SOAPFactory;

/**
 * 
 * @author amiguel
 *
 * An HTTP server implementation used in the SOAP binding
 * Allows listening for certain HTTP resources on the localhost on a given port,
 * receive() method blocks until a full HTTP request has been received
 */
public class HTTPServer {

static HashMap url_listeners = new HashMap();
static Object listeners_LOCK = new Object();
	
//server pool, indexed by port
static HashMap pool = new HashMap();
static Object pool_LOCK = new Object();

private static final String RECEIVE_PREFIX = "RECEIVE:";
private static final String REPLY_PREFIX = "REPLY:";

	private static PortServer getOrCreatePortServer(String protocol, int port) throws Exception {
		Integer oport = new Integer(port);
		PortServer server;
		//get the server for our port
		synchronized(pool_LOCK) {
			server = (PortServer)pool.get(oport);
			if (server == null) {
				server = new PortServer(protocol, port);
				pool.put(oport,server);
			}
		}
		return server;
	}
	
	private static ArrayList getOrCreateListenersAt(int port, String resource) {
		String key = port+" : "+resource;
		
		ArrayList listeners = null;
		synchronized(listeners_LOCK) {
			listeners = (ArrayList)url_listeners.get(key);
			if (listeners == null) {
				listeners = new ArrayList();
				url_listeners.put(key,listeners);
			}
		}
		return listeners;
	}
	
	public static void addHttpListener(URL url, HTTPServerListener listener) throws Exception {
		int port = url.getPort();
		if (port == -1) {
			try {
				port = HTTPUtils.getDefaultPort(url);
//				port = url.getDefaultPort();
			} catch (NoSuchMethodError t) {
				if (url.getProtocol().equalsIgnoreCase("http")) {
					port = 80;
				} else if (url.getProtocol().equalsIgnoreCase("https")) {
					port = 443;
				} else {
					port = 80;
				}
			}
		}
		String resource = url.getFile();
		if (resource.length() == 0) resource = "/";
		
		//make sure we are listening on this port
		getOrCreatePortServer(url.getProtocol(),port);
		
		//add a listener for this resource
		ArrayList listeners = getOrCreateListenersAt(port,resource);
		synchronized(listeners) {
			if (!listeners.contains(listener)) {
				listeners.add(listener);
			}
		}
	}
	public static void removeHttpListener(URL url, HTTPServerListener listener) throws Exception {
		int port = url.getPort();
		if (port == -1) {
			try {
				port = HTTPUtils.getDefaultPort(url);
//				port = url.getDefaultPort();
			} catch (NoSuchMethodError t) {
				if (url.getProtocol().equalsIgnoreCase("http")) {
					port = 80;
				} else if (url.getProtocol().equalsIgnoreCase("https")) {
					port = 443;
				} else {
					port = 80;
				}
			}
		}
		String resource = url.getFile();
		
		//make sure we are listening on this port
		getOrCreatePortServer(url.getProtocol(),port);
		
		//add a listener for this resource
		ArrayList listeners = getOrCreateListenersAt(port,resource);
		synchronized(listeners) {
			listeners.remove(listener);
		}
	}
	
	static HTTPResponse doRequest(int port, String resource, String payload, HashMap lc_headers) throws HTTPException {
		if (SOAPFactory.DEBUG) System.out.println("Incoming B2J HTTP Server request for "+resource+" with payload size bytes");					

		ArrayList listeners = getOrCreateListenersAt(port,resource);
		
		for (int i = 0; i < listeners.size(); i++) {
			HTTPServerListener listener = null;
			
			//get the next listener
			synchronized(listeners) {
				if (i < listeners.size()) {
					listener = (HTTPServerListener)listeners.get(i);
				} else {
					//no more listeners - return 404
					throw new HTTPNotFoundException("Resource not found");
				}
			}
			
			try {
				if (SOAPFactory.DEBUG) System.out.println("HTTP Server Listener found is "+listener);					
				if (listener != null) {
					if (SOAPFactory.DEBUG) System.out.println("HTTP Server Listener class is "+listener.getClass().getName());					
				}
				HTTPResponse resp = listener.doRequest(port,resource,payload,lc_headers);
//				String response = 

				if (SOAPFactory.DEBUG) System.out.println("HTTP Server Listener response is size "+resp.getSize());					
				
				//got a valid response from this listener
				return resp;
			} catch (HTTPNotFoundException e) {
				//ignore this - this means it's not the right handler
			}
		}
		
		//no more listeners - return 404
		throw new HTTPNotFoundException("Resource not found");
	}
	
}