/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.publicapi.transport.session;

import java.io.Serializable;

import org.eclipse.stp.b2j.core.misc.internal.HexData;
import org.eclipse.stp.b2j.core.misc.internal.XMLUtil;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Document;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;

/**
 * A class to represent all information needed to create and configure 
 * a new communication session.  (A class to represent a host address and
 * all associated session configuration information)
 * 
 * @author amiguel
 */
public class SessionAddress implements Cloneable, Serializable {

	public static final int TRANSPORT_PORT_ANY = -1;
	
	private String transport_provider_class = "";
//	private String transport_provider_plugin = "";
	private boolean requires_multiple_streams = false;
	private boolean requires_encryption = false;
	private boolean requires_link_reconnection = false;
	private int requires_encryption_bits = 128;
	private boolean requires_multiple_connections = false;
	private boolean requires_authentication = false;

	private long abort_timeout_ms = 120000;
	private long startup_abort_timeout_ms = 60000;
	
	private String initiator_host = "";
	private int initiator_port_min = 0;
	private int initiator_port_max = 0;
	private String listener_host = "";
	private int listener_port_min = 0;
	private int listener_port_max = 0;

	private String password = "";
	private boolean requires_password = false;

	/**
	 * Set the password for this session address
	 * @param password the password for this session address
	 */
	public void setPassword(String password) {
		this.password = password;
	}
	
	/**
	 * Get the password for this session address
	 * return the password for this session address
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * Whether the Session will require a password
	 * @return true if the Session will require a password
	 */
	public boolean getRequiresPassword() {
		return requires_password;
	}

	/**
	 * Set whether the Session will require a password
	 * @param b true if the Session will require a password
	 */
	public void setRequiresPassword(boolean b) {
		requires_password = b;
	}
	
	/**
	 * Set whether the Session will require multiple streams (requires multiplexing)
	 * @param b true if the Session will require multiple streams (multiplexing)
	 */
	public void setRequiresMultipleStreams(boolean b) {
		requires_multiple_streams = b;
	}
	
	/**
	 * Get whether the Session will require multiple streams (requires multiplexing)
	 * @return true if the Session will require multiple streams (multiplexing)
	 */
	public boolean getRequiresMultipleStreams() {
		return requires_multiple_streams;
	}
	
	/**
	 * Set whether the Session will need to accept multiple connections on the same port
	 * @param b true if the Session will need to accept multiple connections on the same port
	 */
	public void setRequiresMultipleConnections(boolean b) {
		requires_multiple_connections = b;
	}
	/**
	 * Get whether the Session will need to accept multiple connections on the same port
	 * @param b true if the Session will need to accept multiple connections on the same port
	 */
	public boolean getRequiresMultipleConnections() { 
		return requires_multiple_connections;
	}
	
	/**
	 * Set whether the Session will require encryption
	 * @param b true if the Session will require encryption
	 */
	public void setRequiresEncryption(boolean b) {
		requires_encryption = b;
	}
	/**
	 * Get whether the Session will require encryption
	 * @return true if the Session will require encryption
	 */
	public boolean getRequiresEncryption() {
		return requires_encryption;
	}

	
	/**
	 * Set whether the Session will require authentication
	 * @param b true if the Session will require authentication
	 */
	public void setRequiresAuthentication(boolean b) {
		requires_authentication = b;
	}
	/**
	 * Get whether the Session will require authentication
	 * @return true if the Session will require authentication
	 */
	public boolean getRequiresAuthentication() {
		return requires_authentication;
	}
	
	/**
	 * Set the strength of any required encryption for the Session (in bits)
	 * @param bits the minimum encryption strength in bits required for the encryption in the Session
	 */
	public void setRequiresEncryptionStrengthBits(int bits) {
		requires_encryption_bits = bits;
	}
	/**
	 * Get the strength of any required encryption for the Session (in bits)
	 * @return the minimum encryption strength in bits required for the encryption in the Session
	 */
	public int getRequiresEncryptionStrengthBits() {
		return requires_encryption_bits;
	}
	
	/**
	 * Set whether this Session requires automatic downed transport link reconnection
	 * @param b true if the Session requires automatic downed transport link reconnection
	 */
	public void setRequiresLinkReconnection(boolean b) {
		requires_link_reconnection = b;
	}
	/**
	 * Get whether this Session requires automatic downed transport link reconnection
	 * @return tru if the Session requires automatic downed transport link reconnection
	 */
	public boolean getRequiresLinkReconnection() {
		return requires_link_reconnection;
	}
	/**
	 * Set the failure timeout for automatic link reconnection in milliseconds
	 * @param milliseconds the failure timeout for automatic link reconnection
	 */
	public void setReconnectionFailureAbortTimeout(long milliseconds) {
		abort_timeout_ms = milliseconds;
	}
	/**
	 * Get the failure timeout for automatic link reconnection in milliseconds
	 * @return the failure timeout for automatic link reconnection
	 */
	public long getReconnectionFailureAbortTimeout() {
		return abort_timeout_ms;
	}
	/**
	 * Set the automatic reconnection failure timeout for the initial Session link attempt
	 * @param milliseconds the automatic failure timeout for the initial Session link attempt in milliseconds
	 */
	public void setStartupFailureAbortTimeout(long milliseconds) {
		startup_abort_timeout_ms = milliseconds;
	}
	/**
	 * Get the automatic reconnection failure timeout for the initial Session link attempt
	 * @return the automatic failure timeout for the initial Session link attempt in milliseconds
	 */
	public long getStartupFailureAbortTimeout() {
		return startup_abort_timeout_ms;
	}
	
	
	/**
	 * Set the transport provider class name for the Session
	 * @param classname the class name of the transport provider for the Session
	 */
	public void setTransportProviderClassName(String classname) {
		transport_provider_class = classname;
	}
	/**
	 * Get the transport provider class name for the Session
	 * @return the class name of the transport provider for the Session
	 */
	public String getTransportProviderClassName() {
		return transport_provider_class;
	}
	/**
	 * Set the transport provider plugin name for the Session
	 * @param pluginname the plugin name of the transport provider for the Session
	 */
//	public void setTransportProviderPluginName(String pluginname) {
//		transport_provider_plugin = pluginname;
//	}
	/**
	 * Get the transport provider plugin name for the Session
	 * @return the plugin name of the transport provider for the Session
	 */
//	public String getTransportProviderPluginName() {
//		return transport_provider_plugin;
//	}
	
	/**
	 * Set the name or IP of the host which will initiate the transport link connection (client)
	 * @param host the name or IP of the host which will initiate the transport link connection (client)
	 */
	public void setInitiatorHost(String host) {
		initiator_host = host;
	}
	/**
	 * Get the name or IP of the host which will initiate the transport link connection (client)
	 * @return the name or IP of the host which will initiate the transport link connection (client)
	 */
	public String getInitiatorHost() {
		return initiator_host;
	}
	
	/**
	 * Set the minimum port value the initator host should use (client side port number)
	 * @param port the minimum port value the initiator host should use (client side port number)
	 */
	public void setInitiatorPortMinimum(int port) {
		initiator_port_min = port;
	}
	/**
	 * Get the minimum port value the initiator host should use (client side port number)
	 * @return the minimum port value the initiator host should use (client side port number)
	 */
	public int getInitiatorPortMinimum() {
		return initiator_port_min;
	}
	
	/**
	 * Set the maximum port value the initiator host should use (client side port number)
	 * @param port the maximum port value the initiator host should use (client side port number)
	 */
	public void setInitiatorPortMaximum(int port) {
		initiator_port_max = port;
	}
	/**
	 * Get the maximum port value the initiator host should use (client side port number)
	 * @return the maximum port value the initiator host should use (client side port number)
	 */
	public int getInitiatorPortMaximum() {
		return initiator_port_max;
	}
	
	/**
	 * Set the name or IP of the host which will listen for the transport link connection (server)
	 * @param host the name or IP of the host which will listen for the transport link connection (server)
	 */
	public void setListenerHost(String host) {
		listener_host = host;
	}
	/**
	 * Get the name or IP of the host which will listen for the transport link connection (server)
	 * @return the name or IP of the host which will listen for the transport link connection (server)
	 */
	public String getListenerHost() {
		return listener_host;
	}
	
	/**
	 * Set the minimum port value the listener host should use (server side port number)
	 * @param port the minimum port value the listener host should use (server side port number)
	 */
	public void setListenerPortMinimum(int port) {
		listener_port_min = port;
	}
	/**
	 * Get the minimum port value the listener host should use (server side port number)
	 * @return the minimum port value the listener host should use (server side port number)
	 */
	public int getListenerPortMinimum() {
		return listener_port_min;
	}
	
	/**
	 * Set the maximum port value the listener host should use (server side port number)
	 * @param port the maximum port value the listener host should use (server side port number)
	 */
	public void setListenerPortMaximum(int port) {
		listener_port_max = port;
	}
	/**
	 * Get the maximum port value the listener host should use (server side port number)
	 * @return the maximum port value the listener host should use (server side port number)
	 */
	public int getListenerPortMaximum() {
		return listener_port_max;
	}

	/**
	 * Private constructor for internal use
	 */
	private SessionAddress() {
	}
	
	/**
	 * Create a new SessionAddress describing everything necessary to establish a transport link between two hosts
	 * @param from_host the host or IP which should initiate the connection (client)
	 * @param from_port_min the minimum port number the initiating host should use (client side port)
	 * @param from_port_max the maximum port number the initiating host should use (client side port)
	 * @param to_host the host or IP which should listen for the connection (server)
	 * @param to_port_min the minimum port number the listening host should use (server side port)
	 * @param to_port_max the maximum port number the listening host should use (server side port)
	 */
	public SessionAddress(String from_host, int from_port_min, int from_port_max, String to_host, int to_port_min, int to_port_max) {
		this.initiator_host = from_host;
		this.initiator_port_min = from_port_min;
		this.initiator_port_max = from_port_max;
		this.listener_host = to_host;
		this.listener_port_min = to_port_min;
		this.listener_port_max = to_port_max;
	}
	
	public static String toString(SessionAddress address) throws Exception {
		Document doc = XMLUtil.blankDocument("SessionAddress");
		Element elem_address = doc.getDocumentElement();
		
		elem_address.setAttribute("transport_provider_class",""+address.transport_provider_class);
//		elem_address.setAttribute("transport_provider_plugin",""+address.transport_provider_plugin);
		elem_address.setAttribute("requires_multiple_streams",""+address.requires_multiple_streams);
		elem_address.setAttribute("requires_multiple_connections",""+address.requires_multiple_connections);
		elem_address.setAttribute("requires_encryption",""+address.requires_encryption);
		elem_address.setAttribute("requires_link_reconnection",""+address.requires_link_reconnection);
		elem_address.setAttribute("requires_encryption_bits",""+address.requires_encryption_bits);
		elem_address.setAttribute("requires_authentication",""+address.requires_authentication);

		elem_address.setAttribute("abort_timeout_ms",""+address.abort_timeout_ms);
		elem_address.setAttribute("startup_abort_timeout_ms",""+address.startup_abort_timeout_ms);
		
		elem_address.setAttribute("initiator_host",""+address.initiator_host);
		elem_address.setAttribute("initiator_port_min",""+address.initiator_port_min);
		elem_address.setAttribute("initiator_port_max",""+address.initiator_port_max);
		elem_address.setAttribute("listener_host",""+address.listener_host);
		elem_address.setAttribute("listener_port_min",""+address.listener_port_min);
		elem_address.setAttribute("listener_port_max",""+address.listener_port_max);

		elem_address.setAttribute("requires_password",""+address.requires_password);
		if (address.password != null) {
			elem_address.setAttribute("password",HexData.stringToHexString(address.password));
		}
		
		return XMLUtil.documentToString(elem_address);
	}
	
	public static SessionAddress fromString(String xml) throws Exception {
		Document doc = XMLUtil.documentFromString(xml);
		Element elem_address = doc.getDocumentElement();
		
		SessionAddress address = new SessionAddress();
		address.transport_provider_class = elem_address.getAttribute("transport_provider_class");
//		address.transport_provider_plugin = elem_address.getAttribute("transport_provider_plugin");
		
		address.requires_multiple_streams = elem_address.getAttribute("requires_multiple_streams").equals("true");
		address.requires_multiple_connections = elem_address.getAttribute("requires_multiple_connections").equals("true");
		address.requires_encryption = elem_address.getAttribute("requires_encryption").equals("true");
		address.requires_link_reconnection = elem_address.getAttribute("requires_link_reconnection").equals("true");
		address.requires_encryption_bits = Integer.parseInt(elem_address.getAttribute("requires_encryption_bits"));
		address.requires_authentication = elem_address.getAttribute("requires_authentication").equals("true");
		
		address.abort_timeout_ms = Long.parseLong(elem_address.getAttribute("abort_timeout_ms"));
		address.startup_abort_timeout_ms = Long.parseLong(elem_address.getAttribute("startup_abort_timeout_ms"));
		
		address.initiator_host = ""+elem_address.getAttribute("initiator_host");
		address.initiator_port_min = Integer.parseInt(elem_address.getAttribute("initiator_port_min"));
		address.initiator_port_max = Integer.parseInt(elem_address.getAttribute("initiator_port_max"));
		address.listener_host = ""+elem_address.getAttribute("listener_host");
		address.listener_port_min = Integer.parseInt(elem_address.getAttribute("listener_port_min"));
		address.listener_port_max = Integer.parseInt(elem_address.getAttribute("listener_port_max"));

		if (elem_address.hasAttribute("password")) {
			address.password = HexData.hexStringToString(elem_address.getAttribute("password"));
		} else {
			address.password = null;
		}
		
		if (elem_address.hasAttribute("requires_password")) {
			address.requires_password = elem_address.getAttribute("requires_password").equals("true");
		} else {
			address.requires_password = false;
		}
		
		return address;
	}
	
	public boolean equals(Object o) {
		if (super.equals(o)) {
			return true;
		}
		
		boolean same = true;

		if (o instanceof SessionAddress) {
			SessionAddress addr = (SessionAddress)o;

			if (!addr.initiator_host.equals(initiator_host)) same = false;
			if (addr.initiator_port_min != initiator_port_min) same = false;
			if (addr.initiator_port_max != initiator_port_max) same = false;
			if (!addr.listener_host.equals(listener_host)) same = false;
			if (addr.listener_port_min != listener_port_min) same = false;
			if (addr.listener_port_max != listener_port_max) same = false;

			if (!addr.transport_provider_class.equals(transport_provider_class)) same = false;
//			if (!addr.transport_provider_plugin.equals(transport_provider_plugin)) same = false;
			if (addr.requires_multiple_streams != requires_multiple_streams) same = false;
			if (addr.requires_multiple_connections != requires_multiple_connections) same = false;
			if (addr.requires_encryption != requires_encryption) same = false;
			if (addr.requires_link_reconnection != requires_link_reconnection) same = false;
			if (addr.requires_encryption_bits != requires_encryption_bits) same = false;
			if (addr.requires_authentication != requires_authentication) same = false;

			if (addr.abort_timeout_ms != abort_timeout_ms) same = false;
			if (addr.startup_abort_timeout_ms != startup_abort_timeout_ms) same = false;
			
		} else {
			same = false;
		}
		return same;
	}
	
	public Object clone() {
		SessionAddress addr = new SessionAddress();
		
		addr.initiator_host = initiator_host;
		addr.initiator_port_min = initiator_port_min;
		addr.initiator_port_max = initiator_port_max;
		addr.listener_host = listener_host;
		addr.listener_port_min = listener_port_min;
		addr.listener_port_max = listener_port_max;
		
		addr.transport_provider_class = transport_provider_class;
//		addr.transport_provider_plugin = transport_provider_plugin;
		addr.requires_multiple_streams = requires_multiple_streams;
		addr.requires_multiple_connections = requires_multiple_connections;
		addr.requires_encryption = requires_encryption;
		addr.requires_link_reconnection = requires_link_reconnection;
		addr.requires_encryption_bits = requires_encryption_bits;
		addr.requires_authentication = requires_authentication;
		
		addr.abort_timeout_ms = abort_timeout_ms;
		addr.startup_abort_timeout_ms = startup_abort_timeout_ms;
		
		addr.requires_password = requires_password;
		addr.password = password;
		return addr;
	}
	
	private void appendPort(StringBuffer sb, int p) {
		if (p == TRANSPORT_PORT_ANY) {
			sb.append("ANY");
		} else {
			sb.append(p);
		}
	}
	
	public String toString() {
		StringBuffer sb = new StringBuffer();
		
		sb.append(initiator_host);
		sb.append(":");
		appendPort(sb,initiator_port_min);
		sb.append("-");
		appendPort(sb,initiator_port_max);
		
		sb.append(" -> ");
		
		sb.append(listener_host);
		sb.append(":");
		appendPort(sb,listener_port_min);
		sb.append("-");
		appendPort(sb,listener_port_max);
		
		if (requires_multiple_streams) {
			sb.append(" (daemon)");
		}
		if (requires_multiple_streams) {
			sb.append(" (multiplexed)");
		}
		if (requires_encryption) {
			sb.append(" (encrypted "+requires_encryption_bits+")");
		}
		if (requires_authentication) {
			sb.append(" (authenticated)");
		}
		if (requires_link_reconnection) {
			sb.append(" (reconnect "+startup_abort_timeout_ms+"ms "+abort_timeout_ms+"ms)");
		}
		if (transport_provider_class.length() > 0) {
			sb.append(" (transport "+transport_provider_class+")");
		}
		if (requires_password) {
			sb.append(" (password)");
		}
		
		return sb.toString();
	}

}