/*********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.ui.internal.launchconfig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.debug.ui.ILaunchConfigurationTab;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.stp.b2j.core.B2jPlugin;
import org.eclipse.stp.b2j.core.publicapi.extension.sessiontransport.SessionTransportLoader;
import org.eclipse.stp.b2j.core.publicapi.extension.sessiontransport.SessionTransportProvider;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;
import org.eclipse.stp.b2j.ui.B2jImageManager;
import org.eclipse.stp.b2j.ui.UiPlugin;
import org.eclipse.stp.b2j.ui.internal.misc.HexData;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;

public class LaunchConfigDistributionTab extends AbstractLaunchConfigurationTab implements ILaunchConfigurationTab, SelectionListener, KeyListener, IStructuredContentProvider, ITableLabelProvider, ICellModifier  {

	//
	// Should have only session level options.   The transport is just something the user selects
	//
	
	public static final int COL_NAME = 0;
	public static final int COL_DAEMONPORT = 1;
	public static final int COL_DAEMONTRANSPORT = 2;
	public static final int COL_DAEMONPASSWORD = 3;
	public static final int COL_TRANSPORT = 4;
	public static final int COL_ENCRYPTED = 5;
	public static final int COL_AUTHENTICATED = 6;
	public static final int COL_RECONNECT = 7;
	public static final int COL_RECONNECT_TIMEOUT = 8;

	static Object[][] columns = new Object[][] {
		new Object[] {UiPlugin.getString("COL_HOST"), new Integer(SWT.LEFT), new Integer(200)},
		new Object[] {UiPlugin.getString("COL_DAEMONPORT"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_DAEMONTRANSPORT"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_DAEMONPASSWORD"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_TRANSPORT"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_ENCRYPTED"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_AUTHENTICATED"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_RECONNECTION"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_RECONNECT_TIMEOUT"), new Integer(SWT.CENTER), new Integer(100)},
	};
	
	static String[] properties = new String[] {
			(String)columns[COL_NAME][0],
			(String)columns[COL_DAEMONPORT][0],
			(String)columns[COL_DAEMONTRANSPORT][0],
			(String)columns[COL_DAEMONPASSWORD][0],
			(String)columns[COL_TRANSPORT][0],
			(String)columns[COL_ENCRYPTED][0],
			(String)columns[COL_AUTHENTICATED][0],
			(String)columns[COL_RECONNECT][0],
			(String)columns[COL_RECONNECT_TIMEOUT][0],
	};
		
	Composite main_panel;
	
	Button mini_engine;
	Button main_engine;
	
	Label hosts_label;
	//List hosts_list;
	TableViewer hosts_viewer;
	
	
	Button hosts_add;
	Button hosts_remove;
	Button hosts_major;
	
	String message;
	String error_message;
	
	Map daemon_transports = new HashMap();
	Map daemon_passwords = new HashMap();
	SessionAddress[] addresses = new SessionAddress[0];
	
	String[] daemon_transport_options;
	SessionTransportProvider[] providers;
	/*
	private SessionTransportProvider getProviderByName(String name) {
		for (int i = 0; i < providers.length; i++) {
			if (providers[i].getHumanReadableTypeName().equals(name)) {
				return providers[i];
			}
		}
		return null;
	}*/
	private int getProviderIndexByClass(SessionAddress address) {
		String name = address.getTransportProviderClassName();
		for (int i = 0; i < providers.length; i++) {
			if (providers[i].getProviderClassName().equals(name)) {
				return i;
			}
		}
		try {
			address.setTransportProviderClassName( providers[0].getProviderClassName() );
//			address.setTransportProviderPluginName( providers[0].getProviderPluginName() );
			if (providers[0].supportsEncryption()) {
				address.setRequiresEncryption( true );
			} else {
				address.setRequiresEncryption( false );
			}
			if (providers[0].supportsAuthentication()) {
				address.setRequiresAuthentication( true );
			} else {
				address.setRequiresAuthentication( false );
			}
		} catch (Exception e) {
		}
		return 0;
	}
	
	public void checkForErrors() {
		error_message = null;
	}

	public boolean isValid(ILaunchConfiguration launchConfig) {
		return error_message == null;
	}
	
	public String getErrorMessage() {
		return error_message;
	}
	
	public String getMessage() {
		return message;
	}
	
	public void setDirty(boolean b) {
		super.setDirty(b);
		checkForErrors();
		updateLaunchConfigurationDialog();
	}
	
	private void setHostsEnabled(boolean b) {
		hosts_viewer.getTable().setEnabled(b);
//		hosts_list.setEnabled(b);
		hosts_add.setEnabled(b);
		hosts_remove.setEnabled(b);
		hosts_major.setEnabled(b);
	}

	public void createControl(Composite parent) {
		
		main_panel = new Composite(parent,0);
		main_panel.setLayout(new GridLayout(4,false));
		
		mini_engine = new Button(main_panel,SWT.RADIO);
		mini_engine.setText(UiPlugin.getString("USE_MINI_ENGINE"));
		mini_engine.setLayoutData(new GridData(GridData.BEGINNING,GridData.END,false,false,4,1));

		main_engine = new Button(main_panel,SWT.RADIO);
		main_engine.setText(UiPlugin.getString("USE_MAIN_ENGINE"));
		main_engine.setLayoutData(new GridData(GridData.BEGINNING,GridData.END,false,false,4,1));

		mini_engine.setSelection(true);
		
//		public GridData(int horizontalAlignment, int verticalAlignment, boolean grabExcessHorizontalSpace, boolean grabExcessVerticalSpace, int horizontalSpan, int verticalSpan) {

		hosts_label = new Label(main_panel,SWT.NONE);
		hosts_label.setText(UiPlugin.getString("HOST_LIST"));
		hosts_label.setLayoutData(new GridData(GridData.BEGINNING,GridData.END,false,false,4,1));
		
		hosts_viewer = new TableViewer(new Table(main_panel, SWT.FULL_SELECTION | SWT.SINGLE | SWT.HIDE_SELECTION));
		hosts_viewer.getTable().setLayoutData(new GridData(GridData.FILL,GridData.FILL,true,true,3,4));
		
		hosts_add = new Button(main_panel,SWT.NONE);
		hosts_add.setText(UiPlugin.getString("HOST_ADD"));
		hosts_add.setLayoutData(new GridData(GridData.FILL,GridData.FILL,false,false,1,1));
		
		hosts_remove = new Button(main_panel,SWT.NONE);
		hosts_remove.setText(UiPlugin.getString("HOST_REMOVE"));
		hosts_remove.setLayoutData(new GridData(GridData.FILL,GridData.FILL,false,false,1,1));
		
		hosts_major = new Button(main_panel,SWT.NONE);
		hosts_major.setText(UiPlugin.getString("MAKE_MAJOR_CONTROLLER"));
		hosts_major.setLayoutData(new GridData(GridData.FILL,GridData.FILL,false,false,1,1));

		setHostsEnabled(false);
		
		//table stuff
		hosts_viewer.getTable().setHeaderVisible(true); 
		hosts_viewer.getTable().setLinesVisible(true);
		
		hosts_viewer.setContentProvider(this);
		hosts_viewer.setLabelProvider(this);

		for (int i = 0; i < columns.length; i++) {
			String col_name = (String)columns[i][0];
			int col_hints = ((Integer)columns[i][1]).intValue();
			int col_width = ((Integer)columns[i][2]).intValue();
			
			TableColumn tc = new TableColumn(hosts_viewer.getTable(),col_hints);
			tc.setText(col_name);
			tc.setWidth(col_width);
		}
		
		try {
			providers = SessionTransportLoader.getTransports();
		} catch (Exception e) {
			UiPlugin.DBG.logVisibleError(e,"Error loading session transports",false);
			providers = B2jPlugin.getDefaultSessionTransportProviders();
		}
		
		daemon_transport_options = new String[]{
				UiPlugin.getString("HTTP"),
				UiPlugin.getString("HTTPS")
		};
		
		String[] transport_hrnames = new String[providers.length];
		for (int i = 0; i < providers.length; i++) {
			transport_hrnames[i] = providers[i].getHumanReadableTypeName();
		}
		
		CellEditor[] editors = new CellEditor[] {
				new TextCellEditor(hosts_viewer.getTable()),
				new TextCellEditor(hosts_viewer.getTable()),
				new ComboBoxCellEditor(hosts_viewer.getTable(),daemon_transport_options),
				new TextCellEditor(hosts_viewer.getTable()),
				new ComboBoxCellEditor(hosts_viewer.getTable(),transport_hrnames),
				new CheckboxCellEditor(hosts_viewer.getTable()),
				new CheckboxCellEditor(hosts_viewer.getTable()),
				new CheckboxCellEditor(hosts_viewer.getTable()),
				new TextCellEditor(hosts_viewer.getTable()),
		};
		
		
		hosts_viewer.setCellEditors(editors);
		hosts_viewer.setCellModifier(this);
		hosts_viewer.setColumnProperties(properties);
		
		//
		// Event handling
		//
		mini_engine.addSelectionListener(this);
		main_engine.addSelectionListener(this);
		hosts_add.addSelectionListener(this);
		hosts_remove.addSelectionListener(this);
		hosts_major.addSelectionListener(this);
	}
	
	public Control getControl() {
		return main_panel;
	}

	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute("engine_impl","mini");
		configuration.setAttribute("hosts_list",new ArrayList());
	}

	public void initializeFrom(ILaunchConfiguration configuration) {
		try {
			String engine_impl = configuration.getAttribute("engine_impl","mini");
			
			java.util.List hosts = configuration.getAttribute("hosts_list",new ArrayList());
			
			SessionAddress[] items = new SessionAddress[hosts.size()];
			for (int i = 0; i < items.length; i++) {
				try {
					String tmp = (String)hosts.get(i);
					tmp = HexData.hexStringToString(tmp);
					items[i] = SessionAddress.fromString(tmp);
				} catch (Exception e) {
					items[i] = new SessionAddress("localhost",SessionAddress.TRANSPORT_PORT_ANY,SessionAddress.TRANSPORT_PORT_ANY,(String)hosts.get(i),B2jPlugin.getDefaultSoapDaemonPort(),B2jPlugin.getDefaultSoapDaemonPort());
				}

				boolean OK = false;
				for (int k = 0; k < providers.length; k++) {
					if (items[i].getTransportProviderClassName().equals(providers[k].getProviderClassName())) {
						OK = true;
					}
				}
				if (!OK) {
					items[i].setTransportProviderClassName( providers[0].getProviderClassName() );
//					items[i].setTransportProviderPluginName( providers[0].getProviderPluginName() );
					if (providers[0].supportsEncryption()) {
						items[i].setRequiresEncryption( true );
					} else {
						items[i].setRequiresEncryption( false );
					}
					if (providers[0].supportsAuthentication()) {
						items[i].setRequiresAuthentication( true );
					} else {
						items[i].setRequiresAuthentication( false );
					}
				}
			}
			
			if (engine_impl.equalsIgnoreCase("main")) {
				main_engine.setSelection(true);
				mini_engine.setSelection(false);

				setHostsEnabled(true);
			} else {
				mini_engine.setSelection(true);
				main_engine.setSelection(false);

				setHostsEnabled(false);
			}
			addresses = items;
			
			Map str_daemon_transports = configuration.getAttribute("daemon_transports",new HashMap());
			Map str_daemon_passwords = configuration.getAttribute("daemon_passwords",new HashMap());

			daemon_transports.clear();
			daemon_passwords.clear();
			
			for (int i = 0; i < hosts.size(); i++) {
//				Object str_key = hosts.get(i);
				Object str_key = ""+i;
				Object obj_key = items[i];
				daemon_transports.put(obj_key,str_daemon_transports.get(str_key));
				daemon_passwords.put(obj_key,str_daemon_passwords.get(str_key));
			}
			
			if (addresses.length == 0) {
				addAddress("localhost");
			}

			hosts_viewer.setInput(addresses);
			
		} catch (CoreException e) {
			 e.printStackTrace();
		} catch (Exception e) {
			 e.printStackTrace();
		}
		checkForErrors();
	}

	public void performApply(ILaunchConfigurationWorkingCopy configuration) {

		ArrayList list = new ArrayList();
		
		for (int i = 0; i < addresses.length; i++) {
			try {
				String tmp = SessionAddress.toString(addresses[i]);
				tmp = HexData.stringToHexString(tmp);
				list.add(tmp);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		
		if (main_engine.getSelection()) {
			configuration.setAttribute("engine_impl","main");
		} else {
			configuration.setAttribute("engine_impl","mini");
		}
		configuration.setAttribute("hosts_list",list);
		
		Map str_daemon_transports = new HashMap();
		Map str_daemon_passwords = new HashMap();
		
		for (int i = 0; i < addresses.length; i++) {
//			Object str_key = list.get(i);
			Object str_key = ""+i;
			Object obj_key = addresses[i];
			
			Object o = daemon_transports.get(obj_key);
			if (o != null) {
				str_daemon_transports.put(str_key,o);
			}
			o = daemon_passwords.get(obj_key);
			if (o != null) {
				str_daemon_passwords.put(str_key,o);
			}
		}
		
		configuration.setAttribute("daemon_transports",str_daemon_transports);
		configuration.setAttribute("daemon_passwords",str_daemon_passwords);
	}

	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		initializeFrom(workingCopy);
	}
	
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		performApply(workingCopy);
	}
	
	public String getName() {
		return UiPlugin.getString("HOSTS_TAB_NAME");
	}

	public Image getImage() {
		return UiPlugin.imageManager.getImage(B2jImageManager.IMG_HOST);
	}
	
	public void dispose() {
		try {
			main_panel.dispose();
		} catch (Exception e) {
		}
	}

	//
	// Event handlers
	//
	
	private void addAddress(String host) {
		SessionAddress[] tmp = new SessionAddress[addresses.length+1];
		System.arraycopy(addresses,0,tmp,0,addresses.length);
		tmp[tmp.length-1] = new SessionAddress("localhost",SessionAddress.TRANSPORT_PORT_ANY,SessionAddress.TRANSPORT_PORT_ANY,host,B2jPlugin.getDefaultSoapDaemonPort(),B2jPlugin.getDefaultSoapDaemonPort());
		addresses = tmp;
	}
	
	public void keyPressed(KeyEvent e) {
		setDirty(true);
	}

	public void keyReleased(KeyEvent e) {
		setDirty(true);
	}

	public void widgetSelected(SelectionEvent e) {
		Object o = e.getSource();
		if (o == hosts_add) {
			
			addAddress("HOSTNAME");
			
			hosts_viewer.setInput(addresses);
			
			setDirty(true);

		} else if (o == hosts_remove) {
			
			//TODO remove the daemon info in here too
			
			ArrayList list = new ArrayList();
			
			IStructuredSelection selection = (IStructuredSelection)hosts_viewer.getSelection();
			Object[] removals = selection.toArray();
			
			for (int k = 0; k < addresses.length; k++) {
				boolean remove = false;
				for (int i = 0; i < removals.length; i++) {
					if (addresses[k] == removals[i]) {
						//selected for removal
						remove = true;
						break;
					}
				}
				if (!remove) {
					list.add(addresses[k]);
				}
			}
			
			SessionAddress[] tmp = new SessionAddress[list.size()];
			list.toArray(tmp);
			
			if (addresses.length != tmp.length) {
				setDirty(true);
			}
			
			if (addresses.length == 0) {
				addAddress("localhost");
			}
			
			addresses = tmp;

			hosts_viewer.setInput(addresses);

		} else if (o == hosts_major) {
			
			IStructuredSelection selection = (IStructuredSelection)hosts_viewer.getSelection();
			SessionAddress address = (SessionAddress)selection.getFirstElement();
			
			for (int i = 1; i < addresses.length; i++) {
				if (addresses[i] == address) {
					//swap with index 0
					Object tmp = addresses[0];
					addresses[0] = address;
					addresses[i] = (SessionAddress)tmp;
					
					setDirty(true);
					hosts_viewer.setInput(addresses);
				}
			}

			
		} else if (o == mini_engine) {
			
			setHostsEnabled(false);
			
			setDirty(true);

		} else if (o == main_engine) {

			setHostsEnabled(true);
			
			setDirty(true);

		}
		
		checkForErrors();
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	
	class IgnoredInputValidator implements IInputValidator {

		public String isValid(String newText) {
			return null;
		}
	
	}
	
	//
	// Table content provider
	//

	public void inputChanged(Viewer currviewer, Object oldInput, Object newInput) {
	}
	
	
	public Object[] getElements(Object newInput) {
		return (SessionAddress[])newInput;
	}
	
	//
	// Label provider
	//
	
	private boolean isMajorController(SessionAddress address) {
		if (addresses.length == 0) {
			return false;
		}
		if (address == addresses[0]) {
			return true;
		}
		return false;
	}
	
	public Image getColumnImage(Object element, int columnIndex) {
		SessionAddress address = (SessionAddress)element;
		boolean major = isMajorController(address);

		if (columnIndex == COL_NAME) {
			if (addresses.length == 1) {
				return UiPlugin.imageManager.getImage(B2jImageManager.IMG_BOTH_CONTROLLER);
			}
			
			if (major) {
				return UiPlugin.imageManager.getImage(B2jImageManager.IMG_MAJOR_CONTROLLER);
			} else {
				return UiPlugin.imageManager.getImage(B2jImageManager.IMG_MINOR_CONTROLLER);
			}
		}
		return null;
	}
	public String getColumnText(Object element, int columnIndex) {
		SessionAddress address = (SessionAddress)element;
		boolean major = isMajorController(address);
		
		if (columnIndex == COL_NAME) {
			if (addresses.length == 1) {
				return address.getListenerHost()+" ("+UiPlugin.getString("BOTH_CONTROLLER")+")";
			}

			if (major) {
				return address.getListenerHost()+" ("+UiPlugin.getString("MAJOR_CONTROLLER")+")";
			} else {
				return address.getListenerHost()+" ("+UiPlugin.getString("MINOR_CONTROLLER")+")";
			}
		} else if (columnIndex == COL_TRANSPORT) {
			return providers[getProviderIndexByClass(address)].getHumanReadableTypeName();
		} else if (columnIndex == COL_DAEMONPORT) {
			return ""+address.getListenerPortMinimum();
		} else if (columnIndex == COL_DAEMONTRANSPORT) {
			String transport = (String)daemon_transports.get(address);
			if (transport == null) {
				return UiPlugin.getString("HTTP");
			} else if (transport.equals("HTTPS")) {
				return UiPlugin.getString("HTTPS");
			} else {
				return UiPlugin.getString("HTTP");
			}
		} else if (columnIndex == COL_DAEMONPASSWORD) {
			String password = (String)daemon_passwords.get(address);
			if (password == null) {
				return UiPlugin.getString("NO_PASSWORD");
			} else if (password.length() == 0) {
				return UiPlugin.getString("NO_PASSWORD");
			} else {
				return password;
			}
		} else if (columnIndex == COL_ENCRYPTED) {
			if (address.getRequiresEncryption()) {
				return UiPlugin.getString("ENCRYPTED");
			} else {
				return UiPlugin.getString("UNENCRYPTED");
			}
		} else if (columnIndex == COL_AUTHENTICATED) {
			if (address.getRequiresAuthentication()) {
				return UiPlugin.getString("AUTHENTICATED");
			} else {
				return UiPlugin.getString("UNAUTHENTICATED");
			}
		} else if (columnIndex == COL_RECONNECT) {
			return ""+address.getRequiresLinkReconnection();
		} else if (columnIndex == COL_RECONNECT_TIMEOUT) {
			if (address.getRequiresLinkReconnection()) {
				return ""+address.getReconnectionFailureAbortTimeout()+" ("+toHumanReadableTime(address.getReconnectionFailureAbortTimeout())+")";
			} else {
				return "";
			}
		}
		return "unknown";
	}
	
	private String toHumanReadableTime(long t) {
		StringBuffer sb = new StringBuffer();
		
		//hours
		long hrs  = t / (60l * 60l * 1000l);
		t = t % (60l * 60l * 1000l);
		if (hrs > 0) {
			sb.append(hrs).append(UiPlugin.getString("HOURS")).append(" ");
		}
		
		//mins
		long mins = t / (60l * 1000l);
		t = t % (60l * 1000l);
		if (mins > 0) {
			sb.append(mins).append(UiPlugin.getString("MINUTES")).append(" ");
		}
		
		//secs
		double secs = ((double)t) / 1000.0d;
		if (secs > 0) {
			if (secs % 1.0d > 0d) {
				sb.append(secs).append(UiPlugin.getString("SECONDS"));
			} else {
				sb.append((long)secs).append(UiPlugin.getString("SECONDS"));
			}
		}
		
		return sb.toString();
	}

	public void addListener(ILabelProviderListener listener) {
	}
	public boolean isLabelProperty(Object element, String property) {
		return false;
	}
	public void removeListener(ILabelProviderListener listener) {
	}

	//
	// Cell editor
	//
	
	private int getIndexForProperty(String property) {
		for (int i = 0; i < properties.length; i++) {
			if (property.equals(properties[i])) {
				return i;
			}
		}
		return -1;
	}
	
	public boolean canModify(Object element, String property) {
		SessionAddress address = (SessionAddress)element;
		int index = getIndexForProperty(property);
		
		if (index == COL_RECONNECT_TIMEOUT) {
			if (address.getRequiresLinkReconnection()) {
				return true;
			} else {
				return false;
			}
		} else if (index == COL_RECONNECT) {
			return true;
		} else if (index == COL_ENCRYPTED) {
//			return providers[getProviderIndexByClass(address)].supportsEncryption();
			return false;
		} else if (index == COL_AUTHENTICATED) {
			return false;
		}
		
		return true;
	}

	public Object getValue(Object element, String property) {
		SessionAddress address = (SessionAddress)element;
		int index = getIndexForProperty(property);
		if (index == COL_NAME) {
			return address.getListenerHost();
		} else if (index == COL_TRANSPORT) {
			return new Integer(getProviderIndexByClass(address));
		} else if (index == COL_DAEMONPORT) {
			return ""+address.getListenerPortMinimum();
		} else if (index == COL_DAEMONTRANSPORT) {
			Object o = daemon_transports.get(address);
			
			for (int i = 0; i < daemon_transport_options.length; i++) {
				if (o == daemon_transport_options[i]) {
					return new Integer(i);
				}
			}
			return new Integer(0);
			
		} else if (index == COL_DAEMONPASSWORD) {
			Object o = daemon_passwords.get(address);
			if (o == null) {
				return "";
			}
			return o;
		} else if (index == COL_ENCRYPTED) {
			return new Boolean(address.getRequiresEncryption());
		} else if (index == COL_AUTHENTICATED) {
			return new Boolean(address.getRequiresAuthentication());
		} else if (index == COL_RECONNECT) {
			return new Boolean(address.getRequiresLinkReconnection());
		} else if (index == COL_RECONNECT_TIMEOUT) {
			return ""+address.getReconnectionFailureAbortTimeout();
		}
		return null;
	}

	public void modify(Object element, String property, Object value) {
		if (value == null) return;
		
		boolean dirty = false;
		
		TableItem item = (TableItem)element;
		SessionAddress address = (SessionAddress)item.getData();
		SessionAddress tmp = (SessionAddress)address.clone();
		int index = getIndexForProperty(property);
		if (index == COL_NAME) {
			address.setListenerHost( ""+value );
		} else if (index == COL_TRANSPORT) {
			SessionTransportProvider provider = providers[((Integer)value).intValue()];
			address.setTransportProviderClassName( provider.getProviderClassName() );
//			address.setTransportProviderPluginName( provider.getProviderPluginName() );
			if (provider.supportsEncryption()) {
				address.setRequiresEncryption( true );
			} else {
				address.setRequiresEncryption( false );
			}
			if (provider.supportsAuthentication()) {
				address.setRequiresAuthentication( true );
			} else {
				address.setRequiresAuthentication( false );
			}
		} else if (index == COL_DAEMONPORT) {
			address.setListenerPortMinimum( Integer.parseInt(""+value) );
			address.setListenerPortMaximum( Integer.parseInt(""+value) );
		} else if (index == COL_DAEMONTRANSPORT) {
			Object set = daemon_transport_options[((Integer)value).intValue()];
			if (daemon_transports.get(address) != set) {
				dirty = true;
				daemon_transports.put(address,set);
			}
		} else if (index == COL_DAEMONPASSWORD) {
			System.out.println("SET PASSWORD "+value);
			Object set = ""+value;
			if (daemon_passwords.get(address) == null) {
				dirty = true;
				daemon_passwords.put(address,set);
			} else if (!daemon_passwords.get(address).equals(set)) {
				dirty = true;
				daemon_passwords.put(address,set);
			}
		} else if (index == COL_ENCRYPTED) {
			address.setRequiresEncryption( ((Boolean)value).booleanValue() );
		} else if (index == COL_AUTHENTICATED) {
			address.setRequiresAuthentication( ((Boolean)value).booleanValue() );
		} else if (index == COL_RECONNECT) {
			address.setRequiresLinkReconnection( ((Boolean)value).booleanValue() );
		} else if (index == COL_RECONNECT_TIMEOUT) {
			address.setReconnectionFailureAbortTimeout( Long.parseLong(""+value) );
		}
		
		if (dirty || !tmp.equals(address)) {
			setDirty(true);
			hosts_viewer.setInput(addresses);
		}
	}
	
}