/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap;

import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.TranslatorLog;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLOperation;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLPortType;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDMap;
import org.eclipse.stp.b2j.core.jengine.internal.utils.TranslatorUtils;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslator;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslatorException;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslator;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;

/**
 * 
 * @author amiguel
 *
 * A class that translates any SOAP binging specific information in 
 * a WSDL source file into an engine Program part (e.g. method) 
 */
public class SOAPBindingTranslator implements WSDLBindingTranslator {

	private static final String LOG_SOURCE = "SOAP Binding";
	
	XSDTypeTranslator[] codecs = new XSDTypeTranslator[0];
	
	WSDLMap wsdlmap;
	XSDMap xsdmap;

	TranslatorLog log;
	Util compiler_util;
	
	ClassLoader dependancy_loader;

	int PICK_ID = 0;
	
	class PortTypeBinding {
		String qname;
		String qportType;
		
		String style;
		String transport;
		
		String address_location;
		
		String bindingName;
		
		HashMap opbindings = new HashMap();
	}
	class OperationBinding {
		String soapAction;
		String opname;
		InputBinding input;
		OutputBinding output;
		
		//TODO at the moment we only support 1 encoding type per binding
		//this means that this encodingStyle is always populated and will always be valid for all operations
		//under this binding
		String encodingStyle;
	}
	class InputBinding {
		String namespace;
		String use;
		String encodingStyle;
	}
	class OutputBinding {
		String namespace;
		String use;
		String encodingStyle;
	}
	
	HashMap bindings = new HashMap();

	public String getID() {
		return getClass().getName();
	}
	
	public String getHumanReadableName() {
//		return B2jPlugin.getString("BINDING_TYPE_SOAP");
		return "Default SOAP Binding";
	}

	
//	public String getProviderPluginName() {
//		return B2jPlugin.getDefault().getBundle().getSymbolicName();
//	}

	public void init(Util compiler_util, TranslatorLog log, XSDMap xsdmap, WSDLMap wsdlmap, ClassLoader dependancy_loader) {
		this.xsdmap = xsdmap;
		this.wsdlmap = wsdlmap;
		this.dependancy_loader = dependancy_loader;
		this.compiler_util = compiler_util;
		this.log = log;
		
		//Uses XML codec already provided by Internal Messaging binding 
	}

	private void checkForBindingNamed(String name) throws WSDLBindingTranslatorException {
		if (name == null) return;
		PortTypeBinding binding = (PortTypeBinding)bindings.get(name);
		if (binding == null) {
			throw new WSDLBindingTranslatorException("no SOAP binding found named '"+compiler_util.getQNameFromQName(name)+"'");
		}
	}
	
	private void checkForBinding(String qportType) throws WSDLBindingTranslatorException {
		PortTypeBinding binding = (PortTypeBinding)bindings.get(qportType);
		if (binding == null) {
			throw new WSDLBindingTranslatorException("no SOAP binding found for port type "+compiler_util.getQNameFromQName(qportType));
		}
	}
	
	public void readWSDL(Element wsdl_definitions) throws NamespaceException, WSDLBindingTranslatorException {
		
		NamespaceTranslator nt = compiler_util.createNamespaceTranslator();
		nt.addNamespaces(wsdl_definitions);
		
		ArrayList elems = Util.getAllElements(wsdl_definitions);
		for (int i = 0; i < elems.size(); i++) {
			Element elem = (Element)elems.get(i);
			nt.addNamespaces(elem);

			if (NamespaceTranslator.getName(elem).equals("binding")) {
				//we've found a binding
				//get the port type
				String ptype = elem.getAttribute("type");

				PortTypeBinding binding = new PortTypeBinding();

				binding.bindingName = nt.qualify(elem.getAttribute("name"),true);
				
				binding.qname = nt.qualify(elem.getAttribute("name"),true);
				binding.qportType = nt.qualify(ptype,false);

				ArrayList subelems = Util.getAllElements(elem);
				for (int k = 0; k < subelems.size(); k++) {
					Element bindingelem = (Element)subelems.get(k);
					nt.addNamespaces(bindingelem);
					
					//
					// SOAP binding for this portType
					//
					if (nt.checkQName(bindingelem,SOAPNamespaces.NAMESPACE_SOAP_WSDL,"binding")) {
						
						binding.style = bindingelem.getAttribute("style");
						binding.transport = bindingelem.getAttribute("transport");
						
						//add two entries, one referencing it via the port type, one via the binding type
						bindings.put(binding.qportType,binding);
						bindings.put(binding.qname,binding);
						bindings.put(binding.bindingName,binding);

						log.logInfo(LOG_SOURCE,"Found SOAP Binding named "+binding.bindingName);
						log.logInfo(LOG_SOURCE,"SOAP Binding "+compiler_util.getQNameFromQName(binding.bindingName)+" maps to port type "+compiler_util.getQNameFromQName(binding.qportType));
					}
					
					//
					// WSDL operation element
					//
					if (NamespaceTranslator.getName(bindingelem).equals("operation")) {
						ArrayList opelems = Util.getAllElements(bindingelem);
						
						OperationBinding opbinding = new OperationBinding();
						opbinding.opname = bindingelem.getAttribute("name");
						
						for (int j = 0; j < opelems.size(); j++) {
							Element opelem = (Element)opelems.get(j);
							nt.addNamespaces(opelem);
							
							//
							// SOAP binding for this operation
							//
							if (nt.checkQName(opelem,SOAPNamespaces.NAMESPACE_SOAP_WSDL,"operation")) {
								opbinding.soapAction = opelem.getAttribute("soapAction");
								opbinding.encodingStyle = opelem.getAttribute("style");
								
								if (opbinding.encodingStyle.length() > 0) {
									//operation has an encoding style specified
									if (binding.style.length() == 0) {
										//binding has no encoding - use this for entire binding
										binding.style = opbinding.encodingStyle;
									} else if (!binding.style.equalsIgnoreCase(opbinding.encodingStyle)) {
										//binding style != op style (we don't support this)
										throw new WSDLBindingTranslatorException("Binding "+compiler_util.getQNameFromQName(binding.qname)+" has elements which specify more than one type of encoding style - this is currently unsupported");
									}
								}
								
								//add this to the list of operation bindings for this portType binding
								binding.opbindings.put(opbinding.opname,opbinding);
							}
							
							if (NamespaceTranslator.getName(opelem).equals("input")) {
								ArrayList inputelems = Util.getAllElements(opelem);
								for (int m = 0; m < inputelems.size(); m++) {
									Element inputelem = (Element)inputelems.get(m);
									nt.addNamespaces(inputelem);

									if (nt.checkQName(inputelem,SOAPNamespaces.NAMESPACE_SOAP_WSDL,"body")) {
										InputBinding inbinding = new InputBinding();
										
										inbinding.namespace = inputelem.getAttribute("namespace");
										inbinding.use = inputelem.getAttribute("use");
										inbinding.encodingStyle = inputelem.getAttribute("encodingStyle");
	
//TODO do we support non-literal encoding or not?
//										if (!inbinding.use.equalsIgnoreCase("literal")) {
//											throw new BindingTranslatorException("Binding "+binding.qname+" has elements which use non-literal encodings, only use=\"literal\" encoding is supported");
//										}
										
										/*
										if (inbinding.encodingStyle.length() > 0) {
											//input has an encoding style specified
											if (binding.encodingStyle.length() == 0) {
												//binding has no encoding - use this for entire binding
												binding.encodingStyle = inbinding.encodingStyle;
											} else if (!binding.encodingStyle.equalsIgnoreCase(inbinding.encodingStyle)) {
												//binding style != input style (we don't support this)
												throw new BindingTranslatorException("Binding "+binding.qname+" has elements which specify more than one type of encoding style - this is currently unsupported");
											}
										}*/
										
										opbinding.input = inbinding;
									}
									
									nt.removeNamespaces(inputelem);
								}

							} else if (NamespaceTranslator.getName(opelem).equals("output")) {
								ArrayList outputelems = Util.getAllElements(opelem);
								for (int m = 0; m < outputelems.size(); m++) {
									Element outputelem = (Element)outputelems.get(m);
									nt.addNamespaces(outputelem);
									
									if (nt.checkQName(outputelem,SOAPNamespaces.NAMESPACE_SOAP_WSDL,"body")) {
										OutputBinding outbinding = new OutputBinding();
										
										outbinding.namespace = outputelem.getAttribute("namespace");
										outbinding.use = outputelem.getAttribute("use");
										outbinding.encodingStyle = outputelem.getAttribute("encodingStyle");
										
/*										
										if (!outbinding.use.equalsIgnoreCase("literal")) {
											throw new BindingTranslatorException("Binding "+binding.qname+" has elements which use non-literal encodings, only use=\"literal\" encoding is supported");
										}
										
										if (outbinding.encodingStyle.length() > 0) {
											//output has an encoding style specified
											if (binding.encodingStyle.length() == 0) {
												//binding has no encoding - use this for entire binding
												binding.encodingStyle = outbinding.encodingStyle;
											} else if (!binding.encodingStyle.equalsIgnoreCase(outbinding.encodingStyle)) {
												//binding style != output style (we don't support this)
												throw new BindingTranslatorException("Binding "+binding.qname+" has elements which specify more than one type of encoding style - this is currently unsupported");
											}
										}*/
										
										opbinding.output = outbinding;
									}
									
									nt.removeNamespaces(outputelem);
								}

							}
							
							nt.removeNamespaces(opelem);
						}
					}
					
					nt.removeNamespaces(bindingelem);
				}
				
			} else if (NamespaceTranslator.getName(elem).equals("service")) {

				String servicename = elem.getAttribute("name");
				
				ArrayList subelems = Util.getAllElements(elem);
				for (int k = 0; k < subelems.size(); k++) {
					Element serviceelem = (Element)subelems.get(k);
					nt.addNamespaces(serviceelem);
					
					if (NamespaceTranslator.getName(serviceelem).equals("port")) {
//						log.logInfo(LOG_SOURCE,"Found port "+serviceelem.getAttribute("name"));

						String qbindingName = nt.qualify(serviceelem.getAttribute("binding"),false);
						PortTypeBinding ptbinding = (PortTypeBinding)bindings.get(qbindingName);
						
						if (ptbinding != null) {
//							throw new BindingTranslatorException("Binding "+qbindingName+" specified in service "+servicename+" not found");
						
							log.logInfo(LOG_SOURCE,"Found port "+serviceelem.getAttribute("name")+" which maps to SOAP binding "+compiler_util.getQNameFromQName(ptbinding.bindingName));

							ArrayList addresselems = Util.getAllElements(serviceelem);
							for (int j = 0; j < addresselems.size(); j++) {
								Element addresselem = (Element)addresselems.get(j);
								nt.addNamespaces(addresselem);

								if (nt.checkQName(addresselem,SOAPNamespaces.NAMESPACE_SOAP_WSDL,"address")) {
									log.logInfo(LOG_SOURCE,"Found SOAP address under SOAP port "+serviceelem.getAttribute("name")+" (address = "+addresselem.getAttribute("location")+")");
									ptbinding.address_location = addresselem.getAttribute("location");
								}
								
								nt.removeNamespaces(addresselem);
							}
						}
					}
					
					nt.removeNamespaces(serviceelem);
				}
				
			}
			
			
			nt.removeNamespaces(elem);
		}
		//
		// we don't need to care about service elements
		// we will never be copying around internal engine EPRs
		//
		
		nt.removeNamespaces(wsdl_definitions);
		
	}

	public XSDTypeTranslator[] getCodecs() {
		return codecs;
	}

	public String translateInvocation(NamespaceTranslator nt, String qportType, String bindingName, String operation, String eprFieldName, String correlationSetFieldNameIn, String correlationSetFieldNameOut, String inputFieldName, String outputFieldName) throws NamespaceException, WSDLBindingTranslatorException {
		StringBuffer sb = new StringBuffer();

		checkForBindingNamed(bindingName);
		checkForBinding(qportType);

		WSDLPortType wsdlPortType = wsdlmap.getPortType(qportType);
		WSDLOperation wsdlOp = wsdlPortType.getOperation(operation);
		
		PortTypeBinding binding = (PortTypeBinding)bindings.get(qportType);
		OperationBinding opbinding = (OperationBinding)binding.opbindings.get(operation);
		
		if (opbinding == null) {
			throw new WSDLBindingTranslatorException("No SOAP binding found for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		}

		String btype = null;
		
		if (binding.style.equalsIgnoreCase("rpc")) {
			btype = "RPC";
		} else if (binding.style.equalsIgnoreCase("document")) {
			btype = "Document";
		} else if (binding.style.equals("")) {
			btype = "Document";
		} else {
			throw new WSDLBindingTranslatorException("Unsupported SOAP binding style - only RPC or Document style bindings are supported");
		}
		
		boolean rpc_style = btype.equalsIgnoreCase("RPC");

		String soapAction = opbinding.soapAction;
		if (soapAction == null) soapAction = "";
		
		if (wsdlOp.getType() == WSDLOperation.TYPE_REQUEST_RESPONSE) {

			if (opbinding.input == null) {
				throw new WSDLBindingTranslatorException("No SOAP binding found for operation "+operation+" input element in port type "+compiler_util.getQNameFromQName(qportType));
			}
			if (opbinding.output == null) {
				throw new WSDLBindingTranslatorException("No SOAP binding found for operation "+operation+" output element in port type "+compiler_util.getQNameFromQName(qportType));
			}
			
			String inputType = wsdlOp.getMessageType("input");
			String outputType = wsdlOp.getMessageType("output");
			
			sb.append(outputFieldName+" = ("+outputType+")"+outputType+".fromXMLNoRootTags(SOAPClient.invokeRequestResponse("+eprFieldName+","+correlationSetFieldNameOut+",null,"+rpc_style+",\""+TranslatorUtils.toJavaStringStrict(opbinding.input.namespace)+"\",\""+operation+"\",\""+TranslatorUtils.toJavaStringStrict(soapAction)+"\","+inputType+".toXMLNoRootTags("+inputFieldName+",true)));");
		
		} else if (wsdlOp.getType() == WSDLOperation.TYPE_ONEWAY) {
		
			if (opbinding.input == null) {
				throw new WSDLBindingTranslatorException("No SOAP binding found for operation "+operation+" input element in port type "+compiler_util.getQNameFromQName(qportType));
			}

			String inputType = wsdlOp.getMessageType("input");

			sb.append("SOAPClient.invokeOneWay("+eprFieldName+","+correlationSetFieldNameOut+",null,"+rpc_style+",\""+TranslatorUtils.toJavaStringStrict(opbinding.input.namespace)+"\",\""+operation+"\",\""+TranslatorUtils.toJavaStringStrict(soapAction)+"\","+inputType+".toXMLNoRootTags("+inputFieldName+",true));");
			
		} else {
			throw new WSDLBindingTranslatorException("Unsupported WSDL operation type - only One-way and Request-response are supported (if this is supposed to be request-response, check ordering of <in> and <out>)");
		}
			
		
		return sb.toString();
	}

	public String translateReceive(NamespaceTranslator nt, String qportType, String bindingName, String operation, String messageExchange, String eprFieldName, String correlationSetFieldName, String inputFieldName) throws NamespaceException, WSDLBindingTranslatorException {
		StringBuffer sb = new StringBuffer();

		checkForBindingNamed(bindingName);
		checkForBinding(qportType);

		WSDLPortType wsdlPortType = wsdlmap.getPortType(qportType);
		WSDLOperation wsdlOp = wsdlPortType.getOperation(operation);
		
		PortTypeBinding binding = (PortTypeBinding)bindings.get(qportType);
		OperationBinding opbinding = (OperationBinding)binding.opbindings.get(operation);
		
		if (opbinding == null) {
			throw new WSDLBindingTranslatorException("No SOAP binding found for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		}

		String btype = null;
		
		if (binding.style.equalsIgnoreCase("rpc")) {
			btype = "RPC";
		} else if (binding.style.equalsIgnoreCase("document")) {
			btype = "Document";
		} else if (binding.style.equals("")) {
			btype = "Document";
		} else {
			throw new WSDLBindingTranslatorException("Unsupported SOAP binding style - only RPC or Document style bindings are supported");
		}
		
		boolean rpc_style = btype.equalsIgnoreCase("RPC");

		String soapAction = opbinding.soapAction;
		if (soapAction == null) soapAction = "";
		
		String inputType = wsdlOp.getMessageType("input");
		String outputType = wsdlOp.getMessageType("output");
		
		sb.append(inputFieldName+" = ("+inputType+")"+inputType+".fromXMLNoRootTags(SOAPServer.receive("+eprFieldName+",new BPELCorrelatable("+correlationSetFieldName+"),\""+messageExchange+"\",null,"+rpc_style+",\""+TranslatorUtils.toJavaStringStrict(opbinding.input.namespace)+"\",\""+operation+"\",\""+TranslatorUtils.toJavaStringStrict(soapAction)+"\"));");
		
		if (wsdlOp.getType() == WSDLOperation.TYPE_ONEWAY) {
			//this is a one-way operation - we should reply immediately
			sb.append("SOAPServer.reply("+eprFieldName+",\""+messageExchange+"\",null,"+rpc_style+",\""+TranslatorUtils.toJavaStringStrict(opbinding.input.namespace)+"\",\""+operation+"\",\""+TranslatorUtils.toJavaStringStrict(soapAction)+"\",null);\n");
		}
		
		return sb.toString();
	}

	public String translateReply(NamespaceTranslator nt, String qportType, String bindingName, String operation, String messageExchange, String eprFieldName, String correlationSetFieldName, String outputFieldName) throws NamespaceException, WSDLBindingTranslatorException {
		StringBuffer sb = new StringBuffer();

		checkForBindingNamed(bindingName);
		checkForBinding(qportType);

		WSDLPortType wsdlPortType = wsdlmap.getPortType(qportType);
		WSDLOperation wsdlOp = wsdlPortType.getOperation(operation);
		
		PortTypeBinding binding = (PortTypeBinding)bindings.get(qportType);
		OperationBinding opbinding = (OperationBinding)binding.opbindings.get(operation);
		
		if (opbinding == null) {
			throw new WSDLBindingTranslatorException("No SOAP binding found for operaion "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		}

		String btype = null;
		
		if (binding.style.equalsIgnoreCase("rpc")) {
			btype = "RPC";
		} else if (binding.style.equalsIgnoreCase("document")) {
			btype = "Document";
		} else if (binding.style.equals("")) {
			btype = "Document";
		} else {
			throw new WSDLBindingTranslatorException("Unsupported SOAP binding style - only RPC or Document style bindings are supported");
		}
		
		boolean rpc_style = btype.equalsIgnoreCase("RPC");

		String soapAction = opbinding.soapAction;
		if (soapAction == null) soapAction = "";
		
		String inputType = wsdlOp.getMessageType("input");
		String outputType = wsdlOp.getMessageType("output");
		
		sb.append("SOAPServer.reply("+eprFieldName+",\""+messageExchange+"\",null,"+rpc_style+",\""+TranslatorUtils.toJavaStringStrict(opbinding.output.namespace)+"\",\""+operation+"\",\""+TranslatorUtils.toJavaStringStrict(soapAction)+"\","+outputType+".toXMLNoRootTags("+outputFieldName+",true));");
		
		return sb.toString();
	}

	public String translateDefaultEPR(NamespaceTranslator nt, String qportType, String bindingName) throws NamespaceException, WSDLBindingTranslatorException {
		StringBuffer sb = new StringBuffer();

		checkForBindingNamed(bindingName);
		checkForBinding(qportType);
		
		PortTypeBinding ptbinding = (PortTypeBinding)bindings.get(qportType);
		if (bindingName != null) {
			//use a specific binding (and therefore use a specific binding's EPR)
			ptbinding = (PortTypeBinding)bindings.get(bindingName);
			
			if (ptbinding == null) {
				throw new WSDLBindingTranslatorException("Unable to find binding named "+bindingName);
			}
		}
		if (ptbinding == null) {
			throw new WSDLBindingTranslatorException("Unable to find binding for port type "+qportType);
		}
		if (ptbinding.address_location == null) {
			throw new WSDLBindingTranslatorException("No default SOAP service address and location defined for binding "+ptbinding.bindingName+" (for port type "+qportType+")");
		}
		

		sb.append("new WSEndpointReference(\""+TranslatorUtils.toJavaStringStrict(ptbinding.address_location)+"\")");
		
		return sb.toString();
	}

	StringBuffer picksb;

	String pickEprsId;
	String pickTransportUrisId;
	String pickRpcTypesId;
	String pickOperationNamespacesId;
	String pickOperationsId;
	String pickSoapActionsId;
	String pickCorrelationSetsId;
	String pickMessageExchangesId;

	String pickTimeoutId;
	int index;
	ArrayList inputTypes;
	ArrayList inputFields;
	ArrayList wsdlOps;

	public void translatePickStart(int onMessageCount, int onAlarmCount, String timeoutFieldLongMS) throws NamespaceException, WSDLBindingTranslatorException {

		picksb = new StringBuffer();
		index = 0;
		inputTypes = new ArrayList();
		inputFields = new ArrayList();
		wsdlOps = new ArrayList();
		
		this.pickTimeoutId = timeoutFieldLongMS;
		
		pickEprsId = "engine_soapbinding_pick_epr"+(PICK_ID);
		pickTransportUrisId = "engine_soapbinding_pick_transpuri"+(PICK_ID);
		pickRpcTypesId = "engine_soapbinding_pick_rpctype"+(PICK_ID);
		pickOperationNamespacesId = "engine_soapbinding_pick_opns"+(PICK_ID);
		pickOperationsId = "engine_soapbinding_pick_op"+(PICK_ID);
		pickSoapActionsId = "engine_soapbinding_pick_soapaction"+(PICK_ID);
		pickCorrelationSetsId = "engine_soapbinding_pick_correlation"+(PICK_ID);
		pickMessageExchangesId = "engine_soapbinding_pick_msgexchnge"+(PICK_ID);
		
		PICK_ID++;
		
		picksb.append("WSEndpointReference[] "+pickEprsId+" = new WSEndpointReference["+onMessageCount+"];\n");
		picksb.append("String[] "+pickTransportUrisId+" = new String["+onMessageCount+"];\n");
		picksb.append("boolean[] "+pickRpcTypesId+" = new boolean["+onMessageCount+"];\n");
		picksb.append("String[] "+pickOperationNamespacesId+" = new String["+onMessageCount+"];\n");
		picksb.append("String[] "+pickOperationsId+" = new String["+onMessageCount+"];\n");
		picksb.append("String[] "+pickSoapActionsId+" = new String["+onMessageCount+"];\n");
		picksb.append("Correlatable[] "+pickCorrelationSetsId+" = new Correlatable["+onMessageCount+"];\n");
		picksb.append("String[] "+pickMessageExchangesId+" = new String["+onMessageCount+"];\n");
	}

	public void translatePickOnMessage(NamespaceTranslator nt, String qportType, String bindingName, String operation, String messageExchange, String eprFieldName, String correlationSetFieldName, String inputFieldName) throws NamespaceException, WSDLBindingTranslatorException {

		checkForBindingNamed(bindingName);
		checkForBinding(qportType);

		WSDLPortType wsdlPortType = wsdlmap.getPortType(qportType);
		WSDLOperation wsdlOp = wsdlPortType.getOperation(operation);
		
		PortTypeBinding binding = (PortTypeBinding)bindings.get(qportType);
		OperationBinding opbinding = (OperationBinding)binding.opbindings.get(operation);
		
		if (opbinding == null) {
			throw new WSDLBindingTranslatorException("No SOAP binding found for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		}

		String btype = null;
		
		if (binding.style.equalsIgnoreCase("rpc")) {
			btype = "RPC";
		} else if (binding.style.equalsIgnoreCase("document")) {
			btype = "Document";
		} else if (binding.style.equals("")) {
			btype = "Document";
		} else {
			throw new WSDLBindingTranslatorException("Unsupported SOAP binding style - only RPC or Document style bindings are supported");
		}
		
		boolean rpc_style = btype.equalsIgnoreCase("RPC");

		String soapAction = opbinding.soapAction;
		if (soapAction == null) soapAction = "";
		
		String inputType = wsdlOp.getMessageType("input");
		String outputType = wsdlOp.getMessageType("output");

		inputTypes.add(inputType);
		inputFields.add(inputFieldName);
		wsdlOps.add(wsdlOp);
		
		String operationNamespaceID = "\""+TranslatorUtils.toJavaStringStrict(opbinding.input.namespace)+"\"";
		String operationID = "\""+operation+"\"";
		String soapActionID = "\""+TranslatorUtils.toJavaStringStrict(soapAction)+"\"";
		
		// 
		// append each conversation into the conversation array
		//
		picksb.append(pickEprsId+"["+index+"] = "+eprFieldName+";\n");
		picksb.append(pickTransportUrisId+"["+index+"] = null;\n");
		picksb.append(pickRpcTypesId+"["+index+"] = "+rpc_style+";\n");

		picksb.append(pickOperationNamespacesId+"["+index+"] = "+operationNamespaceID+";\n");
		picksb.append(pickOperationsId+"["+index+"] = "+operationID+";\n");
		picksb.append(pickSoapActionsId+"["+index+"] = "+soapActionID+";\n");
		picksb.append(pickCorrelationSetsId+"["+index+"] = new BPELCorrelatable("+correlationSetFieldName+");\n");
		picksb.append(pickMessageExchangesId+"["+index+"] = \""+messageExchange+"\";\n");
		
		index++;
	}

	public String translatePickFinish(String indexFieldName) throws NamespaceException, WSDLBindingTranslatorException {

		String pickedMessage = "engine_soapbinding_pick_message"+(PICK_ID++);

		//
		// Receive the message
		//
		
		picksb.append("String[] "+pickedMessage+" = SOAPServer.pick("+pickEprsId+","+pickCorrelationSetsId+","+pickMessageExchangesId+","+pickTransportUrisId+","+pickRpcTypesId+","+pickOperationNamespacesId+","+pickOperationsId+","+pickSoapActionsId+","+pickTimeoutId+");\n");
		
		//
		// Pop the successful conversation off the end of the message and set the indexFieldName accordingly
		//
		String pickedOperationNamespaceId = "engine_soapbinding_picked_opns"+(PICK_ID++);
		String pickedOperationId = "engine_soapbinding_picked_op"+(PICK_ID++);
		String pickedSoapActionId = "engine_soapbinding_picked_soapaction"+(PICK_ID++);

		picksb.append("if ("+pickedMessage+" == null) {\n");
		picksb.append("  //timeout\n");
		picksb.append("  "+indexFieldName+" = -1;\n");
		picksb.append("} else {\n");
		picksb.append("  //received a message\n");
		picksb.append("String "+pickedOperationNamespaceId+" = "+pickedMessage+"[0];\n");
		picksb.append("String "+pickedOperationId+" = "+pickedMessage+"[1];\n");
		picksb.append("String "+pickedSoapActionId+" = "+pickedMessage+"[2];\n");
		
		for (int i = 0; i < inputFields.size(); i++) {
			if (i > 0) {
				picksb.append("} else ");
			}
			picksb.append("if (");
			picksb.append(""+pickedOperationNamespaceId+" == "+pickOperationNamespacesId+"["+i+"]");
			picksb.append("&& "+pickedOperationId+" == "+pickOperationsId+"["+i+"]");
			picksb.append("&& "+pickedSoapActionId+" == "+pickSoapActionsId+"["+i+"]");
			picksb.append(") {\n");
			
			WSDLOperation wsdlOp = (WSDLOperation)wsdlOps.get(i);
			if (wsdlOp.getType() == WSDLOperation.TYPE_ONEWAY) {
				//this is a one-way operation - we should reply immediately
				picksb.append("SOAPServer.reply("+pickEprsId+"["+i+"],"+pickMessageExchangesId+"["+i+"],"+pickTransportUrisId+"["+i+"],"+pickRpcTypesId+"["+i+"],"+pickOperationNamespacesId+"["+i+"],"+pickOperationsId+"["+i+"],"+pickSoapActionsId+"["+i+"],null);\n");
			}

			String inputType = (String)inputTypes.get(i);
			String inputFieldName = (String)inputFields.get(i);
			
			picksb.append("  "+inputFieldName+" = ("+inputType+")"+inputType+".fromXMLNoRootTags("+pickedMessage+"[3]);\n");
			picksb.append("  "+indexFieldName+" = "+i+";\n");
			
			if (i == (inputFields.size()-1)) {
				picksb.append("}\n");
			}
		}
		picksb.append("}\n");
		
		return picksb.toString();
	}

	public String getDeclarations() {
		return "";
	}

	public String[] getImports() {
		return new String[]{
				getClass().getPackage().getName()+".*",
		};
	}

}