/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap;

import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.core.bpel.WSEndpointReference;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http.HTTPException;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.AlwaysCorrelatable;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.Correlatable;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.CorrelatedMultiQueuedBlockingMap;
import org.eclipse.stp.b2j.core.jengine.internal.utils.UIDPool;

/**
 * 
 * @author amiguel
 *
 * A SOAP client implementation used by the SOAP binding
 */
public class SOAPServer implements SOAPServerTransportListener {

	private static Correlatable[] no_correlation = new Correlatable[] { new AlwaysCorrelatable() };
	
	public static long TIMEOUT = 30000;
	
	private static SOAPServer INSTANCE = new SOAPServer();
	
	private static Object accepted_urls_LOCK = new Object();
	private static HashMap accepted_urls = new HashMap();
	
	private static final String REQ = ">";
	private static final String RES = "<";
	private static final String LISTENFOR = "@";
	
	UIDPool uids = new UIDPool();
	
	private static CorrelatedMultiQueuedBlockingMap conversations = new CorrelatedMultiQueuedBlockingMap();

	public static int acceptAddress(WSEndpointReference epr, String transportURI, boolean rpc_bound) throws Exception {

		synchronized(accepted_urls_LOCK) {
			Integer port = (Integer)accepted_urls.get(epr.getAddress());
			
			if (port != null) {
				//we've already done this
				return port.intValue();
			}
			
			URL url = new URL(epr.getAddress());
			if (url.getPort() == -1) {
				if (url.getProtocol().equalsIgnoreCase("https")) {
					port = new Integer(443);
				} else {
					port = new Integer(80);
				}
			} else {
				port = new Integer(url.getPort());
			}
			
//			port = new Integer(0);
			
			SOAPFactory.addServerTransportListenerForEPR(epr,transportURI,rpc_bound,INSTANCE);
		
			//dont register for listening twice - will just be filtered out anyway but this is quicker
			accepted_urls.put(epr.getAddress(),port);
			
			return port.intValue();
		}
	}
	
	private static String getDocumentRequestKey(int httpport, String soapAction) {
		if (soapAction == null) return httpport+":";

		StringBuffer sb = new StringBuffer();
		
		sb.append(REQ);

		sb.append(httpport);
		sb.append(":");
		sb.append(soapAction);

		return sb.toString();
	}
	private static String getDocumentResponseKey(int httpport, String soapAction, String replyConversation) {
		if (soapAction == null) return httpport+":";
		
		StringBuffer sb = new StringBuffer();

		sb.append(RES);
		sb.append(replyConversation);
		sb.append(":");
		
		sb.append(httpport);
		sb.append(":");
		sb.append(soapAction);
		
		return sb.toString();
	}
	private static String getRpcRequestKey(int httpport, String soapAction, String operationNamespace, String operation) {
		if (soapAction == null) soapAction = "";
		if (operation == null) operation = "";
		if (operationNamespace == null) operationNamespace = "";
		
		StringBuffer sb = new StringBuffer();
		
		sb.append(REQ);

		sb.append(httpport);
		sb.append(":");
		sb.append(soapAction);
		sb.append(":");
		sb.append(operationNamespace);
		sb.append(":");
		sb.append(operation);
		
		return sb.toString();
	}
	private static String getRpcResponseKey(int httpport, String soapAction, String operationNamespace, String operation, String replyConversation) {
		if (soapAction == null) soapAction = "";
		if (operation == null) operation = "";
		if (operationNamespace == null) operationNamespace = "";

		StringBuffer sb = new StringBuffer();
		
		sb.append(RES);
		sb.append(replyConversation);
		sb.append(":");
		
		sb.append(httpport);
		sb.append(":");
		sb.append(soapAction);
		sb.append(":");
		sb.append(operationNamespace);
		sb.append(":");
		sb.append(operation);
		
		return sb.toString();
	}
	
	public static String receive(WSEndpointReference epr, String transportUri, boolean rpc_type, String operationNamespace, String operation, String soapAction) throws Exception {
		return receive(epr,no_correlation[0],"",transportUri,rpc_type,operationNamespace,operation,soapAction);
	}

	public static String receive(WSEndpointReference epr, Correlatable correlatable, String replyConversation, String transportUri, boolean rpc_type, String operationNamespace, String operation, String soapAction) throws Exception {
		int port = acceptAddress(epr,transportUri,rpc_type);
		
		Correlatable[] correlation = new Correlatable[] {correlatable};
		
		String[] key = new String[1];
		String[][] reskey = new String[1][];
		if (rpc_type) {
			key[0] = getRpcRequestKey(port,operationNamespace,operation,soapAction);
			reskey[0] = new String[]{
						getRpcResponseKey(port,operationNamespace,operation,soapAction,replyConversation),
					};
		} else {
			key[0] = getDocumentRequestKey(port,soapAction);
			reskey[0] = new String[]{
						getDocumentResponseKey(port,soapAction,replyConversation),
					};	
		}
		
		if (SOAPFactory.DEBUG) System.out.println("SOAP Server: RECEIVE on "+key);
		
		Message reqmsg = conversations.get(key,correlation,0);
//		String receive_key = (String)reqmsg.pop();
		
		String msg = (String)reqmsg.get(0);
		String uid = (String)reqmsg.get(1);
		
		//tell it what to listen for as a reply
		Message listenformsg = new Message();
		for (int i = 0; i < reskey[0].length; i++) {
			listenformsg.append(reskey[0][i]);
		}
		conversations.put(uid,listenformsg);

		if (rpc_type) {
			msg = SOAPUtils.basicRemoveSoapEnvelopeRPC(msg);
		} else {
			msg = SOAPUtils.basicRemoveSoapEnvelopeDocument(msg);
		}
		
		return msg;
	}
	
	public static String[] pick(WSEndpointReference[] epr, String[] transportUri, boolean[] rpc_type, String[] operationNamespace, String[] operation, String[] soapAction, long timeout) throws Exception {
		Correlatable[] correlation = new Correlatable[epr.length];
		Arrays.fill(correlation,no_correlation[0]);
		String[] replyConversation = new String[epr.length];
		Arrays.fill(replyConversation,"");
		return pick(epr,correlation,replyConversation,transportUri,rpc_type,operationNamespace,operation,soapAction,timeout);
	}
	public static String[] pick(WSEndpointReference[] epr, Correlatable[] correlatable, String[] replyConversation, String[] transportUri, boolean[] rpc_type, String[] operationNamespace, String[] operation, String[] soapAction, long timeout) throws Exception {
//		for (int i = 0; i < epr.length; i++) {
//			acceptAddress(epr[i],transportUri[i],rpc_type[i]);
//		}
		
		String[] key = new String[epr.length];
		String[][] reskey = new String[epr.length][];
		for (int i = 0; i < key.length; i++) {
			int port = acceptAddress(epr[i],transportUri[i],rpc_type[i]);

			if (rpc_type[i]) {
				key[i] = getRpcRequestKey(port,operationNamespace[i],operation[i],soapAction[i]);
				reskey[i] = new String[]{
							getRpcResponseKey(port,operationNamespace[i],operation[i],soapAction[i],replyConversation[i]),
						};
			} else {
				key[i] = getDocumentRequestKey(port,soapAction[i]);
				reskey[i] = new String[]{
							getDocumentResponseKey(port,soapAction[i],replyConversation[i]),
						};
			}
		}
		
		Message reqmsg;
		try {
			if (SOAPFactory.DEBUG) {
				for (int i = 0; i < key.length; i++) {
					System.out.println("SOAP Server: PICK on "+key[i]);
				}
			}
			
			reqmsg = conversations.get(key,correlatable,timeout);
		} catch (InterruptedException e) {
			return null;
		}
		String receive_key = (String)reqmsg.pop();

		String[] msg = new String[5];
		
		for (int i = 0; i < key.length; i++) {
			if (key[i].equals(receive_key)) {
				msg[0] = operationNamespace[i];
				msg[1] = operation[i];
				msg[2] = soapAction[i];
				msg[3] = (String)reqmsg.get(0);
				
				String uid = (String)reqmsg.get(1);
				
				Message listenformsg = new Message();
				for (int k = 0; k < reskey[i].length; k++) {
					listenformsg.append(reskey[i][k]);
				}
				conversations.put(uid,listenformsg);
				
				if (rpc_type[i]) {
					msg[3] = SOAPUtils.basicRemoveSoapEnvelopeRPC(msg[3]);
				} else {
					msg[3] = SOAPUtils.basicRemoveSoapEnvelopeDocument(msg[3]);
				}
				
				break;
			}
		}
		
		return msg;
	}
	
	public static void reply(WSEndpointReference epr, String replyConversation, String transportUri, boolean rpc_type, String operationNamespace, String operation, String soapAction, String wsdl_message) throws Exception {
//Must already be listening
//		for (int i = 0; i < epr.length; i++) {
//			INSTANCE.acceptAddress(epr[i],transportUri[i]);
//		}
		int port = acceptAddress(epr,transportUri,rpc_type);

		String key;
		
		if (rpc_type) {
			//XXX is this right?
			operation = operation+"Response";
		
			key = getRpcResponseKey(port,operationNamespace,operation,soapAction,replyConversation);
			if (wsdl_message != null) {
				wsdl_message = SOAPUtils.basicWrapInSoapEnvelopeRPC(wsdl_message,operationNamespace,operation);
			} else {
				wsdl_message = "";
			}
		} else { 
			key = getDocumentResponseKey(port,soapAction,replyConversation);
			if (wsdl_message != null) {
				wsdl_message = SOAPUtils.basicWrapInSoapEnvelopeDocument(wsdl_message);
			} else {
				wsdl_message = "";
			}
		}
		
		Message msg = new Message();
		msg.append(wsdl_message);
		
		if (SOAPFactory.DEBUG) System.out.println("SOAP Server: REPLY on "+key);
		
		conversations.put(key,msg);
	}
	
	public String doRequest(int httpport, boolean rpc_type, String operationNamespace, String operation, String soapAction, String payload) throws Exception {
		
		try {
			
			String key;
			
			if (rpc_type) {
				key = getRpcRequestKey(httpport,soapAction,operationNamespace,operation);
			} else {
				key = getDocumentRequestKey(httpport,soapAction);
			}
			
			if (SOAPFactory.DEBUG) System.out.println("SOAP Server: putting request ["+key+"]");

			int myuid = uids.getUID();
			
			Message reqmsg = new Message();
			reqmsg.append(payload);
			reqmsg.append(LISTENFOR+myuid);
			
			try {
				//put request and wait for response (MAY THROW TIMEOUT EXCEPTION - InterruptedException)
				//it is OK for this to use the no_correlation correlatable, it is already listening on a unique ID
				Message listenformsg = conversations.putAndGet(key,reqmsg,new String[]{LISTENFOR+myuid},no_correlation,TIMEOUT);
				
				uids.releaseUID(myuid);
				
				//someone has accepted this response and will have specified a number of possible return conversations
				String[] listenforkeys = new String[listenformsg.length()];
				Correlatable[] corrs = new Correlatable[listenforkeys.length];
				for (int i = 0; i < listenforkeys.length; i++) {
					listenforkeys[i] = (String)listenformsg.get(i);
					corrs[i] = no_correlation[0];
				}
				

//				Message resmsg = conversations.get(listenforkeys,no_correlation,0);
				Message resmsg = conversations.get(listenforkeys,corrs,0);
							
				String response = (String)resmsg.get(0);
	
				if (SOAPFactory.DEBUG) System.out.println("SOAP Server: got response["+key+"]");
				
				return response;
			} catch (InterruptedException x) {

				uids.releaseUID(myuid);

				//put has been cancelled and we have timed out
				throw new HTTPException("Operation timed out ("+TIMEOUT+" ms)");
			}

		} catch (HTTPException e) {
			throw e;
		} catch (Exception e) {
			throw new HTTPException("\n-----\n"+SOAPUtils.getStackTrace(e)+"\n-----\n");
		}
	}
	
}