/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.mainengine;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.Switches;
import org.eclipse.stp.b2j.core.jengine.internal.core.Runner;
import org.eclipse.stp.b2j.core.jengine.internal.core.SubControllerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedHashMap;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedVariable;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedBarrier;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedMutex;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedSemaphore;
import org.eclipse.stp.b2j.core.jengine.internal.message.MTTransactionClient;
import org.eclipse.stp.b2j.core.jengine.internal.message.MTTransactionServer;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.message.TransactionListener;
import org.eclipse.stp.b2j.core.jengine.internal.utils.ID;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.jengine.internal.utils.UIDPool;

/**
 * 
 * @author amiguel
 * 
 * The distributed engine main ControllerConnection implementation, see the plugin architecture docs for more info.
 */
public abstract class ControllerConnection implements SubControllerInterface, TransactionListener, ThreadGroupListener {

//protected int host_id = -1;

protected int id;
private Object TEMP_ID_LOCK = new Object();
private int temp_id = 1;

protected Object shared_vars_LOCK = new Object();
//protected Object shared_sems_LOCK = new Object();

protected ArrayList shared_vars = new ArrayList();
//protected ArrayList shared_sems = new ArrayList();
protected HashMap shared_vars_map = new HashMap();
//protected HashMap shared_sems_map = new HashMap();

protected MTTransactionClient tc;
protected MTTransactionServer ts;


protected Object cached_hashmap_LOCK = new Object();
protected Object cached_mutex_LOCK = new Object();
protected Object cached_barrier_LOCK = new Object();
protected Object cached_semaphore_LOCK = new Object();
protected Object cached_hosts_LOCK = new Object();

protected HashMap cached_hashmap = new HashMap();
protected HashMap cached_mutex = new HashMap();
protected HashMap cached_barrier = new HashMap();
protected HashMap cached_semaphore = new HashMap();
protected String[] cached_hosts = null;

private Object waiting_map_LOCK = new Object();
private HashMap waiting_map = new HashMap();

private Object wait_directory_LOCK = new HashMap();
private Object[] wait_directory = new Object[100];

private Object msg_directory_LOCK = new HashMap();
private Object[] msg_directory = new Object[100];

private Object localmaps_LOCK = new Object();
private HashMap localmaps = new HashMap();

UIDPool uidpool = new UIDPool(20);

	private Object[] expandArray(Object[] rwait) {
		return expandArray(rwait,rwait.length-1);
	}
	private Object[] expandArray(Object[] rwait, int minindex) {
		int minlen = minindex+1;
		
		//expand by at least 100%
		minlen = Math.max(minlen*2,rwait.length * 2);

		Object[] nrwait = new Object[minlen];
		System.arraycopy(rwait,0,nrwait,0,rwait.length);
		return nrwait;
	}

	protected ControllerConnection(int id) {
		this.id = id;
		synchronized(TEMP_ID_LOCK) {
			temp_id = 100000 * id; //helps identify users from one machine to another
		}
	}
	
	private void pushStack(String note) {
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			Runner.pushStack("Engine Transaction - "+note);
		}
	}
	private void popStack() {
		if (Switches.APPEND_TRANSACTIONS_TO_CALLSTACK) {
			Runner.popStack();
		}
	}
	
	public HashMap getEngineLocalStorageMap(String name) {
		synchronized(localmaps_LOCK) {
			HashMap map = (HashMap)localmaps.get(name);
			if (map == null) {
				map = new HashMap();
				localmaps.put(name,map);
			}
			return map;
		}
	}

	class Terminator extends Thread {
		public void run() {
			try {
				Message m = new Message(Controller.CONTROLLER_ABORT);
				m = tc.doTransaction(m);
			} catch (Throwable t) {
				Logger.error("Failed to complete engine terminate request",t);
			}
		}
	}
	
	public void terminate() throws Exception {
		Logger.error("Terminate Request",new Throwable("Terminate Request STACKTRACE"));
		new Terminator().start();
		/*
		Message m = new Message(Controller.CONTROLLER_ABORT);
		pushStack("terminate");
		m = tc.doTransaction(m);
		popStack();
		*/
	}
	
	/**
	 * Can be used to wake up sleeping threads when the transport has failed - depends on the subclass's decision
	 * about whether this is appropriate or not
	 */
	public void threadException(Thread t, Throwable e) {
		//wake all runner join waiters
		Logger.warning("Engine about to die, waking all runner waits");
		try {
			synchronized(wait_directory_LOCK) {
				for (int i = 0; i < wait_directory.length; i++) {
					try {
						RunnerLock rlock = (RunnerLock)wait_directory[i];
						synchronized(rlock) {
							rlock.notifyAll();
						}
					} catch (Exception x) {
					}
				}
			}
		} catch (Exception x) {
		}
		//wake all semaphore waiters
		Logger.warning("Engine about to die, waking all semaphore waits");
		try {
			synchronized(waiting_map_LOCK) {
				ArrayList list = new ArrayList(waiting_map.values());
				for (int i = 0; i < list.size(); i++) {
					try {
						Object tlock = list.get(i);
						synchronized(tlock) {
							tlock.notifyAll();
						}
					} catch (Exception x) {
					}
				}
			}
		} catch (Exception x) {
		}
	}
	
	private void setLogLevel(boolean error, boolean warning, boolean info) throws Exception {
		Logger.PRINT_ERROR = error;
		Logger.PRINT_WARNING = warning;
		Logger.PRINT_INFO = info;
	}

	public void joinRunner(Long remote_id) throws Exception {
		int uid = uidpool.getUID();
		long rid = 0;
		
		synchronized(wait_directory_LOCK) {
			RunnerLock rlock = new RunnerLock();
			//create an ID for this runner based on the threads hashcode
//			rid = ID.CREATE_ID(id,Thread.currentThread().hashCode());
			rid = ID.CREATE_ID(id,uid);

			if (uid >= wait_directory.length) wait_directory = expandArray(wait_directory,uid);
			wait_directory[uid] = rlock;

//			wait_directory.put(new Long(rid),rlock);
		}	
		joinRunnerInternal(new Long(rid),remote_id);
	}

	public void joinRunnerInternal(Long waiter_id, Long remote_id) throws Exception {
		Message m = new Message(Controller.CONTROLLER_RUNNER_JOIN);
		m.append(waiter_id);
		m.append(remote_id);
		
		pushStack("joinRunner");
		m = tc.doTransaction(m);
		popStack();
		
		int mtype = m.getType();

		if (mtype != Controller.CONTROLLER_RUNNER_JOIN_OK
			&& mtype != Controller.CONTROLLER_RUNNER_JOIN_WILL_NOTIFY) {
			throw new Exception("Controller launch runner failed "+m);	
		}
		
		int uid = ID.RUNNER_ID(waiter_id.longValue());

		if (mtype == Controller.CONTROLLER_RUNNER_JOIN_WILL_NOTIFY) {
			
			RunnerLock rlock = null;

			synchronized(wait_directory_LOCK) {
				rlock = (RunnerLock)wait_directory[uid];
//				rlock = (RunnerLock)wait_directory.get(waiter_id);
			}	

			synchronized(rlock) {
				if (rlock.lock) {
					//wait has already been notified before we got a chance to wait
					rlock.lock = false;	
				} else {
					rlock.lock = true;
					pushStack("waiting for runner "+uid+" to finish");
					rlock.wait();
					popStack();
				}
			}

		} else {
			Logger.info("ControllerConnection: TOLD TO CONTINUE");	

		}
		
		//release the UID from the pool
		uidpool.releaseUID(uid);
	}

	public Message sendAndReceiveMessage(String conversation, Message msg, String conversationReturn, long mstimeout) throws Exception {
		Message tmp = new Message();
		tmp.append(conversationReturn);
		return sendAndReceiveMessage(conversation,msg,tmp,false,mstimeout);
	}
	public Message sendAndReceiveMessage(String conversation, Message msg, String conversationReturn) throws Exception {
		Message tmp = new Message();
		tmp.append(conversationReturn);
		return sendAndReceiveMessage(conversation,msg,tmp,false,0);
	}

	public Message sendAndReceiveMessage(String conversation, Message msg, Message conversationReturns, long mstimeout) throws Exception {
		return sendAndReceiveMessage(conversation,msg,conversationReturns,true,mstimeout);
	}
	public Message sendAndReceiveMessage(String conversation, Message msg, Message conversationReturns) throws Exception {
		return sendAndReceiveMessage(conversation,msg,conversationReturns,true,0);
	}
	
	public Message sendAndReceiveMessage(String conversation, Message msg, Message conversationReturns, boolean return_conversation, long mstimeout) throws Exception {
		
		int uid = uidpool.getUID();
		long rid = 0;
		synchronized(msg_directory_LOCK) {
			RunnerLock rlock = new RunnerLock();
			
			rid = ID.CREATE_ID(id,uid);
			if (uid >= msg_directory.length) msg_directory = expandArray(msg_directory,uid);
			msg_directory[uid] = rlock;
		}	
		
		Long waiter_id = new Long(rid);
		String timeout_conversation = waiter_id+":~TMT~";

		if (mstimeout > 0) {
			conversationReturns.append(timeout_conversation);
		} else if (mstimeout < 0) {
			mstimeout = 0;
		}
		
		
		Message m = new Message(Controller.CONTROLLER_MESSAGE_SEND_AND_RECEIVE);
		m.append(conversation);
		m.append(msg);
		m.append(rid);
		m.append(conversationReturns);
		
		pushStack("sendAndReceiveMessage");
		m = tc.doTransaction(m);
		popStack();

		int mtype = m.getType();

		if (mtype != Controller.CONTROLLER_MESSAGE_SEND_AND_RECEIVE_OK
			&& mtype != Controller.CONTROLLER_MESSAGE_SEND_AND_RECEIVE_WILL_NOTIFY) {
			throw new Exception("Controller launch runner failed "+m);	
		}

		Message received = null;
		
		if (mtype == Controller.CONTROLLER_MESSAGE_SEND_AND_RECEIVE_WILL_NOTIFY) {
			
			//piggy-backed notify transaction
			if (m.length() > 0) {
				pushStack("sendAndReceiveMessage - piggyBacked transaction");
				doTransaction((Message)m.get(0));
				popStack();
			}

			received = receiveMessageOnRunnerLock(uid,waiter_id,conversationReturns,mstimeout,timeout_conversation);
/*			
			RunnerLock rlock = null;

			synchronized(msg_directory_LOCK) {
				rlock = (RunnerLock)msg_directory[uid];
			}	

			synchronized(rlock) {
				if (rlock.lock) {
					//wait has already been notified before we got a chance to wait
					rlock.lock = false;	
				} else {
					rlock.lock = true;
					pushStack("waiting to receive message on "+conversationReturns);
					rlock.wait();
					popStack();
				}
			}

			received = (Message)rlock.data;
*/			
		} else {
			//piggy-backed notify transaction
			if (m.length() > 1) {
				pushStack("sendAndReceiveMessage - piggyBacked transaction");
				doTransaction((Message)m.get(1));
				popStack();
			}

			received = (Message)m.get(0);
		}

		
		//release the UID from the pool
		uidpool.releaseUID(uid);
		
		//this method always return the conversation on the end
		if (return_conversation) {
			if (received != null && conversationReturns.length() == 1) {
				received.append((String)conversationReturns.get(0));
			}
		}
		
		return received;
	}
	
	public void sendMessage(String conversation, Message msg) throws Exception {
		//this is like a notify
		Message m = new Message(Controller.CONTROLLER_MESSAGE_PUBLISH);
		m.append(conversation);
		m.append(msg);
		m.append(ID.CREATE_ID(id,0));
		
//Logger.direct("SUBCONTROLLER: Sending Message "+msg);
		
		pushStack("sendMessage");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.length() > 0) {
			pushStack("sendMessage - piggyBacked transaction");
			doTransaction((Message)m.get(0));
			popStack();
		}
	}
	
	public Message receiveMessage(String conversation) throws Exception {
		return receiveMessage(conversation,0);
	}
	public Message receiveMessage(String conversation, long mstimeout) throws Exception {
		Message tmp = new Message();
		tmp.append(conversation);
		return receiveMessage(tmp,mstimeout,false);
	}

	public Message receiveMessage(Message conversations) throws Exception {
		return receiveMessage(conversations,0);
	}
	public Message receiveMessage(Message conversations, long mstimeout) throws Exception {
		return receiveMessage(conversations,mstimeout,true);
	}
	
	private Message receiveMessage(Message conversations, long mstimeout, boolean return_conversation) throws Exception {
		//create an ID for this thread
		int uid = uidpool.getUID();
		long rid = 0;
		synchronized(msg_directory_LOCK) {
			RunnerLock rlock = new RunnerLock();
			
			rid = ID.CREATE_ID(id,uid);
			if (uid >= msg_directory.length) msg_directory = expandArray(msg_directory,uid);
			msg_directory[uid] = rlock;
		}	
		return receiveMessageInternal(new Long(rid),conversations,mstimeout,return_conversation);
	}

	private Message receiveMessageInternal(Long waiter_id, Message conversations, long mstimeout, boolean return_conversation) throws Exception {

		String timeout_conversation = waiter_id+":~TMT~";

		if (mstimeout > 0) {
			conversations.append(timeout_conversation);
		} else if (mstimeout < 0) {
			mstimeout = 0;
		}
		
		Message m = new Message(Controller.CONTROLLER_MESSAGE_CONSUME);
		m.append(waiter_id);
		m.append(conversations);
	
		pushStack("receiveMessageInternal");
		m = tc.doTransaction(m);
		popStack();

		int mtype = m.getType();

		if (mtype != Controller.CONTROLLER_MESSAGE_CONSUME_OK
			&& mtype != Controller.CONTROLLER_MESSAGE_CONSUME_WILL_NOTIFY) {
			throw new Exception("Controller launch runner failed "+m);	
		}

		Message received = null;
		
		int uid = ID.RUNNER_ID(waiter_id.longValue());

		if (mtype == Controller.CONTROLLER_MESSAGE_CONSUME_WILL_NOTIFY) {
			
			received = receiveMessageOnRunnerLock(uid,waiter_id,conversations,mstimeout,timeout_conversation);
			
		} else {
//			Logger.info("ControllerConnection: TOLD TO CONTINUE");	
			received = (Message)m.get(0);
		}

		//release the UID from the pool
		uidpool.releaseUID(uid);
		
//Logger.direct("SUBCONTROLLER: Received Message "+received);
		
		if (return_conversation) {
			//this method always return the conversation on the end
			if (received != null && conversations.length() == 1) {
				received.append((String)conversations.get(0));
			}
		}
		
		return received;
	}
	
	private Message receiveMessageOnRunnerLock(int uid, Long waiter_id, Message conversations, long mstimeout, String timeout_conversation) throws InterruptedException, Exception {
			RunnerLock rlock = null;

			synchronized(msg_directory_LOCK) {
				rlock = (RunnerLock)msg_directory[uid];
			}	

			synchronized(rlock) {
				if (rlock.lock) {
					//wait has already been notified before we got a chance to wait
					rlock.lock = false;	
				} else {
					rlock.lock = true;
					pushStack("waiting to receive message on "+conversations);

//					rlock.wait();
					rlock.wait(mstimeout);

					popStack();
				}

				if (rlock.data == null) {
					//we timed out
					//we are not waiting any more
					rlock.lock = false;	
				}
			}//end sync
			
			if (rlock.data == null) {
				//we timed out - send ourselves a timeout message
				sendMessage(timeout_conversation,new Message());

				synchronized(rlock) {
					if (rlock.lock) {
						//wait has already been notified before we got a chance to wait
						rlock.lock = false;	
					} else {
						rlock.lock = true;
						pushStack("waiting to receive timeout message on "+conversations);

						rlock.wait();
						
						popStack();
					}

					Message real_received = (Message)rlock.data;
					
					if (real_received.get(real_received.length()-1).equals(timeout_conversation)) {
						//we got our timeout message
						rlock.data = null;

					} else {

						//we need to listen for our timeout message so nothing else gets it
						Message timeout_conversations = new Message();
						timeout_conversations.append(timeout_conversation);

						Message timeout_cleanup = new Message(Controller.CONTROLLER_MESSAGE_CONSUME);
						timeout_cleanup.append(waiter_id);
						timeout_cleanup.append(timeout_conversations);
						
						pushStack("receiveMessageInternal - timeout cleanup");
						timeout_cleanup = tc.doTransaction(timeout_cleanup);
						popStack();

						//should NEVER be a WILL_NOTIFY message because we've already sent the message
						if (timeout_cleanup.getType() == Controller.CONTROLLER_MESSAGE_CONSUME_WILL_NOTIFY) {
							Logger.direct("ERROR - got a WILL NOTIFY back from a timeout cleanup message");
						}
						
						//restore the real message we received
						rlock.data = real_received;
					}
				}//end sync
			}

			return (Message)rlock.data;
	}
	
	public Message launchRunner(int count, String method, int host_index) throws Exception {
		return launchRunner(count,method,host_index,new ArrayList(1));
	}
	public Message launchRunner(int count, String method, int host_index, String[] args) throws Exception {
		ArrayList list = new ArrayList(args.length);
		for (int i = 0; i < list.size(); i++) {
			list.add(args[i]);
		}
		return launchRunner(count,method,host_index,list);
	}
	public Message launchRunner(int count, String method, int host_index, List args) throws Exception {
		Message margs = new Message();
		for (int i = 0; i < args.size(); i++) {
			margs.append((String)args.get(i));
		}
		return launchRunner(count,method,host_index,margs);
	}
	public Message launchRunner(int count, String method, int host_index, Message args) throws Exception {
		Message m = new Message(Controller.CONTROLLER_LAUNCH_RUNNER);
		m.append(count);
		m.append(method);
		m.append(host_index);
		
		m.append(args);
//		for (int i = 0; i < args.size(); i++) {
//			m.append((String)args.get(i));	
//		}
		
		pushStack("launchRunner");
		m = tc.doTransaction(m);
		popStack();

		if (m.getType() != Controller.CONTROLLER_LAUNCH_RUNNER_OK) {
			throw new Exception("Controller launch runner failed "+m);	
		}
		
		return (Message)m.get(0);
	}

	public SharedVariable newVariable(String name, int type, boolean dirty) throws Exception {

		Message m = null;
		if (dirty) {
			m = new Message(Controller.CONTROLLER_MAKE_DIRTY_VARIABLE);
		} else {
			m = new Message(Controller.CONTROLLER_MAKE_VARIABLE);
		}
		m.append(name);
		m.append(type);

		pushStack("newVariable");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.getType() != Controller.CONTROLLER_MAKE_VARIABLE_OK
			&& m.getType() != Controller.CONTROLLER_MAKE_VARIABLE_EXISTS) {
			throw new Exception("Controller make variable failed "+m);	
		}
		
		Integer vid = (Integer)m.get(0);

		SharedVariable sv = new SharedVariable(this,name,vid.intValue(),type,null);
		
		return sv;
	}

	public SharedSemaphore newSemaphore(String name, int initial) throws Exception {
		Message m = new Message(Controller.CONTROLLER_MAKE_SEMAPHORE);
		m.append(name);
		m.append(initial);

		pushStack("newSemaphore");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.getType() != Controller.CONTROLLER_MAKE_SEMAPHORE_OK
			&& m.getType() != Controller.CONTROLLER_MAKE_SEMAPHORE_EXISTS) {
			throw new Exception("Controller make semaphore failed "+m);	
		}
		
		Integer vid = (Integer)m.get(0);
		SharedSemaphore sm = new SharedSemaphore(this,name,vid.intValue());
		
		return sm;
	}

	public SharedHashMap newHashMap(String name) throws Exception {
		Message m = null;
		m = new Message(Controller.CONTROLLER_MAKE_HASHMAP);

		m.append(name);

		pushStack("newHashMap");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.getType() != Controller.CONTROLLER_MAKE_HASHMAP_OK
			&& m.getType() != Controller.CONTROLLER_MAKE_HASHMAP_EXISTS) {
			throw new Exception("Controller make hashmap failed "+m);	
		}
		
		Integer hid = (Integer)m.get(0);

		SharedHashMap shmp = new SharedHashMap(this,name,hid.intValue());
		
		return shmp;
	}

	public SharedBarrier newBarrier(String name, int size) throws Exception {
		SharedSemaphore waiting = newSemaphore(name+"%_sBARRIER_WAITING",0);
		SharedSemaphore barrier = newSemaphore(name+"%_sBARRIER_BARRIER",0);
		SharedVariable barrier_size = newVariable(name+"%_sBARRIER_SIZE",SharedVariable.INTEGER,true);
	
		barrier_size.setValue(size);
		barrier_size.varStore();
		
		SharedBarrier sbarrier = new SharedBarrier(waiting,barrier,size);
		
		return sbarrier;	
	}
	
	public SharedMutex newMutex(String name) throws Exception {
		SharedSemaphore msem = newSemaphore(name+"%_sMUTEX",1);
		
		SharedMutex smutex = new SharedMutex(msem);
		
		return smutex;	
	}

	public SharedBarrier getBarrier(String name) throws Exception {
		SharedBarrier sbarrier = null;
		synchronized(cached_barrier_LOCK) {
			sbarrier = (SharedBarrier)cached_barrier.get(name);	
		}

		if (sbarrier == null) {		
			SharedSemaphore waiting = getSemaphore(name+"%_sBARRIER_WAITING");
			SharedSemaphore barrier = getSemaphore(name+"%_sBARRIER_BARRIER");
			SharedVariable barrier_size = getVariable(name+"%_sBARRIER_SIZE");
		
			barrier_size.varFetch();
			
			sbarrier = new SharedBarrier(waiting,barrier,barrier_size.getValueInt());
			synchronized(cached_barrier_LOCK) {
				cached_barrier.put(name,sbarrier);	
			}
		}
		
		return sbarrier;	
	}
	
	public SharedMutex getMutex(String name) throws Exception {
		SharedMutex smutex = null;
		synchronized(cached_mutex_LOCK) {
			smutex = (SharedMutex)cached_mutex.get(name);
		}
		if (smutex == null) {
			SharedSemaphore msem = getSemaphore(name+"%_sMUTEX");
			
			smutex = new SharedMutex(msem);
			synchronized(cached_mutex_LOCK) {
				cached_mutex.put(name,smutex);	
			}
		}
		
		return smutex;	
	}

	public SharedHashMap getHashMap(String name) throws Exception {
		
		SharedHashMap map = null;
		synchronized(cached_hashmap_LOCK) {
			map = (SharedHashMap)cached_hashmap.get(name);	
		}

		if (map == null) {
			Message m = new Message(Controller.CONTROLLER_GET_HASHMAP);
			m.append(name);
			
			pushStack("getHashMap");
			m = tc.doTransaction(m);	
			popStack();
			
			if (m.getType() != Controller.CONTROLLER_GET_HASHMAP_OK) {
				throw new Exception("Controller get hashmap failed "+m);	
			}
			
			Integer hid = (Integer)m.get(0);
	
			map = new SharedHashMap(this,name,hid.intValue());
			synchronized(cached_hashmap_LOCK) {
				cached_hashmap.put(name,map);	
			}
		}
		
		return map;

	}

	public SharedVariable getVariable(String name) throws Exception {
		Variable v;
		synchronized(shared_vars_LOCK) {
			v = (Variable)shared_vars_map.get(name);
		}
		SharedVariable sv = new SharedVariable(this,name,v.getID(),v.getType(),null);
		return sv;
	}

	public SharedSemaphore getSemaphore(String name) throws Exception {
		
		SharedSemaphore sem = null;
		
		synchronized(cached_semaphore_LOCK) {
			sem = (SharedSemaphore)cached_semaphore.get(name);	
		}
		
		if (sem == null) {
			Message m = new Message(Controller.CONTROLLER_GET_SEMAPHORE);
			m.append(name);
	
			pushStack("getSemaphore");
			m = tc.doTransaction(m);
			popStack();
			
			if (m.getType() != Controller.CONTROLLER_GET_SEMAPHORE_OK) {
				throw new Exception("Controller get semaphore failed "+m);	
			}
			
			Integer sid = (Integer)m.get(0);
	
			sem = new SharedSemaphore(this,name,sid.intValue());
			synchronized(cached_semaphore_LOCK) {
				cached_semaphore.put(name,sem);	
			}
		}
		
		return sem;
	}

	public void hashmapSet(int id, String key, Object value) throws Exception {
		Message m = new Message(Controller.CONTROLLER_PUT_HASHMAP);
		m.append(id);
		m.append(key);
		
		if (value instanceof String) {
			m.append((String)value);	
		} else if (value instanceof byte[]) {
			m.append((byte[])value);
		} else if (value instanceof Integer) {
			m.append((Integer)value);
		} else if (value instanceof Long) {
			m.append((Long)value);
		} else if (value instanceof Double) {
			m.append((Double)value);
		} else if (value instanceof Message) {
			m.append((Message)value);
		} else {
			throw new Exception("value is of no recognised type");
		}

		pushStack("hashmapSet");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.getType() != Controller.CONTROLLER_PUT_HASHMAP_OK) {
			throw new Exception("Controller store hashmap failed "+m);	
		}
	}

	public String[] hashmapGetKeys(int id) throws Exception {
		Message m = new Message(Controller.CONTROLLER_RETRIEVE_HASHMAP_KEYLIST);
		m.append(id);
		
		pushStack("hashmapGetKeys");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.getType() != Controller.CONTROLLER_RETRIEVE_HASHMAP_KEYLIST_OK) {
			throw new Exception("Controller retrieve hashmap keylist failed "+m);
		}
		
		String[] keys = new String[m.length()];
		
		for (int i = 0; i < keys.length; i++) {
			keys[i] = (String)m.get(i);
		}
		return keys;
	}
	
	public Object hashmapGet(int id, String key) throws Exception {
		Message m = new Message(Controller.CONTROLLER_RETRIEVE_HASHMAP);
		m.append(id);
		m.append(key);

		pushStack("hashmapGet");
		m = tc.doTransaction(m);
		popStack();

		//value in hashmap was null
		if (m.getType() == Controller.CONTROLLER_RETRIEVE_HASHMAP_NULL) {
			return null;	
		}
		
		if (m.getType() != Controller.CONTROLLER_RETRIEVE_HASHMAP_OK) {
			throw new Exception("Controller retrieve hashmap failed "+m);	
		}

		return m.get(0);

	}
	
	public void hashmapClear(int id) throws Exception {
		Message m = new Message(Controller.CONTROLLER_CLEAR_HASHMAP);
		m.append(id);
		
		pushStack("hashmapClear");
		m = tc.doTransaction(m);
		popStack();

		//value in hashmap was null
		if (m.getType() != Controller.CONTROLLER_CLEAR_HASHMAP_OK) {
			throw new Exception("Controller clear hashmap failed "+m);	
		}
		
	}

	private class StackDumper extends Thread {
		boolean trace;
		int msg_id;
		public StackDumper(boolean trace, int msg_id) {
			this.trace = trace;
			this.msg_id = msg_id;
		}
		public void run() {
			try {
				if (trace) {
					Message m = new Message(Controller.CONTROLLER_RUNNER_STACKTRACE);
					m.append(msg_id);
					m = tc.doTransaction(m);
				} else {
					Message m = new Message(Controller.CONTROLLER_RUNNER_STACKDUMP);
					m = tc.doTransaction(m);
				}
			} catch (Throwable t) {
				Logger.error("Failed to complete stackdump request",t);
			}
		}
	}

	public void doRunnerStackTrace(int stack_message_id) throws Exception {
		new StackDumper(true,stack_message_id).start();
	}
	
	public void doRunnerStackDump() throws Exception {
		//overridden in SubController because notifications are more likely to get through
		new StackDumper(false,0).start();
	}

	public String getHost() throws Exception {
		return "unknown(client)";
	}
	
	public String[] getHosts() throws Exception {
		
		String[] hosts = cached_hosts;

		if (hosts == null) {
			Message m = new Message(Controller.CONTROLLER_GET_HOSTS);
			
			pushStack("getHosts");
			m = tc.doTransaction(m);
			popStack();
	
			if (m.getType() != Controller.CONTROLLER_GET_HOSTS_OK) {
				throw new Exception("Controller get hosts failed "+m);	
			}
			
			Integer count = (Integer)m.get(0);
			
			hosts = new String[count.intValue()];
			
			for (int i = 0; i < hosts.length; i++) {
				hosts[i] = (String)m.get(i+1);	
			}
			
			cached_hosts = hosts;
		}
		
		return hosts; 
	}

	public String[] getVariableNames() throws Exception {
		Message m = new Message(Controller.CONTROLLER_GET_VARIABLE_LIST);
		
		pushStack("getVariableNames");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.getType() != Controller.CONTROLLER_GET_VARIABLE_LIST_OK) {
			throw new Exception("Controller get variable list failed "+m);
		}
		
		Integer count = (Integer)m.get(0);
		
		String[] vars = new String[count.intValue()];
		
		for (int i = 0; i < vars.length; i++) {
			vars[i] = (String)m.get(i+1);	
		}
		
		return vars;
	}

	public void storeVariable(int id, int type, Object newval) throws Exception {
		Message m = new Message(Controller.CONTROLLER_STORE_VARIABLE);
		m.append(id);
		m.append(type);

		switch (type) {
			case SharedVariable.INTEGER:
				m.append((Integer)newval);
				break;
			case SharedVariable.STRING:
				m.append((String)newval);
				break;
			case SharedVariable.BYTES:
				m.append((byte[])newval);
				break;
			case SharedVariable.LONG:
				m.append((Long)newval);
				break;
			case SharedVariable.MESSAGE:
				m.append((Message)newval);
				break;
			case SharedVariable.DOUBLE:
				m.append((Double)newval);
//				long l = Double.doubleToRawLongBits(((Double)newval).doubleValue());
//				m.append(l);
				break;
			case SharedVariable.FLOAT:
				m.append(((Number)newval).doubleValue());
//				int n = Float.floatToRawIntBits(((Float)newval).floatValue());
//				m.append(n);
				break;
			default:
				throw new Exception("variable has no recognised type");
		}

		pushStack("storeVariable");
		m = tc.doTransaction(m);
		popStack();
				
		if (m.getType() != Controller.CONTROLLER_STORE_VARIABLE_OK) {
			throw new Exception("Controller store variable failed "+m);	
		}
	}
	
	public boolean isVariableDirty(int id) throws Exception {
		Variable v;
		synchronized(shared_vars_LOCK) {
			v = (Variable)shared_vars.get(id-1);
		}
		return v.isDirty();
	}

	public Object fetchVariable(int id, int type) throws Exception {
		Variable v;
		synchronized(shared_vars_LOCK) {
			v = (Variable)shared_vars.get(id-1);
		}

//Debugger.warning("(FETCH!) VARIABLE "+id+" DIRTY = "+v.isDirty()+" "+v.hashCode());
		
		synchronized(v.LOCK) {
		
		if (v.getDirtyType()) {
			if (v.isDirty()) {
				v.setDirty(false);
				Object o = realFetchVariable(id,type);
				v.setValue(o);
				return o;
			} else {
				return v.getValue();	
			}
		} else {
			return realFetchVariable(id,type);	
		}
		}
	}

	public Object realFetchVariable(int id, int type) throws Exception {
		Message m = new Message(Controller.CONTROLLER_FETCH_VARIABLE);
		m.append(this.id);
		m.append(id);
		m.append(type);

		pushStack("realFetchVariable");
		m = tc.doTransaction(m);
		popStack();

		if (m.getType() != Controller.CONTROLLER_FETCH_VARIABLE_OK) {
			throw new Exception("Controller make variable failed "+m);	
		}

		Integer vtype_o = (Integer)m.get(0);
		int vtype = vtype_o.intValue();

		Object newval = m.get(1);

		switch (vtype) {
			case SharedVariable.INTEGER:
				return((Integer)newval);
			case SharedVariable.STRING:
				return((String)newval);
			case SharedVariable.BYTES:
				return((byte[])newval);
			case SharedVariable.MESSAGE:
				return((Message)newval);
			case SharedVariable.LONG:
				return((Long)newval);
			case SharedVariable.DOUBLE:
				return((Double)newval);
//				long l = ((Long)newval).longValue();
//				return new Double(Double.longBitsToDouble(l));
			case SharedVariable.FLOAT:
				return(new Float(((Double)newval).floatValue()));
//				int n = ((Integer)newval).intValue();
//				return new Float(Float.intBitsToFloat(n));
			default:
				throw new Exception("unrecognised returned variable type");
		}
		
	}

	public void signalSemaphore(int id, int n) throws Exception {
		Message m = new Message(Controller.CONTROLLER_SIGNAL_SEMAPHORE);
		m.append(id);
		m.append(n);

		pushStack("signalSemaphore");
		m = tc.doTransaction(m);
		popStack();
		
		if (m.getType() != Controller.CONTROLLER_SIGNAL_SEMAPHORE_OK) {
			throw new Exception("Controller signal semaphore failed "+m);	
		}
	}

	public void waitSemaphore(int id, int n) throws Exception {
		long rid = 0;
		synchronized(TEMP_ID_LOCK) {
			rid = ID.CREATE_ID(this.id,temp_id);
			temp_id++;	
		}

		Message m = new Message(Controller.CONTROLLER_WAIT_SEMAPHORE);
		
		m.append(id);
		m.append(n);
		m.append(rid);

		Integer key = new Integer( (int)(rid & 0x00000000FFFFFFFF) );

		//this means we dont signal before we get a chance to wait
		Object tlock = new Object();
//		Object tlock = Thread.currentThread();
		synchronized (tlock) {
			
			//
			// we have to populate the hashmap before the transaction in case we get another transaction
			// that tries to signal this runner.
			//
			// we can't synchronize on anything but the thread otherwise we get deadlock.
			//
			// so we populate the hashmap,
			// but before we get to wait the other transaction tries to signal this runner
			// the other transaction finds the thread ok (used to be a problem) but then is forced
			// to wait for the thread lock (which we have inside here already)
			//
			// this means we always get to wait before a notify happens
			//
			synchronized (waiting_map_LOCK) {
				//TODO get rid of this - its inefficient and its debugging
//				if (waiting_map.get(key) != null) {
//					Logger.error("ALREADY EXISTS IN MAP!??");
//				}
				waiting_map.put(key,tlock);
			}
			
			pushStack("waitSemaphore");
			m = tc.doTransaction(m);
			popStack();

			//these have to be inside the transaction lock otherwise the following can occur:
			//
			// we wait on the semaphore and it tells us to wait
			// we get the message back and exit the sync but before we can wait we do a whole
			// other transaction where we signal the semaphore and this exact runner is told to
			// run.
			//
			// because we haven't waited yet and entered this thread into the wait map, we try to
			// signal a runner that hasn't waited and we get a NullPointerException
			//
			if (m.getType() == Controller.CONTROLLER_WAIT_SEMAPHORE_OK) {
				//do nothing
				synchronized (waiting_map_LOCK) {
					waiting_map.remove(key);
				}
			} else if (m.getType() == Controller.CONTROLLER_WAIT_SEMAPHORE_WILL_NOTIFY) {
				pushStack("waiting on semaphore "+id);
				tlock.wait();
				popStack();

//this doesnt work - we sometimes try to notify a runner that hasnt been added to the hashmap				
//				waitObject(rid);
			} else {
				throw new Exception("Controller wait semaphore failed "+m);	
			}

		}	
				
	}

	/**
	 * Wait the thread, it will be woken with a notify message
	 */
//	private void waitObject(long lid) throws InterruptedException {
//		
//		Integer key = new Integer( (int)(lid & 0x00000000FFFFFFFF) );
//		
//		Thread t = Thread.currentThread();
//		synchronized (MAP_LOCK) {
//			waiting_map.put(key,t);
//		}
//		
//		synchronized(Thread.currentThread()) {
//			t.wait();
//		}
//	}

	/**
	 * Wake the specified thread
	 */
	private void signalObject(long lid) throws Exception {

		Integer key = new Integer( (int)(lid & 0x00000000FFFFFFFF) );

		Object t = null;
		synchronized (waiting_map_LOCK) {
			t = (Object)waiting_map.remove(key);
		}

		synchronized(t) {		
			t.notify();
		}
	}

	public Message doTransaction(Message m) {
		Message r;
		int type = m.getType();

		if (type == SubController.SUBCONTROLLER_SIGNAL_RUNNER) {
			Logger.info("SUBCONTROLLER: Signal Runner");

			Long id = (Long)m.get(0);
			long lid = id.longValue();
			
			try {
				if (ID.CONTROLLER_ID(lid) != this.id) {
					throw new Exception("SIGNAL RUNNER SENT TO WRONG CONTROLLER!");
				}
				
				signalObject(lid);
				r = new Message(SubController.SUBCONTROLLER_SIGNAL_RUNNER_OK);
			} catch (Exception e) {
				r = new Message(SubController.SUBCONTROLLER_SIGNAL_RUNNER_FAIL);	
				r.append(Logger.getStackTrace(e));
				r.append(ID.ID_HEX(id.longValue()));

			}

		} else if (type == SubController.SUBCONTROLLER_RECEIVE_MESSAGE) {
			Logger.info("SUBCONTROLLER: Receive Message");
			
			Long rid = (Long)m.get(0);
			Message data = (Message)m.get(1);

//Logger.direct("SUBCONTROLLER: Received Message Consume Notification "+data);
			
			RunnerLock rlock = null;

			synchronized(msg_directory_LOCK) {
				//we can receive a message before anything has waited on that message
				int uid = ID.RUNNER_ID(rid.longValue());
				rlock = (RunnerLock)msg_directory[uid];
				if (rlock == null) {
					rlock = new RunnerLock();
					msg_directory[uid] = rlock;
				}
//				rlock = (RunnerLock)msg_directory.get(rid);
			}	
			
			synchronized(rlock) {
				//set the message
				rlock.data = data;
				
				if (rlock.lock) {
					//the thread is waiting on this object
					rlock.lock = false;	
					rlock.notify();
				} else {
					//we are notifying before it has had a chance to wait
					rlock.lock = true;
				}
			}
			
			r = new Message(SubController.SUBCONTROLLER_RECEIVE_MESSAGE_OK);
			
		} else if (type == SubController.SUBCONTROLLER_MAKE_VARIABLE) {
			Logger.info("SUBCONTROLLER: Make Variable");
			Integer btype = (Integer)m.get(0);
			boolean dirty = (btype.intValue() == Controller.CONTROLLER_MAKE_DIRTY_VARIABLE);
			
			String vname = (String)m.get(1);
			Integer vtype = (Integer)m.get(2);
			Integer vid = (Integer)m.get(3);

			try {
				Variable v = new Variable(vname, vid.intValue(), vtype.intValue());
				if (dirty) {
					v.setDirtyType(true);
					v.setDirty(true);
				} else {
					v.setDirtyType(false);
					v.setDirty(true);
				}

				synchronized(shared_vars_LOCK) {
					shared_vars.add(v);
					shared_vars_map.put(vname,v);
				}
				
				r = new Message(SubController.SUBCONTROLLER_MAKE_VARIABLE_OK);
			} catch (Exception e) {
				r = new Message(SubController.SUBCONTROLLER_MAKE_VARIABLE_FAIL);
				r.append(Logger.getStackTrace(e));	
			}

		} else if (type == SubController.SUBCONTROLLER_NOTIFY_DIRTY) {
			Logger.info("SUBCONTROLLER: Notify Dirty");
			
			Integer vid = (Integer)m.get(0);

			Variable v;
			synchronized(shared_vars_LOCK) {
				v = (Variable)shared_vars.get(vid.intValue()-1);
			}
			v.setDirty(true);
//Debugger.warning("VARIABLE "+vid+" SET TO DIRTY "+v.isDirty()+" "+v.hashCode());
			r = new Message(SubController.SUBCONTROLLER_NOTIFY_DIRTY_OK);

		} else if (type == SubController.SUBCONTROLLER_RECEIVE_MESSAGE) {
			Logger.info("SUBCONTROLLER: Receive Message");
			
			Long rid = (Long)m.get(0);
			Message data = (Message)m.get(1);

//Logger.direct("SUBCONTROLLER: Received Message Consume Notification "+data);
			
			RunnerLock rlock = null;

			synchronized(msg_directory_LOCK) {
				//we can receive a message before anything has waited on that message
				int uid = ID.RUNNER_ID(rid.longValue());
				rlock = (RunnerLock)msg_directory[uid];
				if (rlock == null) {
					rlock = new RunnerLock();
					msg_directory[uid] = rlock;
				}
//				rlock = (RunnerLock)msg_directory.get(rid);
			}	
			
			synchronized(rlock) {
				//set the message
				rlock.data = data;
				
				if (rlock.lock) {
					//the thread is waiting on this object
					rlock.lock = false;	
					rlock.notify();
				} else {
					//we are notifying before it has had a chance to wait
					rlock.lock = true;
				}
			}
			
			r = new Message(SubController.SUBCONTROLLER_RECEIVE_MESSAGE_OK);
			
			
		} else if (type == SubController.SUBCONTROLLER_WAKE_RUNNER) {
			Logger.info("SUBCONTROLLER: Wake Runner");
			
			Long rid = (Long)m.get(0);

			RunnerLock rlock = null;

			synchronized(wait_directory_LOCK) {
				rlock = (RunnerLock)wait_directory[ID.RUNNER_ID(rid.longValue())];
//				rlock = (RunnerLock)wait_directory.get(rid);
			}	

			synchronized(rlock) {
				if (rlock.lock) {
//					Debugger.info("SUBCONTROLLER: Wake Runner "+rid+" notifying lock "+rlock);
					//the thread is waiting on this object
					rlock.lock = false;	
					rlock.notify();
				} else {
//					Debugger.info("SUBCONTROLLER: Wake Runner "+rid+" setting "+rlock+" to true instead of notifying");
					//we are notifying before it has had a chance to wait
					rlock.lock = true;
				}
			}

			r = new Message(SubController.SUBCONTROLLER_WAKE_RUNNER_OK);
			
		} else if (type == SubController.SUBCONTROLLER_SET_LOG_LEVEL) {
			Logger.info("SUBCONTROLLER: Set Log Level");
			
			try {
				setLogLevel(((Integer)m.get(0)).intValue()==1,((Integer)m.get(1)).intValue()==1,((Integer)m.get(2)).intValue()==1);
				r = new Message(SubController.SUBCONTROLLER_SET_LOG_LEVEL_OK);
			} catch (Exception e) {
				r = new Message(SubController.SUBCONTROLLER_SET_LOG_LEVEL_FAIL);
				r.append(Logger.getStackTrace(e));				
			}
			
					
		} else {
			r = new Message(SubController.SUBCONTROLLER_UNRECOGNISED_MESSAGE);	
		}

		return r;
	} 

}