package org.eclipse.stp.core.infrastructure.emf;

import java.util.Map;
import java.util.WeakHashMap;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;

/**
 * A FeatureAdapter provides the feature that a given EObject should be
 * persisted as. Each feature represents a possible substitution group from a
 * (potentially) custom package.
 * 
 * <p>
 * <b>Usage:</b> <table>
 * <tr>
 * <td> <code>
 * ACustomModelImplementation implementation = 
 * 		ACustomModelFactory.eINSTANCE.createACustomModelImplementation();
 * implementation.setCustomProperty(...);
 * 
 * FeatureAdapter.setFeature(implementation, 
 * 		ACustomModelPackage.eINSTANCE.getDocumentRoot_ImplementationACustomModel());
 * </code>
 * </td>
 * </tr>
 * </table>
 * <p>
 * For more information, take a look at Properties Implementation in
 * <i>com.ibm.ccl.soa.core.tests</i>. In that plugin, see schemas/ for defining
 * the EMF model, and props-ext-model/ for the generated version of the model.
 * </p>
 * 
 * </p>
 * 
 * @since 1.0
 * 
 */
public final class FeatureAdapter extends AdapterImpl implements Adapter {

   private static final Map   featureAdapters = new WeakHashMap();

   /**
    * The adapter type to search for when looking for the FeatureAdapter.
    * 
    * @see EcoreUtil#getAdapter(java.util.List, java.lang.Object)
    */
   public static final String ADAPTER_TYPE    = FeatureAdapter.class.getName();

   /**
    * 
    * <p>
    * This method may return null.
    * </p>
    * 
    * @param anObject
    *           The object representing a substitution group.
    * @return The structural feature that should be used for the feature map
    *         when setting this implementation.
    */
   public static EStructuralFeature getFeature(EObject anObject) {
      FeatureAdapter adapter = findAdapter(anObject);
      return adapter != null ? adapter.getFeature() : null;
   }

   /**
    * Add a FeatureAdapter on the given object for the given feature if not
    * already present. An invocation of this method for a feature already noted
    * on the object will not result in any modifications.
    * 
    * @param anObject
    *           The object representing a substitution group.
    * @param aFeature
    *           The feature that represents how the element should be persisted.
    */
   public static void setFeature(EObject anObject, EStructuralFeature aFeature) {
      synchronized (anObject) {
         FeatureAdapter adapter = findAdapter(anObject);
         if (adapter == null || adapter.getFeature() != aFeature) {
            adapter = getAdapterForFeature(aFeature);
            anObject.eAdapters().add(adapter);
         }
      }
   }

   /**
    * 
    * @param anObject
    *           The object to search for the FeatureAdapter.
    * @return The adapter for {@link #ADAPTER_TYPE} from the given object, or
    *         null if not found.
    */
   public static FeatureAdapter findAdapter(EObject anObject) {
      return (FeatureAdapter) EcoreUtil.getAdapter(anObject.eAdapters(),
            ADAPTER_TYPE);
   }

   private static FeatureAdapter getAdapterForFeature(
         EStructuralFeature aFeature) {
      FeatureAdapter adapter = (FeatureAdapter) featureAdapters.get(aFeature);
      if (adapter != null)
         return adapter;
      synchronized (featureAdapters) {
         adapter = (FeatureAdapter) featureAdapters.get(aFeature);
         if (adapter == null)
            featureAdapters.put(aFeature, (adapter = new FeatureAdapter(
                  aFeature)));
      }
      return adapter;
   }

   private EStructuralFeature feature;

   private FeatureAdapter(EStructuralFeature aFeature) {
      feature = aFeature;
   }

   public boolean isAdapterForType(Object type) {
      return type == ADAPTER_TYPE;
   }

   /**
    * @return The structural feature held by this FeatureAdapter.
    */
   public EStructuralFeature getFeature() {
      return feature;
   }

}
