/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*
 * Created on Mar 3, 2004
 * 
 * To change the template for this generated file go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
package org.eclipse.stp.core.internal.infrastructure.emfworkbench;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jem.util.emf.workbench.EMFWorkbenchContextBase;
import org.eclipse.jem.util.emf.workbench.ISynchronizerExtender;
import org.eclipse.jem.util.emf.workbench.ProjectResourceSet;
import org.eclipse.jem.util.emf.workbench.WorkbenchURIConverter;
import org.eclipse.jem.util.logger.proxy.Logger;
import org.eclipse.stp.core.infrastructure.emf.EditModelEvent;
import org.eclipse.stp.core.infrastructure.emf.IEMFWorkbenchContext;
import org.eclipse.stp.core.infrastructure.emf.IEditModel;
import org.eclipse.stp.core.infrastructure.emf.IResourceAssistant;
import org.eclipse.stp.core.infrastructure.emf.WorkbenchResourceHelper;
import org.eclipse.stp.core.internal.infrastructure.emf.EMFInfrastructurePlugin;
import org.eclipse.stp.core.internal.infrastructure.emf.EditModel;
import org.eclipse.stp.core.internal.infrastructure.emf.ProjectResourceSetEditImpl;

public class EMFWorkbenchContext extends EMFWorkbenchContextBase implements
      ISynchronizerExtender, IEMFWorkbenchContext {

   private Map       editableModels      = new HashMap();

   protected Adapter resourceSetListener;

   private final Set managedResourceSets = new HashSet();

   private final Set assistants          = new HashSet();

   /**
    * @param aProject
    */
   public EMFWorkbenchContext(IProject aProject) {
      super(aProject);

      getResourceSet();
   }

   public static EMFWorkbenchContext getInstance(IProject aProject) {
      return (EMFWorkbenchContext) EMFWorkbenchEditContextFactory.sINSTANCE
            .createEMFContext(aProject, null /* contributor */);
   }

   public void connect(ResourceSet aResourceSet) {
      if (aResourceSet != null) {
         ResourceSetListener adapter = (ResourceSetListener) EcoreUtil
               .getAdapter(aResourceSet.eAdapters(), EMFWorkbenchContext.this);
         if (adapter == null) {
            managedResourceSets.add(aResourceSet);
            aResourceSet.eAdapters().add(getResourceSetListener());
            if (shouldNotifyEditModels()) {
               EditModelEvent event = new EditModelEvent(
                     EditModelEvent.ADDED_RESOURCE, aResourceSet.getResources());
               notifyEditModels(event);
            }
         }
      }
   }

   public void disconnect(ResourceSet aResourceSet) {
      if (aResourceSet != null) {

         ResourceSetListener adapter = (ResourceSetListener) EcoreUtil
               .getAdapter(aResourceSet.eAdapters(), EMFWorkbenchContext.this);
         if (adapter != null) {
            aResourceSet.eAdapters().remove(getResourceSetListener());

            if (shouldNotifyEditModels()) {
               EditModelEvent event = new EditModelEvent(
                     EditModelEvent.REMOVED_RESOURCE, aResourceSet
                           .getResources());
               notifyEditModels(event);
            }
         }
      }
   }

   // TODO implement a reference counting mechanism here
   public void installResourceAssistant(IResourceAssistant anAssistant/*
                                                                         * ,
                                                                         * ResourceHandlerPriority
                                                                         * priority
                                                                         */) {

      synchronized (this) {
         if (assistants.contains(anAssistant))
            return;
         assistants.add(anAssistant);

         ResourceSet[] managedSets = anAssistant.getManagedResourceSets();
         if (managedSets != null) {
            for (int i = 0; i < managedSets.length; i++) {
               connect(managedSets[i]);
            }
         }
         getResourceSet().addFirst(anAssistant);
      }
   }

   public void uninstallResourceHandler(IResourceAssistant anAssistant) {
      getResourceSet().remove(anAssistant);
   }

   public ProjectResourceSet getResourceSet() {
      if (resourceSet == null) {
         resourceSet = (ProjectResourceSet) WorkbenchResourceHelper
               .getResourceSet(getProject());
         initializeResourceSet(resourceSet);
      }
      return resourceSet;
   }

   public ResourceSet[] getResourceSets() {
      return (ResourceSet[]) managedResourceSets
            .toArray(new ResourceSet[managedResourceSets.size()]);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.wst.common.internal.emfworkbench.EMFWorkbenchContext#initializeResourceSet(org.eclipse.wst.common.internal.emfworkbench.ProjectResourceSet)
    */
   protected void initializeResourceSet(ProjectResourceSet aResourceSet) {

      // Resource.Factory.Registry registry = new
      // DefaultOverridableResourceFactoryRegistry();
      // Resource.Factory factory = new ReferencedXMIFactoryImpl();
      // registry.getExtensionToFactoryMap().put(
      // Resource.Factory.Registry.DEFAULT_EXTENSION, factory);
      // // add xmi because other plugins are registering it globally
      // registry.getExtensionToFactoryMap().put("xmi", factory);
      // //$NON-NLS-1$
      // aResourceSet.setResourceFactoryRegistry(registry);

      aResourceSet.setURIConverter(createURIConverter(aResourceSet));
      aResourceSet.add(new WorkspaceResourceHandler());

      aResourceSet
            .setResourceFactoryRegistry(EMFResourceFactoryRegistry.sINSTANCE);

      // added so we can be informed of closes to the project.
      aResourceSet.getSynchronizer().addExtender(this);

      startListeningToResourceSet();

   }

   /**
    * This is the API that clients should use when they have an intent to modify
    * a particular resource. You should only access the resources through the
    * J2EEEditModel that is returned by this method if you have the intent to
    * modify.
    * 
    * @see J2EEEditModel
    */
   public final IEditModel getEditModel(String editModelID) {
      synchronized (this) {
         IEditModel editModel = getExistingEditModel(editModelID);

         if (null == editModel || editModel.isDisposed()) {
            editModel = createEditModel(editModelID);
            cacheEditModel(editModel);
         }
         return editModel;
      }
   }

   protected IEditModel getExistingEditModel(String editModelID) {
      IEditModel editModel = null;
      editModel = (IEditModel) editableModels.get(editModelID);
      return editModel;
   }

   /** 
    */
   protected EditModel createEditModel(String editModelID) {
      return new EditModel(editModelID, this);
   }

   /**
    * Insert the method's description here. Creation date: (4/16/2001 12:25:39
    * PM)
    * 
    * @return java.util.List
    */
   public void cacheEditModel(IEditModel editModel) {
      editableModels.put(editModel.getEditModelLabel(), editModel);
   }

   protected void discardAllEditModels() {
      discardModels(editableModels.values());
   }

   private void discardModels(Collection editModels) {
      if (editModels != null && !editModels.isEmpty()) {
         // Make a copy for safety against concurrent modification
         Iterator it = new ArrayList(editModels).iterator();
         while (it.hasNext()) {
            ((EditModel) it.next()).dispose();
         }
      }
   }

   public void removeEditModel(EditModel editModel, boolean readOnly) {
      // The best way would be to recompute the cache id, but we don't care
      // because the edit model should only be cached once anyway
      editableModels.values().remove(editModel);
   }

   /**
    * Notify all editModels of the change.
    */
   protected void notifyEditModels(EditModelEvent anEvent) {
      if (anEvent == null)
         return;
      List editModelsToNotify = new ArrayList((editableModels.values()));
      EditModel editModel;
      for (int i = 0; i < editModelsToNotify.size(); i++) {
         editModel = (EditModel) editModelsToNotify.get(i);
         try {
            editModel.resourceChanged(anEvent);
         } catch (Exception e) {
            EMFInfrastructurePlugin.log(new Status(IStatus.ERROR,
                  EMFInfrastructurePlugin.PLUGIN_ID, 0, e.getMessage(), e));
         }
      }
   }

   protected boolean shouldNotifyEditModels() {
      return !editableModels.isEmpty();
   }

   protected Adapter getResourceSetListener() {
      if (resourceSetListener == null)
         resourceSetListener = new ResourceSetListener();
      return resourceSetListener;
   }

   public static final String RESOURCESET_ADAPTER_TYPE = ResourceSetListener.class
                                                             .getName();

   protected class ResourceSetListener extends AdapterImpl {

      public boolean isAdapterForType(Object type) {

         return EMFWorkbenchContext.this == type;
      }

      /*
       * @see Adapter#notifyChanged(new
       *      ENotificationImpl((InternalEObject)Notifier,
       *      int,(EStructuralFeature) EObject, Object, Object, int))
       */
      public void notifyChanged(Notification notification) {
         switch (notification.getEventType()) {
            case Notification.ADD:
               addedResource((Resource) notification.getNewValue());
               break;
            case Notification.REMOVE:
               removedResource((Resource) notification.getOldValue());
               break;
            case Notification.REMOVE_MANY:
               removedResources((List) notification.getOldValue());
               break;
         }
      }
   }

   /**
    * Notify all editModels of the change.
    */
   private void addedResource(Resource addedResource) {
      if (shouldNotifyEditModels()) {
         EditModelEvent event = new EditModelEvent(
               EditModelEvent.ADDED_RESOURCE, Collections
                     .singletonList(addedResource));
         notifyEditModels(event);
      }
   }

   /**
    * Notify all editModels of the change.
    */
   private void removedResource(Resource removedResource) {
      if (shouldNotifyEditModels()) {
         EditModelEvent event = new EditModelEvent(
               EditModelEvent.REMOVED_RESOURCE, Collections
                     .singletonList(removedResource));
         notifyEditModels(event);
      }
   }

   /**
    * Notify all editModels of the change.
    */
   private void removedResources(List removedResources) {
      if (shouldNotifyEditModels()) {
         EditModelEvent event = new EditModelEvent(
               EditModelEvent.REMOVED_RESOURCE, removedResources);
         notifyEditModels(event);
      }
   }

   private void startListeningToResourceSet() {
      connect(getResourceSet());
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.wst.common.internal.emfworkbench.ISynchronizerExtender#projectChanged(org.eclipse.core.resources.IResourceDelta)
    */
   public void projectChanged(IResourceDelta delta) {
      // default nothing
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.wst.common.internal.emfworkbench.ISynchronizerExtender#projectClosed()
    */
   public void projectClosed() {
      discardAllEditModels();
      resourceSet = null;
      EMFWorkbenchEditContextFactory.sINSTANCE
            .removeCachedProject(getProject());
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.wst.common.internal.emfworkbench.EMFWorkbenchContextBase#createURIConverter(org.eclipse.wst.common.internal.emfworkbench.ProjectResourceSet)
    */
   public WorkbenchURIConverter createURIConverter(
         ProjectResourceSet aResourceSet) {
      return new CompatibilityWorkbenchURIConverterImpl(getProject(),
            aResourceSet.getSynchronizer());
   }

   public ProjectResourceSet createResourceSet() {
      if (project == null)
         throw new IllegalStateException(
               "Attempt to create resource set with null project"); //$NON-NLS-1$
      return new ProjectResourceSetEditImpl(project);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.wst.common.internal.emfworkbench.EMFWorkbenchContextBase#deleteFile(org.eclipse.emf.ecore.resource.Resource)
    */
   public void deleteFile(Resource resource) {
      try {
         WorkbenchResourceHelper.deleteResource(resource);
      } catch (CoreException ex) {
         Logger.getLogger().logError(ex);
      }

   }

}
