/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.internal.introspection;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.internal.introspection.query.AttributesExpression;
import org.eclipse.stp.core.internal.introspection.query.CombinedQuery;
import org.eclipse.stp.core.internal.introspection.query.IModelQuery;
import org.eclipse.stp.core.internal.introspection.query.StructuralFeatureExpression;
import org.eclipse.stp.core.introspection.IDynamicModelIntrospector;
import org.eclipse.stp.core.sca.SCAObject;

/**
 * Only partially implemented as of 12/01
 * 
 * @since 1.0
 * 
 */
public class IntrospectionRecorder implements IDynamicModelIntrospector {

   private final AttributeSetStack attributeSetStack = new AttributeSetStack();

   private final ModelDeltaStack   deltaStack        = new ModelDeltaStack();

   private SCAObject               rootObject;

   /*
    * (non-Javadoc)
    * 
    * @see IDynamicModelIntrospector#startIntrospection()
    */
   public void startIntrospection(EStructuralFeature theRootFeature,
         SCAObject theRootObject) {

      reset();
      rootObject = theRootObject;
      deltaStack.push(new CombinedQuery());
   }

   /*
    * (non-Javadoc)
    * 
    * @see IDynamicModelIntrospector#startFeature(org.eclipse.emf.ecore.EStructuralFeature)
    */
   public void startFeature(EStructuralFeature feature, EClass type,
         EStructuralFeature substitutionGroupFeature) {
      startFeature(feature, type, substitutionGroupFeature,
            IDynamicModelIntrospector.BATCH);
   }

   /*
    * (non-Javadoc)
    * 
    * @see IDynamicModelIntrospector#startFeature(EStructuralFeature, EClass,
    *      EStructuralFeature, int)
    */
   public void startFeature(EStructuralFeature feature, EClass type,
         EStructuralFeature substitutionGroupFeature, int announcementType) {

      deltaStack.push(new StructuralFeatureExpression(feature, type,
            substitutionGroupFeature, announcementType));
      attributeSetStack.push(new AttributeSet());

   }

   /*
    * (non-Javadoc)
    * 
    * @see IDynamicModelIntrospector#attribute(org.eclipse.emf.ecore.EAttribute,
    *      java.lang.Object)
    */
   public void attribute(EAttribute attribute, Object value, boolean isUniqueKey) {
      attributeSetStack.peek().add(attribute, value, isUniqueKey);
   }

   /*
    * (non-Javadoc)
    * 
    * @see IDynamicModelIntrospector#endFeature(org.eclipse.emf.ecore.EStructuralFeature)
    */
   public void endFeature(EStructuralFeature feature) {

      AttributeSet attributes = attributeSetStack.pop();
      IModelQuery matchesAttributes = new AttributesExpression(attributes);

      IModelQuery child = deltaStack.pop();
      child.addChildQuery(matchesAttributes);

      IModelQuery parent = deltaStack.peek();
      parent.addChildQuery(child);
   }

   /*
    * (non-Javadoc)
    * 
    * @see IDynamicModelIntrospector#endIntrospection()
    */
   public void endIntrospection() {
      if (!deltaStack.isEmpty() && rootObject != null)
         deltaStack.peek().applyDelta(rootObject.getEObject());
      Assert.isTrue(attributeSetStack.isEmpty());

   }

   public void removeUntouched(EStructuralFeature feature) {
      if (!deltaStack.isEmpty())
         deltaStack.peek().removeUntouched(rootObject, feature);
   }

   public void reset() {
      while (!deltaStack.isEmpty())
         deltaStack.pop();
      rootObject = null;

   }

}
