/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.internal.introspection.module;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModelFactory;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.WorkbenchResourceHelper;
import org.eclipse.stp.core.internal.STPCorePlugin;
import org.eclipse.stp.core.introspection.AbstractComponentTypeIntrospector;
import org.eclipse.stp.core.introspection.IComponentTypeIntrospector;
import org.eclipse.stp.core.resources.CompositeScribblerDomain;
import org.eclipse.stp.core.sca.EntryPoint;
import org.eclipse.stp.core.sca.ExternalService;
import org.eclipse.stp.core.sca.Module;
import org.eclipse.stp.core.sca.ModuleReference;
import org.eclipse.stp.core.sca.ModuleService;
import org.eclipse.stp.core.sca.OverrideOptions;
import org.eclipse.stp.core.sca.Reference;
import org.eclipse.stp.core.sca.SCACoreRoot;
import org.eclipse.stp.core.sca.SCAFactory;
import org.eclipse.stp.core.sca.SCAPackage;
import org.eclipse.stp.core.sca.Service;

public class ModuleComponentTypeIntrospector extends
      AbstractComponentTypeIntrospector implements IComponentTypeIntrospector {

   private IEditModelScribbler      scribbler;

   private Module                   module;

   private CompositeScribblerDomain domain;

   private boolean                  disposed = false;

   protected void doInit() {

      URI platformURI = getImplementationURI().replacePrefix(
            URI.createURI("comptype:/"), URI.createURI("platform:/resource/")); //$NON-NLS-1$//$NON-NLS-2$
      IFile moduleFile = WorkbenchResourceHelper.getPlatformFile(platformURI);
      domain = new CompositeScribblerDomain(moduleFile);

   }

   protected IStatus doIntrospection(int theFieldType, List theCurrentValues) {
      if (disposed)
         return STPCorePlugin
               .createErrorStatus(
                     0,
                     "ModuleComponentTypeIntrospector is already disposed in doIntrospection()!", null); //$NON-NLS-1$

      Module module = getModule();
      if (module == null)
         return hasValidComponentType();

      switch (theFieldType) {
         case SCAPackage.COMPONENT_TYPE__SERVICES:
            List entryPoints = module.getEntryPoints();
            ModuleService moduleService = null;
            EntryPoint entryPoint = null;
            List services = new ArrayList();
            for (int i = 0; i < entryPoints.size(); i++) {
               entryPoint = (EntryPoint) entryPoints.get(i);
               moduleService = SCAFactory.eINSTANCE.createModuleService();
               moduleService.setEntryPoint(entryPoint);
               services.add(moduleService);
            }
            theCurrentValues.addAll(services);
            break;

         case SCAPackage.COMPONENT_TYPE__REFERENCES:
            List externalServices = module.getExternalServices();
            ModuleReference moduleReference = null;
            ExternalService externalService = null;
            List references = new ArrayList();
            for (int i = 0; i < externalServices.size(); i++) {
               externalService = (ExternalService) externalServices.get(i);
               if (externalService.isSetOverridable()
                     && externalService.getOverridable().getValue() == OverrideOptions.NO)
                  continue;
               moduleReference = SCAFactory.eINSTANCE.createModuleReference();
               moduleReference.setExternalService(externalService);
               references.add(moduleReference);
            }
            theCurrentValues.addAll(references);
            break;
      }

      return Status.OK_STATUS;
   }

   public IStatus hasValidComponentType() {
      if (getModule() != null)
         return Status.OK_STATUS;
      return STPCorePlugin
            .createErrorStatus(
                  0,
                  "Valid component type not available because the 'Module' implementation is not available.", null); //$NON-NLS-1$
   }

   public IStatus onModelChange(int theFieldType, int theChangeType,
         Object theChangedObject) {
      if (disposed)
         return STPCorePlugin
               .createErrorStatus(
                     0,
                     "ModuleComponentTypeIntrospector is already disposed in onModelChange()!", null); //$NON-NLS-1$

      EObject eObj = (EObject) theChangedObject;
      switch (theFieldType) {
         case SCAPackage.COMPONENT_TYPE__SERVICES:
            switch (theChangeType) {
               case Notification.ADD:
                  if (eObj instanceof ModuleService) {
                     ModuleService mService = (ModuleService) eObj;
                     Module module = getModule();
                     if (module == null)
                        return hasValidComponentType();
                     if (mService.getEntryPoint() != null) {
                        EntryPoint epoint = mService.getEntryPoint();
                        if (!module.getEntryPoints().contains(epoint)) {
                           module.getEntryPoints().add(epoint);
                        } else {
                           return Status.OK_STATUS;
                        }
                     } else {
                        EntryPoint epoint = SCAFactory.eINSTANCE
                              .createEntryPoint();
                        epoint.setInterface(mService.getInterface());
                        epoint.setName(mService.getName());
                        mService.setEntryPoint(epoint);
                        module.getEntryPoints().add(epoint);
                     }
                  } else if (eObj instanceof Service) {
                     Service service = (Service) eObj;

                     EntryPoint epoint = SCAFactory.eINSTANCE
                           .createEntryPoint();
                     epoint.setInterface(service.getInterface());
                     epoint.setName(service.getName());
                     module.getEntryPoints().add(epoint);
                  }
                  break;
               case Notification.REMOVE:
                  if (eObj instanceof ModuleService) {
                     ModuleService mService = (ModuleService) eObj;
                     Module module = getModule();
                     if (module == null)
                        return hasValidComponentType();
                     if (mService.getEntryPoint() != null) {
                        EntryPoint epoint = mService.getEntryPoint();
                        if (module.getEntryPoints().contains(epoint)) {
                           module.getEntryPoints().remove(epoint);
                        } else {
                           return Status.OK_STATUS;
                        }
                     } else {
                        EntryPoint epoint = module.getEntryPoint(mService
                              .getName());
                        if (epoint != null)
                           module.getEntryPoints().remove(epoint);
                     }
                  } else if (eObj instanceof Service) {
                     Service service = (Service) eObj;

                     EntryPoint epoint = module
                           .getEntryPoint(service.getName());
                     if (epoint != null)
                        module.getEntryPoints().remove(epoint);
                  }
                  break;
            }
            break;
         case SCAPackage.COMPONENT_TYPE__REFERENCES:
            switch (theChangeType) {
               case Notification.ADD:
                  if (eObj instanceof ModuleReference) {
                     ModuleReference mReference = (ModuleReference) eObj;
                     Module module = getModule();
                     if (mReference.getExternalService() != null) {
                        ExternalService eservice = mReference
                              .getExternalService();
                        if (!module.getExternalServices().contains(eservice)) {
                           module.getExternalServices().add(eservice);
                        } else {
                           return Status.OK_STATUS;
                        }
                     } else {
                        ExternalService eservice = SCAFactory.eINSTANCE
                              .createExternalService();
                        eservice.setInterface(mReference.getInterface());
                        eservice.setName(mReference.getName());
                        mReference.setExternalService(eservice);
                        module.getExternalServices().add(eservice);
                     }
                  } else if (eObj instanceof Reference) {
                     Reference service = (Reference) eObj;

                     ExternalService eservice = SCAFactory.eINSTANCE
                           .createExternalService();
                     eservice.setInterface(service.getInterface());
                     eservice.setName(service.getName());
                     module.getExternalServices().add(eservice);
                  }
                  break;
               case Notification.REMOVE:
                  if (eObj instanceof ModuleReference) {
                     ModuleReference mReference = (ModuleReference) eObj;
                     Module module = getModule();
                     if (mReference.getExternalService() != null) {
                        ExternalService eservice = mReference
                              .getExternalService();
                        if (module.getExternalServices().contains(eservice)) {
                           module.getExternalServices().remove(eservice);
                        } else {
                           return Status.OK_STATUS;
                        }
                     } else {
                        ExternalService eservice = module
                              .getExternalService(mReference.getName());
                        if (eservice != null)
                           module.getExternalServices().remove(eservice);
                     }
                  } else if (eObj instanceof Reference) {
                     Reference service = (Reference) eObj;

                     ExternalService eservice = module
                           .getExternalService(service.getName());
                     if (eservice != null)
                        module.getExternalServices().remove(eservice);
                  }
                  break;
            }
            break;
         case SCAPackage.COMPONENT_TYPE__PROPERTIES:
            switch (theChangeType) {
               case Notification.ADD:
                  break;
               case Notification.REMOVE:
            }
            break;
      }
      return Status.OK_STATUS;
   }

   public IStatus save() {
      if (disposed)
         return STPCorePlugin
               .createErrorStatus(
                     0,
                     "ModuleComponentTypeIntrospector is already disposed in save()!", null); //$NON-NLS-1$

      if (scribbler != null)
         try {
            scribbler.save(false, null);
         } catch (EditModelException e) {
            STPCorePlugin.logError(0, e.getMessage(), e);
            return STPCorePlugin.createErrorStatus(0, e.getMessage(), e);
         }
      return Status.OK_STATUS;
   }

   public IStatus dispose() {
      if (scribbler != null)
         try {
            scribbler.close(null);
            scribbler = null;
            module = null;
            disposed = true;
         } catch (EditModelException e) {
            STPCorePlugin.logError(0, e.getMessage(), e);
            return STPCorePlugin.createErrorStatus(0, e.getMessage(), e);
         }
      return Status.OK_STATUS;
   }

   public Object resolveImplementation() {
      if (disposed) {
         STPCorePlugin
               .logError(
                     0,
                     "ModuleComponentTypeIntrospector is already disposed in save()!", null); //$NON-NLS-1$
         return null;
      }

      return getModule();
   }

   private Module getModule() {
      if (module == null || module.getEObject().eIsProxy()) {
         try {
            Resource resource = getScribbler().getResource(
                  domain.getCompositeDescriptor());
            if (resource.getContents().size() == 1) {
               SCACoreRoot root = (SCACoreRoot) resource.getContents().get(0);
               module = root.getModule();
            }
         } catch (EditModelException e) {
            STPCorePlugin.logError(0, e.getMessage(), e);
         }
      }
      return module;
   }

   private IEditModelScribbler getScribbler() throws EditModelException {
      if (scribbler == null)
         scribbler = IEditModelFactory.eINSTANCE.createScribblerForWrite(domain
               .getProject(), domain);
      return scribbler;
   }

}
