/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.introspection;

import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.internal.STPCorePlugin;
import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.Implementation;
import org.eclipse.stp.core.sca.Property;
import org.eclipse.stp.core.sca.Reference;
import org.eclipse.stp.core.sca.SCAPackage;
import org.eclipse.stp.core.sca.Service;
import org.eclipse.stp.core.sca.UnknownImplementation;

/**
 * This class provides a default implementation of
 * {@link org.eclipse.stp.core.introspection.IComponentTypeIntrospector}.
 * 
 * <p>
 * A template pattern is used for the the {@link #init(Implementation)} ({@link #doInit()})
 * and {@link #notifyChanged(Notification)} ({@link #doNotifyChanged(Notification)})
 * methods.
 * </p>
 * 
 * 
 * <p>
 * Clients may extend this class.
 * </p>
 */
public abstract class AbstractComponentTypeIntrospector implements
      IComponentTypeIntrospector {

   /* The componentType is set from the setTarget() method. */
   private ComponentType  componentType;

   /* The implementation URI is passed in the init() method. */
   private URI            implementationURI;

   private boolean        updatingEMFModel        = false;

   private boolean        updatingUnderlyingModel = false;

   private Implementation implementation;

   /**
    * Default constructor.
    * 
    */
   public AbstractComponentTypeIntrospector() {
   }

   /**
    * The init method is called before any introspectXXX() methods are called.
    * 
    * @param theImplementation
    *           Either the URI the URI of the implementation ({@link #getImplementationURI()})
    *           or the actual Implementation model for this Introspector ({@link #getImplementation()})
    *           will be available after this method complete (but not both).
    */
   public final void init(Implementation theImplementation) {
      Assert.isNotNull(theImplementation);
      if (SCAPackage.eINSTANCE.getUnknownImplementation().equals(
            theImplementation.getEObject().eClass())) {
         UnknownImplementation unknownImplementation = (UnknownImplementation) theImplementation;
         implementationURI = URI.createURI(unknownImplementation.getUri());
      } else
         implementation = theImplementation;
      doInit();
   }

   /**
    * Notified when fields change on the ComponentType.
    */
   public final void notifyChanged(Notification notification) {
      if (updatingEMFModel)
         return;

      updatingUnderlyingModel = true;
      try {
         if (notification.getFeature() == SCAPackage.eINSTANCE
               .getComponentType_Services()) {

            switch (notification.getEventType()) {
               case Notification.ADD: {
                  Service service = (Service) notification.getNewValue();
                  onModelChange(SCAPackage.COMPONENT_TYPE__SERVICES,
                        Notification.ADD, service);
                  break;
               }
               case Notification.ADD_MANY: {
                  List services = (List) notification.getNewValue();
                  for (int i = 0; i < services.size(); i++) {
                     Service service = (Service) services.get(i);
                     onModelChange(SCAPackage.COMPONENT_TYPE__SERVICES,
                           Notification.ADD, service);
                  }
                  break;
               }
               case Notification.REMOVE: {
                  Service service = (Service) notification.getOldValue();
                  onModelChange(SCAPackage.COMPONENT_TYPE__SERVICES,
                        Notification.REMOVE, service);
                  break;
               }
               case Notification.REMOVE_MANY: {
                  List services = (List) notification.getOldValue();
                  for (int i = 0; i < services.size(); i++) {
                     Service service = (Service) services.get(i);
                     onModelChange(SCAPackage.COMPONENT_TYPE__SERVICES,
                           Notification.REMOVE, service);
                  }
                  break;
               }
            }

         } else if (notification.getFeature() == SCAPackage.eINSTANCE
               .getComponentType_References()) {

            switch (notification.getEventType()) {
               case Notification.ADD: {
                  Reference reference = (Reference) notification.getNewValue();
                  onModelChange(SCAPackage.COMPONENT_TYPE__REFERENCES,
                        Notification.ADD, reference);
                  break;
               }
               case Notification.ADD_MANY: {
                  List references = (List) notification.getNewValue();
                  for (int i = 0; i < references.size(); i++) {
                     Reference reference = (Reference) references.get(i);
                     onModelChange(SCAPackage.COMPONENT_TYPE__REFERENCES,
                           Notification.ADD, reference);
                  }
                  break;
               }
               case Notification.REMOVE: {
                  Reference reference = (Reference) notification.getOldValue();
                  onModelChange(SCAPackage.COMPONENT_TYPE__REFERENCES,
                        Notification.REMOVE, reference);
                  break;
               }
               case Notification.REMOVE_MANY: {
                  List references = (List) notification.getOldValue();
                  for (int i = 0; i < references.size(); i++) {
                     Reference reference = (Reference) references.get(i);
                     onModelChange(SCAPackage.COMPONENT_TYPE__REFERENCES,
                           Notification.REMOVE, reference);
                  }
                  break;
               }
            }
         } else if (notification.getFeature() == SCAPackage.eINSTANCE
               .getComponentType_Properties()) {

            switch (notification.getEventType()) {
               case Notification.ADD: {
                  Property property = (Property) notification.getNewValue();
                  onModelChange(SCAPackage.COMPONENT_TYPE__PROPERTIES,
                        Notification.ADD, property);
                  break;
               }
               case Notification.ADD_MANY: {
                  List properties = (List) notification.getNewValue();
                  for (int i = 0; i < properties.size(); i++) {
                     Property property = (Property) properties.get(i);
                     onModelChange(SCAPackage.COMPONENT_TYPE__PROPERTIES,
                           Notification.ADD, property);
                  }
                  break;
               }
               case Notification.REMOVE: {
                  Property property = (Property) notification.getOldValue();
                  onModelChange(SCAPackage.COMPONENT_TYPE__PROPERTIES,
                        Notification.REMOVE, property);
                  break;
               }
               case Notification.REMOVE_MANY: {
                  List properties = (List) notification.getOldValue();
                  for (int i = 0; i < properties.size(); i++) {
                     Property property = (Property) properties.get(i);
                     onModelChange(SCAPackage.COMPONENT_TYPE__PROPERTIES,
                           Notification.REMOVE, property);
                  }
                  break;
               }
            }
         }
         doNotifyChanged(notification);
      } finally {
         updatingUnderlyingModel = false;
      }
   }

   public final IStatus introspect(int theFieldType, List theCurrentValues) {

      IStatus result = Status.OK_STATUS;
      updatingEMFModel = true;
      try {
         result = doIntrospection(theFieldType, theCurrentValues);
      } catch (RuntimeException rex) {
         String message = rex.getMessage();
         if (message == null)
            message = rex.toString();
         result = STPCorePlugin.createErrorStatus(message, rex);
      } finally {
         updatingEMFModel = false;
      }
      return result;
   }

   /**
    * Return the current target. Clients may also call
    * {@link #getComponentType()} directly.
    * 
    * @return The componentType saved from setTarget().
    */
   public final Notifier getTarget() {
      if (componentType != null)
         return componentType.getEObject();
      return null;
   }

   /**
    * The componentType that this adapter is attached to is stored through a
    * call to setTarget() when the adapter is added.
    * 
    * @return The componentType saved from setTarget().
    */
   public final ComponentType getComponentType() {
      return componentType;
   }

   /**
    * The implementation URI identifies where the real model is stored. The
    * implementation URI is determined by the implementation of
    * {@link IShareableComponentTypeFactory} defined by the Introspector
    * extension, if any.
    * 
    * @return The implementation URI for the bound ComponentType.
    */
   public final URI getImplementationURI() {
      return implementationURI;
   }

   /**
    * An implementation will not be returned if the
    * {@link #init(Implementation)} was supplied an
    * {@link UnknownImplementation}. The UnknownImplementation will
    * automatically be processed and the URI will be available from
    * {@link #getImplementationURI()}.
    * 
    * @return The custom implementation supplied in
    *         {@link #init(Implementation)}
    */
   public final Implementation getImplementation() {
      return implementation;
   }

   /**
    * Called when this adapter is added to a list of adapters. This adapter may
    * only be attached to a ComponentType.
    * 
    * Attempts to attach it to any other model object will result in failure.
    */
   public final void setTarget(Notifier newTarget) {
      Assert.isTrue(newTarget == null || newTarget instanceof ComponentType);
      if (newTarget == null)
         componentType = null;
      else
         componentType = (ComponentType) newTarget;

   }

   /**
    * Not to be called by clients
    */
   public final boolean isAdapterForType(Object type) {
      return type == IComponentTypeIntrospector.ADAPTER_TYPE;
   }

   /**
    * May be overridden by subclasses to perform initialization. The
    * implementationURI will be available via {@link #getImplementationURI()}
    * when this method is called.
    * 
    */
   protected void doInit() {

   }

   /**
    * Respond to notification from the ComponentType that a field has changed.
    * 
    * <note> This method should be considered experimental. Methods may be
    * defined for onXXXAdd|Remove(List, Property|Service|Reference) in order to
    * support a bidirectional model. </note>
    * 
    * @param notification
    *           The notification from ComponentType
    */
   protected void doNotifyChanged(Notification notification) {
      // removes warning of unused parameter; essentially a no-op
      Assert.isTrue(notification.equals(notification));
   }

   protected abstract IStatus doIntrospection(int theFieldType,
         List theCurrentValues);

   protected final void doUpdateModel(int theFieldType, int theChangeType,
         List theValues) {
      if (updatingUnderlyingModel)
         return;
      updatingEMFModel = true;
      try {
         switch (theFieldType) {
            case SCAPackage.COMPONENT_TYPE__SERVICES:
               Service service = null;
               for (int i = 0; i < theValues.size(); i++) {
                  service = (Service) theValues.get(i);
                  if (theChangeType == Notification.ADD)
                     getComponentType().getServices().add(service);
                  else if (theChangeType == Notification.REMOVE)
                     getComponentType().getServices().remove(service);
               }
               break;
            case SCAPackage.COMPONENT_TYPE__PROPERTIES:
               Property property = null;
               for (int i = 0; i < theValues.size(); i++) {
                  property = (Property) theValues.get(i);
                  if (theChangeType == Notification.ADD)
                     getComponentType().getProperties().add(property);
                  else if (theChangeType == Notification.REMOVE)
                     getComponentType().getProperties().remove(property);
               }
               break;
            case SCAPackage.COMPONENT_TYPE__REFERENCES:
               Reference reference = null;
               for (int i = 0; i < theValues.size(); i++) {
                  reference = (Reference) theValues.get(i);
                  if (theChangeType == Notification.ADD)
                     getComponentType().getReferences().add(reference);
                  else if (theChangeType == Notification.REMOVE)
                     getComponentType().getReferences().remove(reference);
               }
               break;
         }
      } finally {
         updatingEMFModel = false;
      }
   }

}
