/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.wizards.annotations.webmethod;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.stp.sc.common.wizards.AbstractScWizardPage;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;



/**
 * modified version of the original <code>MethodSelectionPage</code> class. the main difference being that it 
 * only allows 1 method to be selected. This is necessary until the wizard can handled every methods 
 * separately (see https://bugs.eclipse.org/bugs/show_bug.cgi?id=162386) 
 */
public class MethodSelectionPage2 extends AbstractScWizardPage implements ICheckStateListener {
    private static final int METHOD_SELECTION_PREFERRED_HEIGHT = 200;
    private static final String SELECT_ALL = "all";
    private static final String SELECT_NONE = "none";

    private IMethod selectedMethod;
    protected CheckboxTableViewer methodViewer;
    HashMap<String, IMethod> mMapMethods;
    
    protected CreateWebMethodWizard parentWizard;
    
    private static ImageDescriptor defaultImgDesc =
        ScJaxWsResources.getImageDescriptor("wizban/newannotation_wiz.gif");

    public MethodSelectionPage2(CreateWebMethodWizard aParentWizard, String pageName, String aHelpContextId) {
        super(aParentWizard, pageName, aHelpContextId);
        parentWizard = aParentWizard;
        setImageDescriptor(defaultImgDesc);
        setTitle(ScJaxWsResources.getString("method.selection.page.title"));
        setDescription(ScJaxWsResources.getString("method.selection.page.desc"));
    }

    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        container.setLayout(new GridLayout(2, false));

        /* The list panel */
        Group listPanel = new Group(container, SWT.SHADOW_ETCHED_IN);
        listPanel.setLayout(new GridLayout(4, true));
        listPanel.setText(ScJaxWsResources.getString("method.selection.page.available.method.Label"));
        listPanel.setLayoutData(new GridData(GridData.FILL_BOTH));
        createCheckBoxList(listPanel);

        setControl(container);
    }

    public void createCheckBoxList(Composite panel) {
        methodViewer = CheckboxTableViewer.newCheckList(panel, SWT.FILL | SWT.SINGLE);

        GridData data = new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL);
        data.horizontalSpan = 4;
        data.heightHint = METHOD_SELECTION_PREFERRED_HEIGHT;
        methodViewer.getTable().setLayoutData(data);
        methodViewer.setContentProvider(new MethodListContentProvider());
        methodViewer.setLabelProvider(new MethodLabelProvider());
        methodViewer.setSorter(new ViewerSorter());
        methodViewer.setInput(new Object());
        methodViewer.addCheckStateListener(this);
        methodViewer.getTable()
                    .setBackground(parentWizard.getShell().getDisplay()
                                               .getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
        }

    protected void updateTable(String selection) {
        //TODO        LOG.debug("updating table selection=" + selection);
        if (selection.equals(SELECT_ALL)) {
            methodViewer.setAllChecked(true);
        } else if (selection.equals(SELECT_NONE)) {
            methodViewer.setAllChecked(false);
        }

        updatePageState();
    }

    @Override
    public void applyToData(Map<String, Object> data) throws Exception {
        // validation should garanty that one and only one method was selected
        
        IMember currentMethodTarget = parentWizard.getJavaMemberFromData();
        
        IMethod newMethodTarget = (IMethod)(methodViewer.getCheckedElements()[0]);
        
        if (!newMethodTarget.equals(currentMethodTarget)) {
            parentWizard.initializeWizardData(newMethodTarget);
        }
    }

    @Override
    public void initializeFromData(Map<String, Object> data) throws Exception {

        methodViewer.setAllChecked(false);
        
        // check for any selected methods
        IMember member = parentWizard.getJavaMemberFromData();

        if (member != null && member instanceof IMethod) {
            String signature = getMethodSignature((IMethod)member);
    
            if (mMapMethods.containsKey(signature)) {
                methodViewer.setChecked(member, true);
            }
        }

        updatePageState();
    }

    /**
     * @return
     * @see org.eclipse.stp.sc.common.wizards.AbstractScWizardPage#validateInputs()
     */
    protected boolean validateInputs() {
        setErrorMessage(null);
        setMessage(null);

        // must be at least one method selected
        if (methodViewer != null) {
            Object[] methods = methodViewer.getCheckedElements();

            if (methods.length < 1) {
                setErrorMessage(ScJaxWsResources.getString("method.selection.page.missing.method.error"));

                return false;
            }
        }

        return true;
    }

    public void checkStateChanged(CheckStateChangedEvent event) {
        if (selectedMethod != null) {
            methodViewer.setChecked(selectedMethod, false);
        }
        Object[] checkedElements = methodViewer.getCheckedElements();

        selectedMethod = checkedElements.length > 0 ? (IMethod)(checkedElements[0]) : null;
        updatePageState();
    }

    private String getMethodSignature(IMethod method) {
        try {
            return Signature.toString(method.getSignature(),
                                      method.getElementName(), null, false, false);
        } catch (Exception e) {
            return null;
        }
    }

    private IMethod[] getFileMethods() {
        mMapMethods = new HashMap<String, IMethod>();

        try {
            IJavaElement[] children = parentWizard.getJavaMemberFromData().getCompilationUnit()
                                       .findPrimaryType().getChildren();

            for (int i = 0; i < children.length; i++) {
                if (children[i].getElementType() == IJavaElement.METHOD) {
                    IMethod method = (IMethod)children[i];
                    int flags = method.getFlags();

                    if (Flags.isPrivate(flags) || Flags.isProtected(flags)) {
                        continue;
                    }

                    String methodSignature = getMethodSignature(method);

                    if (!mMapMethods.containsKey(methodSignature)) {
                        mMapMethods.put(methodSignature, method);
                    }
                }
            }

            return mMapMethods.values().toArray(new IMethod[0]);
        } catch (Exception e) {
            return null;
        }
    }

    public class MethodListContentProvider implements IStructuredContentProvider {
        private Object[] items = getFileMethods();

        public Object[] getElements(Object inputElement) {
            return items;
        }

        public void dispose() {
        }

        public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        }
    }

    class MethodLabelProvider extends LabelProvider {
        public String getText(Object element) {
            if ((element == null) || !(element instanceof IMethod)) {
                return null;
            }

            return getMethodSignature((IMethod)element);
        }
    }
}
