/*******************************************************************************

* Copyright (c) 2007 IONA Technologies PLC

* All rights reserved. This program and the accompanying materials

* are made available under the terms of the Eclipse Public License v1.0

* which accompanies this distribution, and is available at

* http://www.eclipse.org/legal/epl-v10.html

* 

* Contributors:

*     IONA Technologies PLC - initial API and implementation

*******************************************************************************/
package org.eclipse.stp.sc.sca.java.builders;


import java.io.File;
import java.net.URL;
import java.util.ArrayList;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.builders.ScJavaBuilder;
import org.eclipse.stp.sc.common.internal.model.RuntimeCore;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.utils.LaunchUtil;
import org.eclipse.stp.sc.sca.java.ScaJavaResources;
import org.eclipse.stp.sc.sca.java.workspace.ScaJavaNature;
import org.eclipse.stp.sc.sca.java.workspace.ScaWorkspaceManager;
import org.osoa.sca.annotations.Service;

/**
 * @author jma
 * integrate with tuscany javatowsdl tool to generate wsdl from 
 * java interface with @Service annotation
 *
 */
public class ScaJavaToWsdlBuilder extends ScJavaBuilder {

	public static final String BUILDER_ID = "org.eclipse.stp.sc.sca.java.javatowsdlbuilder";
	
	private static final LoggingProxy LOG = LoggingProxy.getlogger(ScaJavaToWsdlBuilder.class);
	
	private static final String[] DEFAULT_VM_ARGS = 
		new String[]{"-Dlog4j.configuration=log4j.properties"};
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.sc.common.builders.ScJavaBuilder#buildOneJavaFile(org.eclipse.core.resources.IFile)
	 */
	@Override
	protected void buildOneJavaFile(IFile javaFile) throws CoreException {
		try {
            if (javaFile.getFileExtension() == null
                || !javaFile.getFileExtension().equals("java")) {
                return;
            }
            generate(javaFile, getProject(), null);
        } catch (Throwable e) {
        	e.printStackTrace();
            LOG.debug("generation failure", e);

            Status status = new Status(IStatus.ERROR,
                                       ResourcesPlugin.PI_RESOURCES,
                                       IResourceStatus.BUILD_FAILED,
                                       e.getMessage(),
                                       e);
            throw new CoreException(status);
        }		

	}

	private void generate(IFile path, IProject project, Object data) throws CoreException {
		
		String wsdlFolder = ScaWorkspaceManager.getWSDLFolder(project).getRawLocation().toOSString();
		String clsName = JDTUtils.getJavaClassNameFromFile(path);
        
        
        ArrayList<String> cmds = new ArrayList<String>();
        cmds.add("-cn");
        cmds.add(clsName);
        cmds.add("-o");
        cmds.add(wsdlFolder);
        String portType = clsName;
        if (portType.indexOf(".") > 0) {
        	portType = portType.substring(portType.lastIndexOf(".") + 1);
        }
        cmds.add("-sn");
        cmds.add(ScaWorkspaceManager.getDefaultWSServiceName(portType));
        
        cmds.add("-l");
        cmds.add(ScaWorkspaceManager.getDefaultWSUrl());
        
        URL[] clsPathUrls = JDTUtils.getProjectClasspath(project);
        if (clsPathUrls.length > 0) {
        	//we just add the output classpath of current project
            cmds.add("-cp");
            String urlPath = clsPathUrls[0].getFile().substring(1);
            cmds.add(urlPath);
        }
		String cmd[] = cmds.toArray(new String[0]);
		LOG.debug("sca java2wsdl " + cmds.toString());
		
		IJavaProject javaProject = JDTUtils.findJavaProject(project.getName());
		
    	//get kit location
	    String kitPath = RuntimeCore.getInstallPath(project).makeAbsolute().toOSString();
        if (kitPath == null || kitPath.equals("")) {
        	if (System.getProperty(ScaJavaNature.SCA_KIT_LOCATION) != null) {
        		kitPath = System.getProperty(ScaJavaNature.SCA_KIT_LOCATION);
        	} else {
                MessageDialog.openWarning(null,
                    ScaJavaResources.getString("sca.warning.title"),
                    ScaJavaResources.getString("sca.missing.install.text"));
                return;
        	}	
        }
        
        String[] clsPath = null;
        if(kitPath != null && kitPath.trim().length() > 0){
        	clsPath = new String[]{
        			kitPath + File.separator + "lib" + File.separator + "tuscany-sca-manifest.jar",
        			ScaWorkspaceManager.getResourcesFolder(project).getRawLocation().toOSString()
        			};
        }
		
        //launch a java program to generate wsdl
		LaunchUtil.launchJavaProgram(javaProject, 
				"org.apache.ws.java2wsdl.Java2WSDL", 
				DEFAULT_VM_ARGS, 
				cmd, 
				clsPath);

	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.sc.common.builders.ScJavaBuilder#checkAnnotation(org.eclipse.core.resources.IFile)
	 */
	@Override
	protected boolean checkAnnotation(IFile javaFile) {
		try {
			ICompilationUnit compUnit = JDTUtils.getJavaUnitFromFile(javaFile);
			if (!compUnit.findPrimaryType().isInterface()) {
				return false;
			}
			// 2. check it is annotation
			return JDTUtils.hasAnnotationOnPrimaryType(javaFile, Service.class);
		} catch (Exception e) {
			LOG.error("error duing check ann", e);
			return false;
		}
	}
	
	protected void clean(IProgressMonitor monitor) throws CoreException {
        IFolder wsdlFolder = ScaWorkspaceManager.getWSDLFolder(getProject());
        wsdlFolder.delete(true, monitor);
        getProject().refreshLocal(IProject.DEPTH_INFINITE, null);
    }
	
    
	protected void removeResourceFile(IResource res) throws CoreException {
		if (!(res instanceof IFile)) {
			return;
		}

		IFile file = (IFile) res;

		if (!file.getFileExtension().equals("java")) {
			return;
		}

		if (checkAnnotation(file)) {
			LOG.debug("WebService java file has been removed:"
					+ file.getFullPath());

			String clsName = JDTUtils.getJavaClassNameFromFile(file);

			if (clsName.lastIndexOf(".") > 0) {
				clsName = clsName.substring(clsName.lastIndexOf("."));
			}

			String wsdlName = clsName + ".wsdl";
			LOG.debug("  need to remove:" + wsdlName);

			IFolder folder = ScaWorkspaceManager.getWSDLFolder(getProject());
			IResource wsdlResource = folder.findMember(wsdlName);

			if (wsdlResource.exists()) {
				LOG.debug("delete wsdl resource:" + wsdlResource.getFullPath());
				wsdlResource.delete(IResource.FORCE, null);
			}
		}
	}
	
}
