/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.editors;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.command.AddCommand;
import org.eclipse.emf.edit.command.RemoveCommand;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerDropAdapter;
import org.eclipse.stp.soas.deploy.core.IConfigurablePackage;
import org.eclipse.stp.soas.deploy.core.IConfigurablePackageExtension;
import org.eclipse.stp.soas.deploy.core.IHelpConstants;
import org.eclipse.stp.soas.deploy.core.IPackageConfiguration;
import org.eclipse.stp.soas.deploy.core.IPackageConfigurationManager;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.Utilities;
import org.eclipse.stp.soas.deploy.core.ui.configuration.ConfigurationDialog;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployFilePackage;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployPackage;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployServer;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.ui.TargetMapTreeViewer;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.Dialog;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.PackageSelectionDialogPage;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.ServerSelectionDialogPage;
import org.eclipse.stp.soas.internal.deploy.ui.editors.EnhancedFormEditor;
import org.eclipse.stp.soas.internal.deploy.ui.help.HelpUtilities;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.part.ResourceTransfer;
import org.eclipse.ui.views.navigator.LocalSelectionTransfer;

/**
 * @author rcernich
 * 
 * Created on Jan 21, 2004
 */
public class TargetMapSection extends DeployEditorSectionPart {

	public static final String SECTION_TITLE = DeployCorePlugin
			.getDefault().getResourceString(
					"DeployDefinitionEditor.SECTITLE.TargetMap"); //$NON-NLS-1$
	public static final String SECTION_DESC = DeployCorePlugin
			.getDefault().getResourceString(
					"DeployDefinitionEditor.SECDESC.TargetMap"); //$NON-NLS-1$

	public static final String LABEL_PACKAGES = DeployCorePlugin
			.getDefault().getResourceString("LABEL.Packages"); //$NON-NLS-1$
	public static final String LABEL_ADD_PACKAGE = DeployCorePlugin
			.getDefault().getResourceString("BUTTON.AddPackage"); //$NON-NLS-1$
	public static final String LABEL_REMOVE_PACKAGE = DeployCorePlugin
			.getDefault().getResourceString("BUTTON.RemovePackage"); //$NON-NLS-1$
	public static final String LABEL_ADD_TARGET = DeployCorePlugin
			.getDefault().getResourceString("BUTTON.AddTarget"); //$NON-NLS-1$
	public static final String LABEL_CHANGE_TARGET = DeployCorePlugin
			.getDefault().getResourceString("BUTTON.ChangeTarget"); //$NON-NLS-1$
	public static final String LABEL_REMOVE_TARGET = DeployCorePlugin
			.getDefault().getResourceString("BUTTON.RemoveTarget"); //$NON-NLS-1$
	public static final String LABEL_CONFIGURE = DeployCorePlugin
			.getDefault().getResourceString("BUTTON.ConfigurePackage"); //$NON-NLS-1$
	public static final String LABEL_VALIDATE = DeployCorePlugin
			.getDefault().getResourceString("BUTTON.ValidateConfiguration"); //$NON-NLS-1$

	private static final int TYPE_PACKAGE = 0x1 << 1;
	private static final int TYPE_TARGET = 0x1 << 2;
	private static final int TYPE_MIXED = TYPE_PACKAGE | TYPE_TARGET;

	private TargetMapTreeViewer mTargetMapViewer;
	private Button mRemoveButton;
	private Button mAddTargetButton;
	private Button mChangeTargetButton;
	private Button mRemoveTargetButton;
	private Button mConfigureButton;

	//private Button mValidateButton;

	/**
	 * @param parent
	 * @param toolkit
	 * @param style
	 */
	public TargetMapSection(Composite parent, FormToolkit toolkit, int style) {
		super(parent, toolkit, style);
		getSection().setText(SECTION_TITLE);
		getSection().setDescription(SECTION_DESC);
	}

	public void createClient(FormToolkit toolkit) {
		Composite container = toolkit.createComposite(getSection());
		GridLayout layout = EnhancedFormEditor.newSectionGridLayout();
		layout.numColumns = 2;
		container.setLayout(layout);

		Label label = toolkit.createLabel(container, LABEL_PACKAGES);
		GridData gd = new GridData();
		gd.horizontalSpan = 2;
		label.setLayoutData(gd);

		Tree tree = toolkit.createTree(container, SWT.FULL_SELECTION
				| SWT.H_SCROLL | SWT.V_SCROLL);
		mTargetMapViewer = new TargetMapTreeViewer(tree);
		gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = tree.getItemHeight() * 5;
		gd.verticalSpan = 8;
		mTargetMapViewer.getViewer().getControl().setLayoutData(gd);
		mTargetMapViewer.getViewer().addSelectionChangedListener(
				new ISelectionChangedListener() {

					public void selectionChanged(SelectionChangedEvent event) {
						handleTargetMapSelectionChanged(event);
					}
				});
		mTargetMapViewer.getViewer().addDoubleClickListener(
				new IDoubleClickListener() {

					public void doubleClick(DoubleClickEvent event) {
						handleTargetMapDoubleClick(event);
					}
				});

		Button button = toolkit.createButton(container, LABEL_ADD_PACKAGE,
				SWT.PUSH);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_BEGINNING);
		button.setLayoutData(gd);
		button.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				handleAddPackage();
			}
		});

		mRemoveButton = toolkit.createButton(container, LABEL_REMOVE_PACKAGE,
				SWT.PUSH);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_BEGINNING);
		mRemoveButton.setLayoutData(gd);
		mRemoveButton.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				handleRemovePackage();
			}
		});

		label = toolkit.createSeparator(container, SWT.HORIZONTAL);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_CENTER);
		label.setLayoutData(gd);

		mAddTargetButton = toolkit.createButton(container, LABEL_ADD_TARGET,
				SWT.PUSH);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_BEGINNING);
		mAddTargetButton.setLayoutData(gd);
		mAddTargetButton.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				handleAddTarget();
			}
		});

		mChangeTargetButton = toolkit.createButton(container,
				LABEL_CHANGE_TARGET, SWT.PUSH);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_BEGINNING);
		mChangeTargetButton.setLayoutData(gd);
		mChangeTargetButton.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				handleChangeTarget();
			}
		});

		mRemoveTargetButton = toolkit.createButton(container,
				LABEL_REMOVE_TARGET, SWT.PUSH);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_BEGINNING);
		mRemoveTargetButton.setLayoutData(gd);
		mRemoveTargetButton.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				handleRemoveTarget();
			}
		});

		label = toolkit.createSeparator(container, SWT.HORIZONTAL);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_CENTER);
		label.setLayoutData(gd);

		mConfigureButton = toolkit.createButton(container, LABEL_CONFIGURE,
				SWT.PUSH);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
				| GridData.VERTICAL_ALIGN_BEGINNING);
		mConfigureButton.setLayoutData(gd);
		mConfigureButton.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				handleConfigurePackage();
			}
		});

		//mValidateButton = toolkit.createButton(container, LABEL_VALIDATE,
		//		SWT.PUSH);
		//gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL
		//		| GridData.VERTICAL_ALIGN_BEGINNING);
		//mValidateButton.setLayoutData(gd);
		//mValidateButton.addSelectionListener(new SelectionAdapter() {
		//
		//	public void widgetSelected(SelectionEvent e) {
		//		handleValidateConfiguration();
		//	}
		//});

		getSection().setClient(container);

		updateButtons();

		toolkit.paintBordersFor(container);
	}

    protected void helpActivated(HyperlinkEvent event)
    {
       HelpUtilities.displayHelp(IHelpConstants.CONTEXT_ID_CONFIGURATION_PAGE_TARGET_MAP_SECTION);
    }
	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.internal.deploy.emf.editors.EMFSectionPart#setModelRoot(org.eclipse.emf.ecore.EObject)
	 */
	protected void setModelRoot(EObject modelRoot) {
		if (!(modelRoot instanceof Root)) {
			throw new IllegalArgumentException(
					"modelRoot must be of type " + Root.class); //$NON-NLS-1$
		}
		mTargetMapViewer.getViewer().setInput((Root) modelRoot);
	}

	private IProject getSourceProject() {
		IProject project;
		IEditorInput input = getEMFFormEditor().getEditorInput();
		if (input instanceof IFileEditorInput) {
			project = ((IFileEditorInput) input).getFile().getProject();
		}
		else {
			project = null;
		}
		return project;
	}

	private void handleAddPackage() {
		PackageSelectionDialogPage psd = new PackageSelectionDialogPage();
		psd.init(getSourceProject(), (Root) getResourceEditingContainer()
				.getModelRoot());
		Dialog dialog = new Dialog(getSection().getShell(), psd);
		if (dialog.open() == Dialog.OK) {
			IResource[] resources = psd.getSelection();
			Vector newPackages = new Vector(resources.length);
			for (int index = 0, count = resources.length; index < count; ++index) {
				newPackages.add(DeployFilePackage.eINSTANCE
						.getDeployFileFactory().createDeployPackage(
								resources[index].getFullPath()
										.toString()));
			}
			EditingDomain editingDomain = getResourceEditingContainer()
					.getEditingDomain();
			Command command = AddCommand.create(editingDomain,
					getResourceEditingContainer().getModelRoot(),
					DeployFilePackage.eINSTANCE.getRoot_Package(), newPackages);
			editingDomain.getCommandStack().execute(command);
		}
	}

	private void handleRemovePackage() {
		List packages = new Vector(((IStructuredSelection) mTargetMapViewer
				.getViewer().getSelection()).toList());

		// See if removing these packages will create any dangling servers.
		Root root = (Root) getResourceEditingContainer().getModelRoot();
		List danglingServers = new Vector(root.getServer());
		List configs = new Vector(danglingServers.size());
		// Collect all the configurations from the packages being deleted.
		for (Iterator it = packages.iterator(); it.hasNext();) {
			configs
					.addAll(((DeployPackage) it.next())
							.getTargetConfiguration());
		}
		for (Iterator it = danglingServers.iterator(); it.hasNext();) {
			if (!configs.containsAll(((DeployServer) it.next())
					.getTargetingConfiguration())) {
				// If any of the server's configuration references is not in
				// the list, it won't be left dangling, so we don't need to
				// delete the server entry.
				it.remove();
			}
		}

		// Remove any dangling servers as well.
		packages.addAll(danglingServers);

		// Remove the configuration entries just to keep the model clean.
		packages.addAll(configs);

		EditingDomain editingDomain = getEditingDomain();
		Command removeCommand = RemoveCommand.create(editingDomain, packages);
		editingDomain.getCommandStack().execute(removeCommand);
	}

	private void handleAddTarget() {
		DeployPackage pkg = (DeployPackage) ((IStructuredSelection) mTargetMapViewer
				.getViewer().getSelection()).getFirstElement();
		ServerSelectionDialogPage page = new ServerSelectionDialogPage();
		page.init(pkg, null);
		Dialog dialog = new Dialog(getSection().getShell(), page);
		if (dialog.open() == Dialog.OK) {
			EditingDomain editingDomain = getEditingDomain();
			editingDomain.getCommandStack()
					.execute(
							Utilities.createAddDeployConfigurationCommand(
									editingDomain, pkg, page
											.getSelectedDeployServer()));
		}
	}

	private void handleChangeTarget() {
		DeployConfiguration config = (DeployConfiguration) ((IStructuredSelection) mTargetMapViewer
				.getViewer().getSelection()).getFirstElement();
		DeployServer target = config.getTargetServer();
		ServerSelectionDialogPage page = new ServerSelectionDialogPage();
		page.init(config.getSourcePackage(), target);
		Dialog dialog = new Dialog(getSection().getShell(), page);
		if (dialog.open() == Dialog.OK) {
			EditingDomain editingDomain = getEditingDomain();
			DeployServer newTarget = page.getSelectedDeployServer();
			if (newTarget.equals(target)) {
				// No work to do.
				return;
			}
			List commands = new Vector(3);
			commands.add(SetCommand.create(editingDomain, config,
					DeployFilePackage.eINSTANCE
							.getDeployConfiguration_TargetServer(), newTarget));
			if (newTarget.getRoot() == null) {
				commands.add(AddCommand
						.create(editingDomain,
								(Root) getResourceEditingContainer()
										.getModelRoot(),
								DeployFilePackage.eINSTANCE.getRoot_Server(),
								newTarget));
			}
			if (target.getTargetingConfiguration().size() == 1) {
				// This target will be left dangling. We need to remove it as
				// well
				commands.add(RemoveCommand.create(editingDomain, target));
			}
			editingDomain.getCommandStack().execute(
					new CompoundCommand(commands));
		}
	}

	private void handleRemoveTarget() {
		List configs = new Vector(((IStructuredSelection) mTargetMapViewer
				.getViewer().getSelection()).toList());
		// See if removing these targets will create any dangling servers.
		Root root = (Root) getResourceEditingContainer().getModelRoot();
		List danglingServers = new Vector(root.getServer());
		for (Iterator it = danglingServers.iterator(); it.hasNext();) {
			if (!configs.containsAll(((DeployServer) it.next())
					.getTargetingConfiguration())) {
				// If any of the server's configuration references is not in
				// the list, it won't be left dangling, so we don't need to
				// delete the server entry.
				it.remove();
			}
		}

		// Remove any dangling servers as well.
		configs.addAll(danglingServers);

		EditingDomain editingDomain = getEditingDomain();
		editingDomain.getCommandStack().execute(RemoveCommand.create(editingDomain,configs));
	}

	private void handleConfigurePackage() {
		DeployConfiguration config = (DeployConfiguration) ((IStructuredSelection) mTargetMapViewer
				.getViewer().getSelection()).getFirstElement();
		IConfigurablePackage pkg;
		IPackageConfigurationManager packageConfigManager;
		IPackageConfiguration packageConfiguration = null;

		try {
			pkg = Utilities.adaptToIConfigurablePackage(config);
			if(pkg == null){
				return;
			}
			packageConfigManager = ((IConfigurablePackageExtension) pkg
					.getExtension()).getPackageConfigurationManager();

			byte[] configBytes = config.getConfigOverride();
			if (configBytes != null) {
				try {
					packageConfiguration = packageConfigManager
							.createPackageConfiguration(pkg,
									new ByteArrayInputStream(configBytes));
				}
				catch (IOException e) {
					// RJC Auto-generated catch block
					e.printStackTrace();
				}
			}

			if (packageConfiguration == null) {
				packageConfiguration = packageConfigManager
						.createPackageConfiguration(pkg);
			}

			ConfigurationDialog dialog = new ConfigurationDialog(getSection()
					.getShell(), packageConfiguration);
			dialog.create();
			dialog.getShell()
					.setText(
							MessageFormat.format(DeployCorePlugin
									.getDefault().getResourceString(
											"ConfigurationDialog.TITLE"), //$NON-NLS-1$
									new Object[] {
											new Path(config.getSourcePackage()
													.getPackageFile())
													.lastSegment(),
											config.getTargetServer()
													.getProfileName()}));
			if (dialog.open() == ConfigurationDialog.OK) {
				ByteArrayOutputStream baos = new ByteArrayOutputStream();
				try {
					packageConfigManager.serializePackageConfiguration(
							packageConfiguration, baos);

					EditingDomain editingDomain = getEditingDomain();
					editingDomain
							.getCommandStack()
							.execute(
									SetCommand
											.create(
													editingDomain,
													config,
													DeployFilePackage.eINSTANCE
															.getDeployConfiguration_ConfigOverride(),
													baos.toByteArray()));
				}
				catch (IOException e) {
					// RJC Auto-generated catch block
					e.printStackTrace();
				}
				finally {
					try {
						baos.close();
					}
					catch (IOException e) {
					}
				}
			}
		}
		finally {
			if (packageConfiguration != null) {
				packageConfiguration.dispose();
			}
		}
	}

	private void handleTargetMapSelectionChanged(SelectionChangedEvent event) {
		updateButtons();
		fireSelectionChanged(event.getSelection());
	}

	private void handleTargetMapDoubleClick(DoubleClickEvent event) {
		Object element = ((IStructuredSelection) event.getSelection())
				.getFirstElement();
		if (element instanceof DeployPackage) {
			handleAddTarget();
		}
		else { // must be a config element
			handleConfigurePackage();
		}
	}

	private void updateButtons() {
		IStructuredSelection selection = (IStructuredSelection) mTargetMapViewer
				.getViewer().getSelection();
		if (selection.isEmpty()) {
			mRemoveButton.setEnabled(false);
			mAddTargetButton.setEnabled(false);
			mChangeTargetButton.setEnabled(false);
			mRemoveTargetButton.setEnabled(false);
			mConfigureButton.setEnabled(false);
			//mValidateButton.setEnabled(false);
		}
		else if (selection.size() > 1) {
			// Figure out what objects are in the selection
			int selectionType = 0;
			for (Iterator it = selection.iterator(); it.hasNext();) {
				Object element = it.next();
				if (element instanceof DeployPackage) {
					selectionType |= TYPE_PACKAGE;
				}
				else { // Must be a DeployConfig
					selectionType |= TYPE_TARGET;
				}
			}
			mRemoveButton.setEnabled(selectionType == TYPE_PACKAGE);
			mAddTargetButton.setEnabled(false);
			mChangeTargetButton.setEnabled(false);
			mRemoveTargetButton.setEnabled(selectionType == TYPE_TARGET);
			mConfigureButton.setEnabled(false);
			//mValidateButton.setEnabled(true);
		}
		else {
			Object element = selection.getFirstElement();
			mRemoveButton.setEnabled(element instanceof DeployPackage);
			mAddTargetButton.setEnabled(element instanceof DeployPackage);
			mChangeTargetButton
					.setEnabled(element instanceof DeployConfiguration);
			mRemoveTargetButton
					.setEnabled(element instanceof DeployConfiguration);
			mConfigureButton
					.setEnabled(element instanceof DeployConfiguration
							&& org.eclipse.stp.soas.deploy.core.Utilities
									.adaptToIConfigurablePackage((EObject) element) != null);
			//mValidateButton.setEnabled(true);
		}
	}

	private EditingDomain getEditingDomain() {
		return getResourceEditingContainer().getEditingDomain();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.forms.IFormPart#setFormInput(java.lang.Object)
	 */
	public boolean setFormInput(Object input) {
		boolean retVal;
		StructuredSelection selection = null;
		if (input instanceof DeployPackage
				|| input instanceof DeployConfiguration) {
			selection = new StructuredSelection(input);
		}
		else if (input instanceof Collection) {
			selection = new StructuredSelection(((Collection) input).toArray());
		}
		if (selection != null) {
			mTargetMapViewer.getViewer().setSelection(selection);
			mTargetMapViewer.getViewer().getTree().setFocus();
			retVal = true;
		}
		else {
			retVal = false;
		}
		return retVal;
	}

	private class TargetMapDropListener extends ViewerDropAdapter {

		public TargetMapDropListener(Viewer viewer) {
			super(viewer);
		}

		public void dragOver(DropTargetEvent event) {
			if ((event.operations & DND.DROP_LINK) == DND.DROP_LINK) {
				event.detail = DND.DROP_LINK;
			}
			super.dragOver(event);
		}

		public void drop(DropTargetEvent event) {
			super.drop(event);
			event.detail = DND.DROP_LINK;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ViewerDropAdapter#performDrop(java.lang.Object)
		 */
		public boolean performDrop(Object data) {
			return true;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ViewerDropAdapter#validateDrop(java.lang.Object,
		 *      int, org.eclipse.swt.dnd.TransferData)
		 */
		public boolean validateDrop(Object target, int operation,
				TransferData transferType) {
			if (transferType != null
					&& (LocalSelectionTransfer.getInstance().isSupportedType(
							transferType) || ResourceTransfer.getInstance()
							.isSupportedType(transferType))) {
				return validateSource();
			}
			return false;
		}

		private boolean validateSource() {
			// Check to see if source is a package or server.
			return false;
		}

	}

}