/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.deploy.core.operations;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.IPackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.Utilities;
import org.eclipse.stp.soas.deploy.core.utils.DeploymentUtil;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployPackage;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.DeployStatusDialogPage;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.IProgressConstants;
import org.eclipse.wst.server.core.IServer;

/**
 * @author rcernich
 * 
 * Created on Feb 24, 2004
 */
public class ExecuteUndeploymentJob extends Job {

	private Root mRoot;
	private Map mDeployConfigToServerMap = new HashMap();
	private Map mDeployStatusMap = new HashMap();
	private boolean mHasErrorStatus = false;
	private CreateDeployPackagesJob mCreatePackages;
	private Shell mShell;

	/**
	 * 
	 */
	public ExecuteUndeploymentJob(Root root, Shell shell) {
		super(DeployCorePlugin.getDefault().getResourceString(
				"ExecuteUndeploymentJob.task.name"));
		mRoot = root;
		mShell = shell;
		mCreatePackages = new CreateDeployPackagesJob(mRoot);
		setRule(ResourcesPlugin.getWorkspace().getRoot());
		setUser(true);
		setSystem(false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IStatus run(IProgressMonitor monitor) {
		monitor.beginTask(DeployCorePlugin.getDefault().getResourceString(
				"ExecuteUndeploymentJob.task.name"), 3000); //$NON-NLS-1$

		mDeployStatusMap.clear();
		mHasErrorStatus = false;

		IStatus status = Status.OK_STATUS;

		try {
			Map packageDescriptors = createPackages(monitor);

			preUndeployCheck(packageDescriptors, monitor);

			undeployPackages(packageDescriptors, monitor);

			if (mHasErrorStatus) {
				status = new Status(Status.ERROR, DeployCorePlugin
						.getDefault().getBundle().getSymbolicName(), -1,
						DeployCorePlugin.getDefault().getResourceString(
								"ExecuteUndeploymentJob.error.job"), null); //$NON-NLS-1$
			}
		}
		catch (CoreException e) {
			status = e.getStatus();
		}
		catch (InterruptedException e) {
			status = Status.CANCEL_STATUS;
		}
		finally {
			
			mDeployConfigToServerMap.clear();
		}

		logStatus();

		monitor.done();

		setProperty(IProgressConstants.KEEP_PROPERTY, Boolean.TRUE);
		setProperty(IProgressConstants.ACTION_PROPERTY, new Action(
				DeployCorePlugin.getDefault().getResourceString(
						"ExecuteUndeploymentJob.action.name")) { //$NON-NLS-1$

			public void run() {
				displayStatus();
			}
		});

		return status;
	}

	public Map getDeployStatusMap() {
		return mDeployStatusMap;
	}

	private void checkCancelled(IProgressMonitor monitor)
			throws InterruptedException {
		if (monitor.isCanceled()) {
			throw new InterruptedException();
		}
	}

	private Map createPackages(IProgressMonitor monitor) throws CoreException,
			InterruptedException {
		SubProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1000);
		IStatus status = mCreatePackages.run(subMonitor);
		if (status.getSeverity() != IStatus.OK) {
			throw new CoreException(status);
		}

		checkCancelled(monitor);

		return mCreatePackages.getPackageDescriptors();
	}

	private void preUndeployCheck(Map packageDescriptors,
			final IProgressMonitor monitor) throws CoreException,
			InterruptedException {
		int deployTaskLength = getDeployTaskLength();
		Map deployCheckStatus = new HashMap();
		int severity = 0;

		for (Iterator it = packageDescriptors.entrySet().iterator(); it
				.hasNext();) {
			Map.Entry entry = (Map.Entry) it.next();
			DeployConfiguration config = (DeployConfiguration) entry.getKey();
			List<IPackageOutputDescriptor> descriptors = (List<IPackageOutputDescriptor>) entry
					.getValue();
			List<IPackageOutputDescriptor> exists = new ArrayList<IPackageOutputDescriptor>();
			for (IPackageOutputDescriptor descriptor : descriptors) {
				monitor
						.subTask(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"ExecuteUndeploymentJob.subtask.preDeployCheck", //$NON-NLS-1$
										new Object[] {
												config.getSourcePackage()
														.getPackageFile(),
												config.getTargetServer()
														.getProfileName() }));

				if (descriptor == null) {
					String exceptionMessage = DeployCorePlugin.getDefault()
							.getResourceString(
									"ExecuteUndeploymentJob.error.deployError", //$NON-NLS-1$
									new Object[] {
											config.getSourcePackage()
													.getPackageFile(),
											config.getTargetServer()
													.getProfileName() });
					IStatus status = new Status(
							Status.ERROR,
							DeployCorePlugin.getDefault().getBundle()
									.getSymbolicName(),
							-1,
							DeployCorePlugin.getDefault().getResourceString(
									"ExecuteUndeploymentJob.error.log.message"), //$NON-NLS-1$
							new Exception(exceptionMessage));
					throw new CoreException(new MultiStatus(DeployCorePlugin
							.getDefault().getBundle().getSymbolicName(), -1,
							new IStatus[] { status }, DeployCorePlugin
									.getDefault().getResourceString(
											"UndeployStatusDialog.TITLE"), //$NON-NLS-1$
							null));
				} else {
					IServer server = Utilities.adaptToIServer(config);

					if (DeploymentUtil.containsWTPModule(descriptor, config
							.getTargetServer().getServerId())) {
						mDeployConfigToServerMap.put(config, server);
					} else {
						exists.add(descriptor);
						Status status = new Status(IStatus.ERROR,
								DeployCorePlugin.getDefault().getBundle()
										.getSymbolicName(), -1,
								"The application "
										+ descriptor.getLogicalName()
										+ " does not exist on the server "
										+ server.getName(),

								null);
						mDeployStatusMap.put(config, new Status[] { status });

					}

				}
			}
			descriptors.removeAll(exists);
			monitor.worked(deployTaskLength);
 
			checkCancelled(monitor);
		}
		checkCancelled(monitor);
	}

	private void undeployPackages(Map packageDescriptors, IProgressMonitor monitor)
			throws CoreException, InterruptedException {
		int deployTaskLength = getDeployTaskLength();

		for (Iterator it = packageDescriptors.entrySet().iterator(); it
				.hasNext();) {
			Map.Entry entry = (Map.Entry) it.next();
			DeployConfiguration config = (DeployConfiguration) entry.getKey();
			List<IPackageOutputDescriptor> descriptors = (List<IPackageOutputDescriptor>) entry
					.getValue();
			for (IPackageOutputDescriptor descriptor : descriptors) {
				monitor.subTask(DeployCorePlugin.getDefault()
						.getResourceString(
								"ExecuteUndeploymentJob.subtask.deploy", //$NON-NLS-1$
								new Object[] {
										config.getSourcePackage()
												.getPackageFile(),
										config.getTargetServer()
												.getProfileName() }));

				if (descriptor == null
						|| !new File(descriptor.getFileName()).exists()) {
					mDeployStatusMap
							.put(
									config,
									new IStatus[] { new Status(
											IStatus.ERROR,
											DeployCorePlugin.getDefault()
													.getBundle()
													.getSymbolicName(),
											-1,
											DeployCorePlugin
													.getDefault()
													.getResourceString(
															"Undeploy.ERROR.PackageDoesNotExist", //$NON-NLS-1$
															new Object[] {
																	config
																			.getSourcePackage()
																			.getPackageFile(),
																	config
																			.getTargetServer()
																			.getProfileName() }),
											null) });
					mHasErrorStatus = true;
				} else {

					IServer server = (IServer) mDeployConfigToServerMap
							.get(config);
					if (server == null)
						continue;

					try {
						DeploymentUtil.undeployWTPModule(descriptor, server
								.getId());
						mDeployStatusMap
								.put(
										config,
										new IStatus[] { new Status(
												IStatus.INFO,
												DeployCorePlugin.getDefault()
														.getBundle()
														.getSymbolicName(),
												-1,
												DeployCorePlugin
														.getDefault()
														.getResourceString(
																"Undeploy.SUCCESS.Generic", //$NON-NLS-1$
																new Object[] {
																		config
																				.getSourcePackage()
																				.getPackageFile(),
																		config
																				.getTargetServer()
																				.getProfileName() }),
												null) });
					} catch (Exception e) {
						mDeployStatusMap
								.put(
										config,
										new IStatus[] { new Status(
												IStatus.ERROR,
												DeployCorePlugin.getDefault()
														.getBundle()
														.getSymbolicName(),
												-1,
												DeployCorePlugin
														.getDefault()
														.getResourceString(
																"Undeploy.ERROR.Generic", //$NON-NLS-1$
																new Object[] {
																		config
																				.getSourcePackage()
																				.getPackageFile(),
																		server
																				.getName(),
																		e
																				.getMessage() }),
												e) });
						mHasErrorStatus = true;
					}

				}
			}
			monitor.worked(deployTaskLength);

			checkCancelled(monitor);
		}

		Display display = PlatformUI.getWorkbench().getDisplay();
		final DeployStatusDialogPage page = new DeployStatusDialogPage(
				mDeployStatusMap, DeployCorePlugin.getDefault()
						.getResourceString("UndeployStatusDialog.TITLE"));

		display.syncExec(new Runnable() {

			public void run() {
				boolean disposeShell = false;
				Display display = Display.getCurrent();
				Shell shell = display.getActiveShell();
				if (shell == null) {
					Shell shells[] = display.getShells();
					if (shells.length > 0) {
						shell = shells[shells.length - 1];
					} else {
						disposeShell = true;
						shell = new Shell();
					}
				}
				try {
					Dialog dialog = new Dialog(shell, page, SWT.RESIZE);
					dialog.open();
				} finally {
					if (disposeShell) {
						shell.dispose();
					}
				}
			}
		});

		checkCancelled(monitor);
	}

	private int getDeployTaskLength() {
		int retVal = 0;
		for (Iterator it = mRoot.getPackage().iterator(); it.hasNext();) {
			retVal += ((DeployPackage) it.next()).getTargetConfiguration()
					.size();
		}
		return retVal / 1000 + 1;
	}

	private int getRefreshTaskLength() {
		return mRoot.getServer().size() / 1000 + 1;
	}

	private void logStatus() {
		Collection statuses = new ArrayList();
		for (Iterator it = mDeployStatusMap.values().iterator(); it.hasNext();) {
			statuses.addAll(Arrays.asList((IStatus[]) it.next()));
		}
		DeployCorePlugin.getDefault().log(
				new MultiStatus(DeployCorePlugin.getDefault().getBundle()
						.getSymbolicName(), -1, (IStatus[]) statuses
						.toArray(new IStatus[statuses.size()]),
						DeployCorePlugin.getDefault().getResourceString(
								"UndeployStatusDialog.TITLE"), //$NON-NLS-1$
						null));
	}

	private void displayStatus() {
		Display display = PlatformUI.getWorkbench().getDisplay();
		final DeployStatusDialogPage page = new DeployStatusDialogPage(
				mDeployStatusMap, DeployCorePlugin.getDefault().getResourceString(
                "UndeployStatusDialog.TITLE"));

		display.syncExec(new Runnable() {

			public void run() {
				boolean disposeShell = false;
				Display display = Display.getCurrent();
				Shell shell = mShell;
				if (shell == null) {
					Shell shells[] = display.getShells();
					if (shells.length > 0) {
						shell = shells[shells.length - 1];
					}
					else {
						disposeShell = true;
						shell = new Shell();
					}
				}
				try {
					Dialog dialog = new Dialog(shell, page, SWT.RESIZE);
					dialog.open();
				}
				finally {
					if (disposeShell) {
						shell.dispose();
					}
				}
			}
		});
	}

}
