/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.utils.ext;

import java.util.ArrayList;
import java.util.List;

import java.lang.annotation.Annotation;

import javax.xml.ws.soap.SOAPBinding;

import javax.xml.ws.RequestWrapper;
import javax.xml.ws.ResponseWrapper;
import javax.xml.ws.ServiceMode;
import javax.xml.ws.WebFault;
import javax.xml.ws.WebServiceClient;
import javax.xml.ws.WebEndpoint;
import javax.xml.ws.WebServiceProvider;
import javax.xml.ws.BindingType;
import javax.xml.ws.WebServiceRef;
import javax.xml.ws.WebServiceRefs;

import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MemberValuePair;
import org.eclipse.jdt.core.dom.ArrayInitializer;
import org.eclipse.jdt.core.dom.NormalAnnotation;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;

import org.eclipse.stp.sc.common.annotations.ext.IAnnotationInitializer;
import org.eclipse.stp.sc.common.utils.JDTUtils;


public class JAXWSAnnotationInitializer implements IAnnotationInitializer {
	
	@SuppressWarnings("unchecked")
    public List<MemberValuePair> getDefaultAttributes(Class<? extends Annotation> annotationClass,
                                                      CompilationUnit astRoot,
                                                      IMember jdtMember,
                                                      SingleVariableDeclaration jdtMemberParam) {
        List<MemberValuePair> annotValues = null;
        
        if  (annotationClass.equals(RequestWrapper.class)) {
            /* 
             * @Target(value=METHOD)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface RequestWrapper
             * String className
             * string localName
             * String targetNamespace
             * 
             * sample output for a method: <String sayHi()> of the class <com.iona.helloworld.HelloWorld>
             *  
             * @RequestWrapper(className="com.iona.helloworld.SayHiRequest",
             *                 localName="SayHiRequest",
             *                 targetNamespace="http://helloworld.iona.com/")
             */
 
            annotValues = new ArrayList<MemberValuePair>();
            
            IMethod method = (IMethod)jdtMember;
            IType type = method.getDeclaringType();

            String packageName = type.getPackageFragment().getElementName();
            if(packageName.length() == 0)
            {
            	packageName = "default_package";
            }

            String className = method.getElementName().substring(0, 1).toUpperCase() +
        		method.getElementName().substring(1);
            

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "className",
                                                        packageName + "." + className));
        
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "localName",
                                                        method.getElementName()));
        
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "targetNamespace",
                                                        JDTUtils.getNamespace(packageName)));

        } else if  (annotationClass.equals(ResponseWrapper.class)) {
            /* 
             * @Target(value=METHOD)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface ResponseWrapper
             * String className
             * string localName
             * String targetNamespace
             * 
             * sample output for a method: <String sayHi()> of the class <com.iona.helloworld.HelloWorld>
             *  
             * @ResponseWrapper(className="com.iona.helloworld.SayHiResponse",
             *                  localName="SayHiResponse",
             *                  targetNamespace="http://helloworld.iona.com/")
             */
            
            annotValues = new ArrayList<MemberValuePair>();
            
            IMethod method = (IMethod)jdtMember;
            IType type = method.getDeclaringType();

            String packageName = type.getPackageFragment().getElementName();
            if(packageName.length() == 0)
            {
            	packageName = "default_package";
            }

            String className = method.getElementName().substring(0, 1).toUpperCase() +
            	method.getElementName().substring(1) + "Response";

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "className",
                                                        packageName + "." + className));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "localName",
                                                        method.getElementName()
                                                        + "Response"));
        
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                    "targetNamespace",
                    JDTUtils.getNamespace(packageName)));

        } else if (annotationClass.equals(ServiceMode.class)) {
            /* 
             * @Target(value=TYPE)
             * @Retention(value=RUNTIME)
             * @Inherited
             * @Documented
             * public @interface ServiceMode
             * Service.Mode value
             * 
             * sample output for a class: <com.iona.helloworld.HelloWorld>
             *  
             * ServiceMode(value=javax.xml.ws.Service.Mode.PAYLOAD)
             */

        	annotValues = new ArrayList<MemberValuePair>();
        	
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "value",
                                                        javax.xml.ws.Service.Mode.PAYLOAD));

        } else if (annotationClass.equals(WebFault.class)) {
            /* 
             * @Target(value=TYPE)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface WebFault
             * String faultBean
             * String name
             * String targetNamespace
             * 
             * sample output for a class: <com.iona.helloworld.HelloWorld>
             *  
             * @WebFault(faultBean="com.iona.helloworld.HelloWorldFaultBean",
             *           name="HelloWorldFault",
             *           targetNamespace = "http://helloworld.iona.com/")
             */

        	annotValues = new ArrayList<MemberValuePair>();
        	
            IType type = (IType)jdtMember;

            String typeName = type.getElementName();
            String packageName = type.getPackageFragment().getElementName();

            String faultPostfix = "Fault";
            String faultBeanPostfix = "FaultBean";
            String[] faultPostfixes = { "Error", "Fault", "Exception" }; 

            String shortName = type.getElementName();

            for (String postfix : faultPostfixes) {
            	if (typeName.endsWith(postfix)) {
            		shortName = typeName.substring(0, typeName.length() - postfix.length());
            		break;
            	}
            }
            
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "faultBean",
                                                        packageName + "."  + shortName.concat(faultBeanPostfix)));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "name",
                                                        typeName.equals(shortName) ? shortName.concat(faultPostfix) : typeName));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "targetNamespace",
                                                        JDTUtils.getNamespace(packageName)));

        } else if (annotationClass.equals(WebServiceClient.class)) {
            /* 
             * @Target(value=TYPE)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface WebServiceClient
             * String name
             * string targetNamespace
             * String wsdlLocation
             * 
             * sample output for a class: <com.iona.helloworld.HelloWorld>
             *  
             * @WebServiceClient(name="HelloWorld",
             *                   targetNamespace="http://helloworld.iona.com/",
             *                   wsdlLocation="http://helloworld.iona.com/helloworld.wsdl")
             */

        	annotValues = new ArrayList<MemberValuePair>();
        	
            IType type = (IType)jdtMember;

            String packageName = type.getPackageFragment().getElementName();
            String targetNamespace = JDTUtils.getNamespace(packageName);

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "name",
                                                        packageName + "." + type.getElementName()));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "targetNamespace",
                                                        JDTUtils.getNamespace(packageName)));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "wsdlLocation",
                                                        targetNamespace + type.getElementName().toLowerCase() + "." + "wsdl"));

            
        } else if (annotationClass.equals(WebEndpoint.class)) {
            /* 
             * @Target(value=METHOD)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface WebEndpoint
             * String name
             * 
             * sample output for a method: <String sayHi()> of the class <com.iona.helloworld.HelloWorld>
             *  
             * @WebEndpoint(name="HelloWorldHTTPPort")
             */

        	annotValues = new ArrayList<MemberValuePair>();
        	
            IMethod method = (IMethod)jdtMember;
            IType type = method.getDeclaringType();
            String typeName = type.getElementName();

            String typePostfix = "Service";
            String portPostfix = "HTTPPort";
            String endpoint = typeName.endsWith(typePostfix) ? typeName.substring(0, typeName.length() - typePostfix.length()) : typeName;
            endpoint = endpoint.concat(portPostfix);

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "name",
                                                        endpoint));
        	
        } else if (annotationClass.equals(WebServiceProvider.class)) {
            /* 
             * @Target(value=TYPE)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface WebServiceProvider
             * String portName
             * String serviceName
             * String targetNamespace
             * String wsdlLocation
             * 
             * sample output for a class: <com.iona.helloworld.HelloWorld>
             *  
             * @WebServiceProvider(portName="HelloWorldPort",
             *                     serviceName="HelloWorldService",
             *                     targetNamespace="http://helloworld.iona.com/",
             *                     wsdlLocation="http://helloworld.iona.com/helloworld.wsdl")
             */

        	annotValues = new ArrayList<MemberValuePair>();
        	
            IType type = (IType)jdtMember;
            String typeName = type.getElementName();

            String packageName = type.getPackageFragment().getElementName();
            String targetNamespace = JDTUtils.getNamespace(packageName);

            String portPostfix = "Port";
            String servicePostfix = "Service";
            String shortName = typeName.endsWith(servicePostfix) ? typeName.substring(0, typeName.length() - servicePostfix.length()) : typeName;

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "portName",
                                                        shortName + portPostfix));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "serviceName",
                                                        shortName + servicePostfix));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "targetNamespace",
                                                        JDTUtils.getNamespace(packageName)));

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "wsdlLocation",
                                                        targetNamespace + type.getElementName().toLowerCase() + "." + "wsdl"));
        	
        } else if (annotationClass.equals(BindingType.class)) {
            /* 
             * @Target(value=TYPE)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface BindingType
             * String value
             * 
             * sample output for a class: <com.iona.helloworld.HelloWorld>
             *  
             * @BindingType(value="http://schemas.xmlsoap.org/wsdl/soap/http")
             */

        	annotValues = new ArrayList<MemberValuePair>();

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "value",
                                                        SOAPBinding.SOAP11HTTP_BINDING));
        
        } else if (annotationClass.equals(WebServiceRef.class)) {
            /* 
             * @Target(value={TYPE,METHOD,FIELD})
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface WebServiceRef
             * String mappedName
             * String name
             * Class type
             * Class value
             * String wsdlLocation
             * 
             * sample output for a class: <com.iona.helloworld.HelloWorld>
             *  
             * @WebServiceRef(mappedName="",
             *                name="<specify name here>",
             *                type=java.lang.Object.class,
             *                value=java.lang.Object.class,
             *                wsdlLocation="http://helloworld.iona.com/helloworld.wsdl")
             */

        	annotValues = new ArrayList<MemberValuePair>();
        	
        	/*
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "mappedName",
                                                        ""));
            */

        	if (!(jdtMember instanceof IField)) {
        		annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                            "name",
                                                            "<specify name here>"));
        	}

            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "type",
                                                        java.lang.Object.class));
            
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "value",
                                                        java.lang.Object.class));

            /*
            annotValues.add(JDTUtils.newMemberValuePair(astRoot,
                                                        "wsdlLocation",
                                                        ""));
            */

        } else if (annotationClass.equals(WebServiceRefs.class)) {
            /* 
             * @Target(value=TYPE)
             * @Retention(value=RUNTIME)
             * @Documented
             * public @interface WebServiceRefs
             * WebServiceRef[] value
             * 
             * sample output for a class: <com.iona.helloworld.HelloWorld>
             *  
             * @WebServiceRefs({@WebServiceRef(mappedName="",
             *                                 name="<specify name here>",
             *                                 type=java.lang.Object.class,
             *                                 value=java.lang.Object.class,
             *                                 wsdlLocation=""),
             *                  @WebServiceRef(mappedName="",
             *                                 name="<specify name here>",
             *                                 type=java.lang.Object.class,
             *                                 value=java.lang.Object.class,
             *                                 wsdlLocation="")})
             */

        	annotValues = new ArrayList<MemberValuePair>();

        	@WebServiceRef(mappedName="",
                           name="<specify name here>",
                           type=java.lang.Object.class,
                           value=java.lang.Object.class,
                           wsdlLocation="")
            class DummyClass {};
            
            Annotation[] annos = DummyClass.class.getAnnotations();
            WebServiceRef[] refs = {(WebServiceRef)annos[0]}; 
        	
    		ArrayInitializer arrayInit = astRoot.getAST().newArrayInitializer();
    		List<NormalAnnotation> expList = arrayInit.expressions();
    		
    		for (WebServiceRef ref : refs) {
    			NormalAnnotation anno = astRoot.getAST().newNormalAnnotation();
    			anno.setTypeName(astRoot.getAST().newName("WebServiceRef"));
    			List<MemberValuePair> pairList = anno.values();
    			if (ref.mappedName() != null && ref.mappedName().length() > 0) {
        			pairList.add(JDTUtils.newMemberValuePair(astRoot,
                                                             "mappedName",
                                                             ref.mappedName()));
    			}
    			if (ref.name() != null && ref.name().length() > 0) {
    				pairList.add(JDTUtils.newMemberValuePair(astRoot,
                                                             "name",
                                                             ref.name()));
    			}
    			pairList.add(JDTUtils.newMemberValuePair(astRoot,
                                                         "type",
                                                         ref.type()));
    			pairList.add(JDTUtils.newMemberValuePair(astRoot,
                                                         "value",
                                                         ref.value()));
    			if (ref.wsdlLocation() != null && ref.wsdlLocation().length() > 0) {
    				pairList.add(JDTUtils.newMemberValuePair(astRoot,
                                                             "wsdlLocation",
                                                             ref.wsdlLocation()));
    			}
    			expList.add(anno);
    		}

    		MemberValuePair pair = astRoot.getAST().newMemberValuePair();
            pair.setName(astRoot.getAST().newSimpleName("value"));
            pair.setValue(arrayInit);
            annotValues.add(pair);
        }

        return annotValues;
    }
}
