/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.ui.internal.launchconfig;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.sql.Date;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.stp.b2j.core.publicapi.B2jPlatform;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.engine.BPELEngineListener;
import org.eclipse.stp.b2j.core.publicapi.engine.WorkbenchBPELEngine;
import org.eclipse.stp.b2j.core.publicapi.program.BPELProgram;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;
import org.eclipse.stp.b2j.ui.UiPlugin;
import org.eclipse.stp.b2j.ui.internal.extensionpoints.BPELProviderLoader;
import org.eclipse.stp.b2j.ui.internal.misc.HexData;
import org.eclipse.stp.b2j.ui.publicapi.extensionpoints.bpelprovider.BPELProvider;
import org.eclipse.stp.b2j.ui.publicapi.extensionpoints.bpelprovider.BPELProviderCategory;
import org.eclipse.stp.b2j.ui.publicapi.extensionpoints.bpelprovider.BPELProviderNode;
import org.eclipse.stp.b2j.ui.publicapi.program.WorkbenchBPELProgram;

public class LaunchConfigDelegate implements ILaunchConfigurationDelegate {
	
	private void printFatalProblem(BPELEngineListener console, String description, Exception e) {
		console.printDebug("\n\nERROR: "+description);
		console.printInfo("\nProblem Description:");
		if (e.getMessage() != null) {
			console.printDebug(e.getMessage());
		} else {
			console.printDebug("");
		}
		console.printInfo("\nProblem Details:");
		console.printDebug(getStacktrace(e));
	}
	
	private static String getStacktrace(Throwable t) {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		PrintStream print = new PrintStream(bout);
		t.printStackTrace(print);
		return new String(bout.toByteArray()).replace('\t',' ');
	}
	
	private class AbortTracker extends Thread {
		boolean finished = false;
		
		WorkbenchBPELEngine engine;
		IProgressMonitor monitor;
		
		public AbortTracker(WorkbenchBPELEngine engine, IProgressMonitor monitor) {
			this.engine = engine;
			this.monitor = monitor;
		}
		
		public void finished() {
			finished = true;
		}
		
		public void run() {
			while (!finished) {
				try {
					if (monitor.isCanceled()) {
						engine.terminate();
					}
				} catch (Exception e) {
				}
				try {
					Thread.sleep(1000);
				} catch (Exception e) {
				}
			}
		}
	}
	
	/**
	 * Launch this launch configuration
	 */
	public void launch(ILaunchConfiguration conf, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException {
		//get the files from the launch config references

		B2jPlatform.startPreRunServices();

		BPELProviderNode provider_node;
		BPELProvider provider;
		String bpel_name;
		
		try {
			String bpel_provider = conf.getAttribute("selected_provider_id","");
			BPELProviderCategory category = BPELProviderLoader.getProviders();
			
			provider_node = category.getProvider(bpel_provider);
			provider = provider_node.getProvider();
			
			LaunchConfigPropertySource propsource = new LaunchConfigPropertySource();
			propsource.initializeFrom(conf);
			
			provider.setPropertySource(propsource);
			bpel_name = provider.getBpelName();
		
		} catch (Exception e) {
			UiPlugin.DBG.logVisibleError(e,"Failed to load and set up BPEL provider",true);
			return;
		}
		
		String engine_impl = conf.getAttribute("engine_impl","mini");
		List ser_hosts_list = conf.getAttribute("hosts_list",new ArrayList());
		
		Map daemon_transports = conf.getAttribute("daemon_transports",new HashMap());
		Map daemon_passwords = conf.getAttribute("daemon_passwords",new HashMap());

		//convert the daemon entries into SessionAddress objects which
		//are then passed in to allow daemon passwords to be used etc
		
		//set up the console listener
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy_MM_dd HH:mm:ss.SSS");
		String engine_name = UiPlugin.getString("B2J_RUN")+" ("+bpel_name+") ("+sdf.format(new Date(System.currentTimeMillis()))+")";
		ConsoleEngineListener console = new ConsoleEngineListener(engine_name);
		
		monitor.setTaskName(engine_name);
		monitor.subTask(engine_name);
		
		console.ok_cstream.println("BPEL File: "+bpel_name);
		
		List hosts_list = new ArrayList();
		List daemon_list = new ArrayList();
		
		for (int i = 0; i < ser_hosts_list.size(); i++) {
			
			//
			// Create a host SessionAddress and add it to the list
			//
			SessionAddress tmp;
			String daemon_transport = "HTTP";
			String daemon_password = null;
			try {
				String dat = (String)ser_hosts_list.get(i);
				dat = HexData.hexStringToString(dat);
				tmp = SessionAddress.fromString(dat);
				
				daemon_transport = (String)daemon_transports.get(""+i);
				daemon_password = (String)daemon_passwords.get(""+i);
			} catch (Exception e) {
				tmp = new SessionAddress("localhost",SessionAddress.TRANSPORT_PORT_ANY,SessionAddress.TRANSPORT_PORT_ANY,(String)ser_hosts_list.get(i),BPELProgram.DEFAULT_DAEMON_PORT,BPELProgram.DEFAULT_DAEMON_PORT);
			}
			hosts_list.add(tmp);
			
			//
			// Create a daemon SessionAddress and add it to the list
			//
			SessionAddress daemontmp = new SessionAddress(
									tmp.getInitiatorHost(),
									tmp.getInitiatorPortMinimum(),
									tmp.getInitiatorPortMaximum(),
									tmp.getListenerHost(),
									tmp.getListenerPortMinimum(),
									tmp.getListenerPortMaximum()
								);
			
			//use the daemon info in the maps to create SessionAddress objects
			if (daemon_password != null) {
				if (daemon_password.length() != 0) {
					daemontmp.setRequiresPassword(true);
					daemontmp.setPassword(daemon_password);
				}
			}
			
			if (daemon_transport != null) {
				if (daemon_transport.equals("HTTPS")) {
					daemontmp.setRequiresEncryption(true);
				} else {
					daemontmp.setRequiresEncryption(false);
				}
			} else {
				daemontmp.setRequiresEncryption(false);
			}
			
			daemon_list.add(daemontmp);
		}
		
		//
		// If no hosts are specified add a local host address with default settings
		//
		if (hosts_list.size() == 0) {
			SessionAddress tmp = new SessionAddress("localhost",SessionAddress.TRANSPORT_PORT_ANY,SessionAddress.TRANSPORT_PORT_ANY,"localhost",BPELProgram.DEFAULT_DAEMON_PORT,BPELProgram.DEFAULT_DAEMON_PORT); 
			hosts_list.add(tmp);

			SessionAddress daemontmp = new SessionAddress(
					tmp.getInitiatorHost(),
					tmp.getInitiatorPortMinimum(),
					tmp.getInitiatorPortMaximum(),
					tmp.getListenerHost(),
					tmp.getListenerPortMinimum(),
					tmp.getListenerPortMaximum()
				);
			
			daemontmp.setRequiresEncryption(false);
			daemontmp.setRequiresPassword(false);

			daemon_list.add(daemontmp);
		}
				
		
		String bpel_source = null;
		String base_uri = null;

		try {
			bpel_source = provider.getBpelSource();
			
			base_uri = provider.getBpelUri();

		} catch (Exception e) {
			printFatalProblem(console,"Failed to get BPEL source from provider "+provider_node.getName()+" ("+provider_node.getId()+")",e);
			UiPlugin.DBG.logVisibleError(e,"Failed to get BPEL source from provider "+provider_node.getName()+" ("+provider_node.getId()+")",false);
			return;
		}
		
		boolean isDebug = mode.equalsIgnoreCase("debug");
		boolean verbose = conf.getAttribute("translator_verbose",false);
		
		boolean exitOnDisconnect = !conf.getAttribute("translator_headless",false);
		
		BPELProgram bpelprog = new WorkbenchBPELProgram(engine_name, base_uri, bpel_source);

		//
		//set the source locator (for debug mode stuff)
		//
//XXX		try {
//XXX			BPELSourceLocator loc = (BPELSourceLocator)launch.getSourceLocator();
//XXX			loc.setSource(bpel_source);
//XXX		} catch (Exception e) {
//XXX			e.printStackTrace();
//XXX		}
		
		//
		// Set the coordinator host in the engine program
		//
		bpelprog.setCoordinatorHost((SessionAddress)daemon_list.get(0),(SessionAddress)hosts_list.get(0));
		
		//
		// Add worker hosts to the engine program
		//
		if (hosts_list.size() == 1) {
			SessionAddress host_address = (SessionAddress)hosts_list.get(0);
			SessionAddress daemon_address = (SessionAddress)daemon_list.get(0);
			bpelprog.addWorkerHost(daemon_address,host_address);
		} else {
			for (int i = 1; i < hosts_list.size(); i++) {
				SessionAddress host_address = (SessionAddress)hosts_list.get(i);
				SessionAddress daemon_address = (SessionAddress)daemon_list.get(i);
				bpelprog.addWorkerHost(daemon_address,host_address);
			}
		}
		
		//
		// Add any dependencies specified by the user
		//
		//
		// Load extra Program dependencies from user settings
		//
		try {
			java.util.List jar_refs = conf.getAttribute("jar_list",new ArrayList());
			for (int i = 0; i < jar_refs.size(); i++) {
				JARDependency tmp = JARDependency.fromHex((String)jar_refs.get(i));
				if (tmp.getEnabled()) {
					bpelprog.addJarDependency(tmp.getFilePath());
				}
			}
		} catch (Exception e) {
			printFatalProblem(console,"Failed to add dependency to BPEL program",e);
			UiPlugin.DBG.logVisibleError(e,"Failed to add dependency to BPEL program",false);
			return;
		}
		
		//
		// Run the program
		//
		WorkbenchBPELEngine bpelengine = new WorkbenchBPELEngine(bpelprog);

		//listen for job abort requests from the user
		AbortTracker tracker = new AbortTracker(bpelengine,monitor);
		
		try {
			//run the program
			bpelengine.runProgram(console,isDebug,verbose,engine_impl.equals("mini"),exitOnDisconnect);
			
			if (isDebug) {
				//get the debug target
				launch.addDebugTarget(bpelengine.getDebugTarget());
			}
			
			//start listening for aborts
			tracker.start();
			
			//wait for the program to complete
			bpelengine.waitForProgramCompletion();
		} catch (Exception e) {
			printFatalProblem(console,"Error running BPEL program",e);
			UiPlugin.DBG.logVisibleError(e,"Error running BPEL program",false);
			return;
		}
		
		//when we're pretty sure we've got all the remaining messages, print the program finish marker
		console.ok_cstream.println("");
		console.ok_cstream.println("BPEL Process Complete");
		
		tracker.finished();
	}
	
}