/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.message;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.eclipse.stp.b2j.core.jengine.internal.multiplex.MultiplexerInputStream;
import org.eclipse.stp.b2j.core.jengine.internal.multiplex.MultiplexerOutputStream;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.UnqueuedMutex;

/**
 * @author amiguel
 * 
 * A class to do Message based Transactions.
 * The basis of this class is that the client can do transactions ( 1
 * message to = 1 message from) against a server, dealing only with Message
 * objects.
 *
 * The transaction client would typically be instantiated on one end of a socket
 * using the input and output streams of the socket.
 */
public class BPlaneTransactionClient implements MTTransactionClient {

MultiplexerOutputStream xout;
MultiplexerInputStream xin;

	/** 
	 * Construct a new TransactionClient
	 * @param bin the InputStream to read messages from
	 * @param bout the OutputStream to write messages to
	 */ 
	public BPlaneTransactionClient(InputStream bin, OutputStream bout, ThreadGroup tg, String name) {

		xin = new MultiplexerInputStream(bin,name+"_BufferPlaneClientIncomingMultiplexer",tg);
		xout = new MultiplexerOutputStream(bout,name+"_BufferPlaneClientOutgoingMultiplexer",tg);

	}//end method

	/** 
	 * Construct a new TransactionClient
	 * @param bin the InputStream to read messages from
	 * @param bout the OutputStream to write messages to
	 */ 
	public BPlaneTransactionClient(InputStream bin, OutputStream bout) {

		xin = new MultiplexerInputStream(bin,"BufferPlaneClientIncomingMultiplexer");
		xout = new MultiplexerOutputStream(bout,"BufferPlaneClientOutgoingMultiplexer");

	}//end method

	/**
	 * Perform a transaction based a message.
	 * @param m the message to send to the TransactionServer
	 */
	public Message doTransaction(Message m) throws IOException {

		if (org.eclipse.stp.b2j.core.jengine.internal.compiler.Switches.MESSAGES_WITH_CALLSTACK) {
			m.appendToCallStack(new Throwable("Message Callstack - transaction"));
		}
		
		Thread thread = Thread.currentThread();
		if (thread instanceof BPlaneServerThread) {
			//we are part of a message somewhere else
			//therefore we get to go up a buffer plane
			BPlaneServerThread bpthread = (BPlaneServerThread)thread;
			int plane = bpthread.plane;
			
			//TODO need to implement mod buffer planes (allow N transactions per buffer plane)
			
			//TODO having buffer planes search from 0 means the longest one doesnt always get served first
			//but this probably isnt a big deal
			
//Logger.info("Transaction rerouted from plane "+plane);					

			short p = 0;
			while (p <= (plane+1)) {
				
				//get the mutex for this plane
				UnqueuedMutex mutex = xout.getLock(p);
				
				boolean locked = false;
				
				if (p == (plane+1)) {
					//we didnt find any free buffer planes above ours,
					//so just block and wait til our one is free	
					mutex = xout.getLock(p);
//Logger.info("No free buffer planes - just waiting on "+(p));					
					mutex.lock();
					locked = true;
				} else {
					locked = mutex.testAndLock();
				}
				
				if (locked) {
					//locked this stream
//Logger.info("Locked plane "+p+", doing transaction");					
					
					try {
						OutputStream out = xout.getOutputStream(p);
						InputStream in = xin.getInputStream(p);
	
						MessageUtils.writeMessage(out,m);
						m = MessageUtils.readMessage(in);
					
					} catch (IOException e) {
						mutex.release();
						throw e;
					} catch (Throwable e) {
						mutex.release();
						throw new IOException("Error performing transaction "+e);
					}
					
					mutex.release();

					return m;
				}

//Logger.info("Failed to lock plane "+p+", trying "+(p+1));					
				
				//failed to lock this plane, try the next one
				p++;
			}

		} else {
			//we are not an incoming message
			//therefore we use buffer plane 0
			
			OutputStream out = xout.getOutputStream((short)0);
			InputStream in = xin.getInputStream((short)0);
			
			UnqueuedMutex mutex = xout.getLock((short)0);

			mutex.lock();
//Logger.info("Source message locked plane 0, doing transaction");					

			//TODO need to implement Fixed Length Transaction Queues as an option
			//NOTE: length 1 = this (no queue)
		
			try {
				MessageUtils.writeMessage(out,m);
				m = MessageUtils.readMessage(in);

			} catch (IOException e) {
				mutex.release();
				throw e;
			} catch (Throwable e) {
				mutex.release();
				throw new IOException("Error performing transaction "+e);
			}
			mutex.release();
		}
		
		return m;
	}//end method

}//end class
