/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.workspace;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.internal.model.RuntimeCore;
import org.eclipse.stp.sc.common.workspace.WorkspaceManager;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.stp.sc.jaxws.builders.ScJavaToWsdlBuilder;
import org.eclipse.stp.sc.jaxws.preferences.PreferencesAccessor;
import org.eclipse.stp.sc.jaxws.preferences.SCPreferenceConstants;
import org.eclipse.stp.sc.jaxws.properties.ScJaxWsPropertyConstants;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor;
import org.eclipse.stp.sc.jaxws.runtimeprovider.RuntimeProviderManager;



/**
 * define a Celtix oriented nature as well as the necessary lifecycle methods to manage it
 */
public class ScNature implements IProjectNature {

	public static final String NATURE_ID = "org.eclipse.stp.sc.jaxws.nature";
	public static final String JAVA_NATURE_ID = "org.eclipse.jdt.core.javanature";
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(ScNature.class);
    
    private IProject theProject;
    
    private String runtimeProviderUID;
    
    private String projectMode = ScJaxWsPropertyConstants.PROJECT_MODE_JAVAFIRST;

    public void configure() throws CoreException {
        
        String mode = theProject.getPersistentProperty(
        		ScJaxWsPropertyConstants.PROPERTY_KEY_MODE);
        if (mode == null || mode.equals(ScJaxWsPropertyConstants.PROJECT_MODE_JAVAFIRST)) {
        	configureJavaFirst();
        } else {
        	configureWsdlFirst();
        }
        
    }
    
    private void configureJavaFirst() throws CoreException {
    	WorkspaceManager.addBuilder(theProject, "org.eclipse.stp.sc.annvalidator", 
        		WorkspaceManager.BuilderOrder.FIRST);
    	//we do not need to create jaxb type class within stp.
    	//since the runtime tool such as apache cxf will generate those for us in the latest version
//        WorkspaceManager.addBuilder(theProject, ScJaxbBuilder.BUILDER_ID, 
//        		WorkspaceManager.BuilderOrder.FIRST);
        //LOG.debug("configure jaxws project with j2wsdl builder");
        //WorkspaceManager.addBuilder(theProject, ScJavaToWsdlBuilder.BUILDER_ID, 
        //		WorkspaceManager.BuilderOrder.LAST);
    }
    
    private void configureWsdlFirst() throws CoreException {
        //comment out the annvalidator first
//    	WorkspaceManager.addBuilder(theProject, "org.eclipse.stp.sc.annvalidator", 
//        		WorkspaceManager.BuilderOrder.FIRST);
    }
    
    public void deconfigure() throws CoreException {
        LOG.debug("deconfigure jaxws project with j2wsdl builder");
//        WorkspaceManager.removeBuilder(theProject, ScJaxbBuilder.BUILDER_ID);
        WorkspaceManager.removeBuilder(theProject, ScJavaToWsdlBuilder.BUILDER_ID);
        WorkspaceManager.removeBuilder(theProject, "org.eclipse.stp.sc.annvalidator"); //todo: need to revisit to use AnnValidator.BUILDER_ID after
    }
    

    public IProject getProject() {
        return theProject;
    }

    public void setProject(IProject value) {
        this.theProject = value;
    }

    public static void addToProject(IProject project, String projectMode) throws CoreException {

        if (!project.hasNature(JavaCore.NATURE_ID)) {
            MessageDialog.openWarning(null,
                                      ScJaxWsResources.getString("ScNature.no.java.dialog.title"),
                                      ScJaxWsResources.getString("ScNature.no.java.dialog.message"));
            return; 
        }
        addToProject(JavaCore.create(project), projectMode);
    }

    private static void initializeProjectProperties(IProject project, IRuntimeKitProcessor rtProcessor) {
		try {

			//soap12 support
			if (PreferencesAccessor.getSoap12Option()) {
				project.setPersistentProperty(
						ScJaxWsPropertyConstants.PROPERTY_KEY_SOAP_OPTION,
                        SCPreferenceConstants.KEY_SOAP12);
			} else {
				project.setPersistentProperty(
						ScJaxWsPropertyConstants.PROPERTY_KEY_SOAP_OPTION,
                        SCPreferenceConstants.KEY_SOAP11);
			}
		} catch (CoreException e) {
			LOG.debug("Exception got in initializing project properties", e);
		}
	}

	public static void addToProject(IJavaProject jproject, String projectMode) throws CoreException {
        IProject project = jproject.getProject();

        if (!project.hasNature(NATURE_ID)) {
            
        	String runtimeType = RuntimeCore.getRuntimeType(project);
            String runtimeId = RuntimeCore.getRuntimeID(project);
            IRuntimeKitProcessor rtProcessor = 
              RuntimeProviderManager.getInstance().getRuntimeKitProcessor(runtimeType, runtimeId);
            

            // add the SC nature
            project.setPersistentProperty(
            		ScJaxWsPropertyConstants.PROPERTY_KEY_MODE,
                    projectMode);
            IProjectDescription desc = project.getDescription();
            String[] natures = desc.getNatureIds();
            String[] newNatures = new String[natures.length + 1];
			System.arraycopy(natures, 0, newNatures, 1, natures.length);
			newNatures[0] = NATURE_ID;
			
            
            desc.setNatureIds(reorderNature(newNatures));
            project.setDescription(desc, null);
            
            ScNature theNature = new ScNature();
            theNature.setProject(project);
            String providerUID = RuntimeProviderManager.getRuntimeProviderUID(runtimeType);
            theNature.setRuntimeProviderId(providerUID);
            theNature.setProjectMode(projectMode);

            LOG.debug("adding libraries to project");
            if (rtProcessor != null) {
            	initializeProjectProperties(project, rtProcessor);
//            	 complete the build path based on the kit used
                rtProcessor.completeJavaProjectBuildPath(jproject);
            }
        }
    }
	
	private static String[] reorderNature(String[] natures){
		
		
		List<String> natureList = new ArrayList<String>();
		for(int i = 0; i < natures.length; i++){
			natureList.add(natures[i]);
		}
		if(natureList.contains(JAVA_NATURE_ID));
		{			
			natureList.remove(NATURE_ID);
			natureList.add(natureList.indexOf(JAVA_NATURE_ID), NATURE_ID);
		}
		
		String[] newNatures = new String[natureList.size()];
		for(int i = 0; i < natureList.size(); i++){
			newNatures[i] = natureList.get(i);
		}
		return newNatures;
	}

    public static void removeFromProject(IProject project) throws CoreException {
        if (project.hasNature(JavaCore.NATURE_ID)) {
            IJavaProject javaProject = JavaCore.create(project);
            removeFromProject(javaProject);
        }
    }

    public static void removeFromProject(IJavaProject jproject) throws CoreException {
        IProject project = jproject.getProject();

        // clean the build path based on the kit used
        String runtimeType = RuntimeCore.getRuntimeType(project);
        String runtimeId = RuntimeCore.getRuntimeID(project);
        RuntimeProviderManager.getInstance().getRuntimeKitProcessor(runtimeType, runtimeId)
            .clearJavaProjectBuildPath(jproject);

        if (project.hasNature(NATURE_ID)) {
            IProjectDescription desc = project.getDescription();
            String[] natures = desc.getNatureIds();
            ArrayList<String> newNatures = new ArrayList<String>(Arrays.asList(natures));
            newNatures.remove(newNatures.indexOf(NATURE_ID));
            desc.setNatureIds(newNatures.toArray(new String[newNatures.size()]));
            project.setDescription(desc, null);
            
            
        }
    }
    
    public static boolean hasNature(IProject aProject) throws CoreException {
    	return aProject.hasNature(NATURE_ID);
        
    }

    public String getRuntimeProviderUID() {
        if (runtimeProviderUID == null) {
            try {
                runtimeProviderUID = theProject.getPersistentProperty(
                    		ScJaxWsPropertyConstants.PROPERTY_KEY_RUNTIME_PROVIDER_UID);
            } catch (CoreException e) {
                // the project is not accessible
                runtimeProviderUID = null;
            }
        }
        return runtimeProviderUID;
    }

    public void setRuntimeProviderId(String aRuntimeProviderId) {
        runtimeProviderUID = aRuntimeProviderId;
        
        try {
            theProject.setPersistentProperty(
            		ScJaxWsPropertyConstants.PROPERTY_KEY_RUNTIME_PROVIDER_UID,
                    aRuntimeProviderId);
        } catch (CoreException e) {
            // the project is not accessible
            runtimeProviderUID = null;
        }
    }
    
    public String getProjectMode() {
        if (projectMode == null) {
            try {
                projectMode = theProject.getPersistentProperty(
                    		ScJaxWsPropertyConstants.PROPERTY_KEY_MODE);
            } catch (CoreException e) {
                projectMode = null;
            }
        }
        return projectMode;
    }
    
    public void setProjectMode(String mode) {
    	projectMode = mode;
        try {
            theProject.setPersistentProperty(
            		ScJaxWsPropertyConstants.PROPERTY_KEY_MODE,
                    mode);
        } catch (CoreException e) {
            projectMode = null;
        }
    }
    
}
