/*******************************************************************************
 * Copyright (c) 2005-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.emf.refactoring;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.ui.PlatformUI;

public class ResourceNameChangeAdapter extends EContentAdapter {

	private IResourceChangeListener mResourceChangeListener = new IResourceChangeListener() {

		public void resourceChanged(IResourceChangeEvent event) {
			handleResourceChanged(event);
		}
	};
	private Map mIResourceToObjectListMap = new HashMap();
	private Map mEReferenceToExtension = new HashMap();
	private boolean mProcessingNameChange = false;

	public static IResource getWorkspaceResource(URI uri) {
		try {
			return ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(
					new Path(FileLocator.toFileURL(new URL(uri.toString()))
							.getFile()));
		}
		catch (MalformedURLException e) {
			e.printStackTrace();
		}
		catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	public ResourceNameChangeAdapter() {
		super();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(
				mResourceChangeListener, IResourceChangeEvent.POST_CHANGE);
	}

	public ResourceNameChangeAdapter(List extensions) {
		this();
		addReferenceExtensions(extensions);
	}

	public void addReferenceExtension(IReferenceNameChangeExtension extension) {
		mEReferenceToExtension.put(extension.getFeature(), extension);
	}

	public void addReferenceExtensions(List extensions) {
		if (extensions != null) {
			for (Iterator it = extensions.iterator(); it.hasNext();) {
				addReferenceExtension((IReferenceNameChangeExtension) it.next());
			}
		}
	}

	public void notifyChanged(Notification msg) {
		try {
			if (!mProcessingNameChange) {
				// We don't need to pollute the model if we're just tracking
				// changes
				// to resources in the resource set.
				if (!mEReferenceToExtension.isEmpty()) {
					super.notifyChanged(msg);
				}

				if (msg.getNotifier() instanceof ResourceSet) {
					handleResourceSetChanged(msg);
				}
				else if (mEReferenceToExtension.containsKey(msg.getFeature())) {
					handleExtensionReferenceChanged(msg);
				}
				else if (msg.getFeature() instanceof EReference) {
					switch (msg.getEventType()) {
					case Notification.SET:
						if (msg.getOldValue() instanceof EObject) {
							unregisterExistingObject((EObject) msg
									.getOldValue());
						}
						if (msg.getNewValue() instanceof EObject) {
							registerExistingObject((EObject) msg.getNewValue());
						}
						break;
					case Notification.UNSET:
						if (msg.getOldValue() instanceof EObject) {
							unregisterExistingObject((EObject) msg
									.getOldValue());
						}
						break;
					case Notification.ADD:
						if (msg.getNewValue() instanceof EObject) {
							registerExistingObject((EObject) msg.getNewValue());
						}
						break;
					case Notification.ADD_MANY:
						for (Iterator it = ((List) msg.getNewValue())
								.iterator(); it.hasNext();) {
							Object next = it.next();
							if (next instanceof EObject) {
								registerExistingObject((EObject) next);
							}
						}
						break;
					case Notification.REMOVE:
						if (msg.getOldValue() instanceof EObject) {
							unregisterExistingObject((EObject) msg
									.getOldValue());
						}
						break;
					case Notification.REMOVE_MANY:
						for (Iterator it = ((List) msg.getOldValue())
								.iterator(); it.hasNext();) {
							Object next = it.next();
							if (next instanceof EObject) {
								unregisterExistingObject((EObject) next);
							}
						}
						break;
					}
				}
			}
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(
				mResourceChangeListener);
		for (Iterator it = mIResourceToObjectListMap.values().iterator(); it
				.hasNext();) {
			((List) it.next()).clear();
		}
		mIResourceToObjectListMap.clear();
	}

	protected void handleResourceSetChanged(Notification msg) {
		switch (msg.getEventType()) {
		case Notification.ADD:
			registerResource((Resource) msg.getNewValue());
			break;
		case Notification.ADD_MANY:
			for (Iterator it = ((List) msg.getNewValue()).iterator(); it
					.hasNext();) {
				registerResource((Resource) it.next());
			}
			break;
		case Notification.REMOVE:
			unregisterResource((Resource) msg.getOldValue());
			break;
		case Notification.REMOVE_MANY:
			for (Iterator it = ((List) msg.getOldValue()).iterator(); it
					.hasNext();) {
				unregisterResource((Resource) it.next());
			}
			break;
		}
	}

	protected void handleExtensionReferenceChanged(Notification msg) {
		IReferenceNameChangeExtension extension = (IReferenceNameChangeExtension) mEReferenceToExtension
				.get(msg.getFeature());
		switch (msg.getEventType()) {
		case Notification.SET:
			unregisterObject(extension, (EObject) msg.getNotifier(), msg
					.getOldValue());
			registerObject(extension, (EObject) msg.getNotifier(), msg
					.getNewValue());
			break;
		case Notification.UNSET:
			unregisterObject(extension, (EObject) msg.getNotifier(), msg
					.getOldValue());
			break;
		case Notification.ADD:
			registerObject(extension, (EObject) msg.getNotifier(), msg
					.getNewValue());
			break;
		case Notification.ADD_MANY:
			for (Iterator it = ((List) msg.getNewValue()).iterator(); it
					.hasNext();) {
				registerObject(extension, (EObject) msg.getNotifier(), it
						.next());
			}
			break;
		case Notification.REMOVE:
			unregisterObject(extension, (EObject) msg.getNotifier(), msg
					.getOldValue());
			break;
		case Notification.REMOVE_MANY:
			for (Iterator it = ((List) msg.getOldValue()).iterator(); it
					.hasNext();) {
				unregisterObject(extension, (EObject) msg.getNotifier(), it
						.next());
			}
			break;
		}
	}

	protected void registerResource(Resource resource) {
		IResource file = getWorkspaceResource(resource.getURI());
		if (file != null) {
			addObjectReference(file, resource);
		}
	}

	protected void unregisterResource(Resource resource) {
		IResource file = getWorkspaceResource(resource.getURI());
		if (file != null) {
			removeObjectReference(file, resource);
		}
	}

	protected void registerObject(IReferenceNameChangeExtension extension,
			EObject object, Object value) {
		IResource file = extension.getReferencedResource(object, value);
		if (file != null) {
			addObjectReference(file,
					new ObjectExtensionEntry(extension, object));
		}
	}

	protected void unregisterObject(IReferenceNameChangeExtension extension,
			EObject object, Object value) {
		IResource file = extension.getReferencedResource(object, value);
		if (file != null) {
			removeObjectReference(file, new ObjectExtensionEntry(extension,
					object));
		}
	}

	protected void registerExistingObject(EObject object) {
		for (Iterator it = object.eClass().getEAllStructuralFeatures()
				.iterator(); it.hasNext();) {
			EStructuralFeature feature = (EStructuralFeature) it.next();
			if (mEReferenceToExtension.containsKey(feature)) {
				registerObject(
						(IReferenceNameChangeExtension) mEReferenceToExtension
								.get(feature), object, object.eGet(feature));
			}
		}
	}

	protected void unregisterExistingObject(EObject object) {
		for (Iterator it = object.eClass().getEAllStructuralFeatures()
				.iterator(); it.hasNext();) {
			EStructuralFeature feature = (EStructuralFeature) it.next();
			if (mEReferenceToExtension.containsKey(feature)) {
				unregisterObject(
						(IReferenceNameChangeExtension) mEReferenceToExtension
								.get(feature), object, object.eGet(feature));
			}
		}
	}

	protected void addObjectReference(IResource file, Object object) {
		List references = (List) mIResourceToObjectListMap.get(file);
		if (references == null) {
			references = new ArrayList();
			mIResourceToObjectListMap.put(file, references);
		}
		references.add(object);
	}

	protected void removeObjectReference(IResource file, Object object) {
		List references = (List) mIResourceToObjectListMap.get(file);
		if (references != null) {
			references.remove(object);
			if (references.isEmpty()) {
				mIResourceToObjectListMap.remove(file);
			}
		}
		else {
			mIResourceToObjectListMap.remove(file);
		}
	}

	protected void handleResourceChanged(IResourceChangeEvent event) {
		if (event.getDelta() == null) {
			return;
		}

		try {
			mProcessingNameChange = true;
			event.getDelta().accept(new IResourceDeltaVisitor() {

				public boolean visit(IResourceDelta delta) {
					if (delta.getResource().getType() == IResource.FILE) {
						if (delta.getKind() == IResourceDelta.REMOVED
								&& (delta.getFlags() & IResourceDelta.MOVED_TO) == IResourceDelta.MOVED_TO) {
							IFile changed = (IFile) delta.getResource();
							if (mIResourceToObjectListMap.containsKey(changed)) {
								List references = (List) mIResourceToObjectListMap
										.get(changed);
								mIResourceToObjectListMap.remove(changed);
								for (Iterator it = references.iterator(); it
										.hasNext();) {
									Object object = it.next();
									if (object instanceof Resource) {
										updateResourceReference(
												(Resource) object, delta);
									}
									else {
										updateObjectReference(
												(ObjectExtensionEntry) object,
												delta);
									}
								}
								mIResourceToObjectListMap.put(ResourcesPlugin
										.getWorkspace().getRoot().getFile(
												delta.getMovedToPath()),
										references);
							}
						}
						return false;
					}
					return true;
				}
			});
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		finally {
			mProcessingNameChange = false;
		}
	}

	protected void updateResourceReference(Resource resource,
			IResourceDelta delta) {
		resource.setURI(URI.createPlatformResourceURI(delta.getMovedToPath()
				.toString()));
	}

	protected void updateObjectReference(final ObjectExtensionEntry entry,
			final IResourceDelta delta) {
		// In case this causes any UI components to be updated.
		PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {

			public void run() {
				try {
					entry.getExtension().updateReference(entry.getObject(),
							delta);
				}
				catch (Exception e) {
					e.printStackTrace();
				}
			}
		});

	}

	protected static class ObjectExtensionEntry {

		private IReferenceNameChangeExtension mExtension;
		private EObject mObject;

		public ObjectExtensionEntry(IReferenceNameChangeExtension extension,
									EObject object) {
			mExtension = extension;
			mObject = object;
		}

		public EObject getObject() {
			return mObject;
		}

		public IReferenceNameChangeExtension getExtension() {
			return mExtension;
		}

		public int hashCode() {
			return mObject.hashCode();
		}

		public boolean equals(Object object) {
			return mObject.equals(object);
		}
	}

}
