package org.eclipse.stp.soas.deploy.core;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.internal.deploy.ui.properties.DeployOutputFolderPropertyPage;
import org.eclipse.wst.server.core.IServer;

public class DeployPackageManager {

	public static final QualifiedName sPackageExtensionPropertyKey = new QualifiedName(
			DeployCorePlugin.getDefault().getBundle().getSymbolicName(),
			"packageExtensionID"); //$NON-NLS-1$
	
	Map mIdToLogicalExtension;
	SortedSet mLogicalExtensionByFileExtension;
	Map mIdToConfigurableExtension;
	SortedSet mConfigurableExtensionByFileExtension;
	Map mIdToPhysicalExtension;
	Map mPhysicalExtensionByFileExtension;
	
	public DeployPackageManager() {
		super();
	}

	public boolean isPackage(IFile file) {
		return getPackageExtension(file) != null;
	}

	public boolean isLogicalPackage(IFile file) {
		return getLogicalPackageExtension(file) != null;
	}

	public boolean isConfigurablePackage(IFile file) {
		return getConfigurablePackageExtension(file) != null;
	}

	public boolean isPhysicalPackage(IFile file) {
		return getPhysicalPackageExtension(file) != null;
	}

	public IPackage getPackage(IFile file) {
		IPackage pkg;
		IPackageExtension ipe = getPackageExtension(file);
		if (ipe == null) {
			pkg = null;
		}
		else if (ipe instanceof IConfigurablePackageExtension) {
			pkg = ((IConfigurablePackageExtension) ipe).adaptFile(file);
		}
		else if (ipe instanceof IServiceDescriptorExtension) {
			pkg = ((IServiceDescriptorExtension) ipe).adaptFile(file);
		}
		else if (ipe instanceof IPhysicalPackageExtension) {
			pkg = ((IPhysicalPackageExtension) ipe).adaptFile(file);
		}
		else {
			if (DeployCorePlugin.getDefault().isDebugging()) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.unknownPackageExtension", //$NON-NLS-1$
										new Object[] { ipe.getClass()}));
			}
			pkg = null;
		}
		return pkg;
	}

	public IPhysicalPackage getPhysicalPackage(IFile file) {
		IPackage pkg = getPackage(file);
		if (pkg != null && pkg instanceof IPhysicalPackage) {
			return (IPhysicalPackage) pkg;
		}
		return null;
	}

	/**
	 * Returns the output folder for a specified logical package and deployment
	 * configuration. If the configuration of the package has not been
	 * overridden, the default location for the package output is returned. If
	 * the configuration of the package has been overridden, the output location
	 * is specific to the deployment configuration. For example:
	 * <p>
	 * No package configuration: /&lt;package project&gt;/&lt;deployment
	 * output&gt;/&lt;project relative path to package&gt;/&lt;deploy driver
	 * name&gt;
	 * </p>
	 * <p>
	 * With package configuration: /&lt;package project&gt;/&lt;deployment
	 * output&gt;/&lt;project relative path to deploy file>/&lt;server name&gt;
	 * </p>
	 * 
	 * @param pkg
	 * @param deployConfig
	 * @return
	 */
	public IFolder getOutputFolder(IServiceDescriptor pkg, DeployConfiguration deployConfig) {
		IPath outputFolderPath;
		IWorkspaceRoot wsRoot = pkg.getFile().getWorkspace().getRoot();
		IDeployTarget target = Utilities.adaptToIDeployTarget(deployConfig);
		IPackageConfiguration pkgConfig = getPackageConfiguration(pkg,
				deployConfig);
		if (pkgConfig == null) {
			IFile pkgFile = pkg.getFile();
			IPath pkgFilePath = pkgFile.getProjectRelativePath();
			outputFolderPath = new Path(DeployOutputFolderPropertyPage
					.getOutputContainerPath(pkgFile.getProject()));
			outputFolderPath = outputFolderPath.append(pkgFilePath);
			outputFolderPath = outputFolderPath.append(target.getServer().getName());
		}
		else {
			// If a configuration override is associated with the package,
			// the deployment profile must exist in the workspace.
			IFile deployFile = wsRoot.getFile(new Path(deployConfig.eResource()
					.getURI().path()).removeFirstSegments(1));
			outputFolderPath = new Path(DeployOutputFolderPropertyPage
					.getOutputContainerPath(deployFile.getProject()));
			outputFolderPath = outputFolderPath.append(deployFile
					.getProjectRelativePath());
			outputFolderPath = outputFolderPath.append(target
					.getServer().getName());
			
		}
		return wsRoot.getFolder(outputFolderPath);
	}

	public IFolder getOutputFolder(IServiceDescriptor pkg,
			String conName) {
		IFile pkgFile = pkg.getFile();
		IPath pkgFilePath = pkgFile.getProjectRelativePath();

		IPath outputFolderPath = new Path(DeployOutputFolderPropertyPage
				.getOutputContainerPath(pkgFile.getProject()));
		outputFolderPath = outputFolderPath.append(pkgFilePath);
		outputFolderPath = outputFolderPath.append(conName);
		
		return pkgFile.getWorkspace().getRoot().getFolder(outputFolderPath);
	}

	private IPackageConfiguration getPackageConfiguration(IServiceDescriptor ilp, DeployConfiguration deployConfig) {
		IPackageConfiguration retVal;
		byte[] configData = deployConfig.getConfigOverride();
		if (ilp instanceof IConfigurablePackage) {
			if (configData == null || configData.length == 0) {
				retVal = null;
			}
			else {
				ByteArrayInputStream bais = new ByteArrayInputStream(configData);
				try {
					retVal = ((IConfigurablePackageExtension) ilp
							.getExtension()).getPackageConfigurationManager()
							.createPackageConfiguration(ilp, bais);
				}
				catch (IOException e) {
					e.printStackTrace();
					retVal = ((IConfigurablePackageExtension) ilp
							.getExtension()).getPackageConfigurationManager()
							.createPackageConfiguration(ilp);
				}
				finally {
					try {
						bais.close();
					}
					catch (IOException e) {
					}
				}
			}
		}
		else {
			if (configData != null && configData.length == 0) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"CreateDeployPackagesJob.error.configurationSpecifiedForLogicalPackage")); //$NON-NLS-1$
			}
			retVal = null;
		}
		return retVal;
	}

	public IConfigurablePackage getConfigurablePackage(IFile file) {
		IPackage pkg = getPackage(file);
		if (pkg != null && pkg instanceof IConfigurablePackage) {
			return (IConfigurablePackage) pkg;
		}
		return null;
	}

	public IServiceDescriptor getLogicalPackage(IFile file) {
		IPackage pkg = getPackage(file);
		if (pkg != null && pkg instanceof IServiceDescriptor) {
			return (IServiceDescriptor) pkg;
		}
		return null;
	}

	public IPackageExtension getPackageExtension(IFile file) {
		IPackageExtension ipe = getConfigurablePackageExtension(file);
		if (ipe == null) {
			ipe = getLogicalPackageExtension(file);
			if (ipe == null) {
				ipe = getPhysicalPackageExtension(file);
			}
		}
		return ipe;
	}

	public IServiceDescriptorExtension getLogicalPackageExtension(IFile file) {
		IServiceDescriptorExtension ilpe = null;
		String extensionID = null;
		try {
			extensionID = file
					.getPersistentProperty(sPackageExtensionPropertyKey);
			if (extensionID != null) {
				ilpe = getLogicalPackageExtension(extensionID);
			}
		}
		catch (CoreException e) {
			// This is just a cached value. Should be safe to ignore.
		}
		if (ilpe == null) {
			String fileExt = file.getFileExtension();
			if (fileExt != null) {
				List extensions = getLogicalExtensions(fileExt);
				for (Iterator it = extensions.iterator(); ilpe == null
						&& it.hasNext();) {
					IServiceDescriptorExtension extension = (IServiceDescriptorExtension) it
							.next();
					if (extension.supportsFile(file)) {
						ilpe = extension;
					}
				}
			}
			if (ilpe != null && extensionID == null) {
				try {
					// the extensionID may represent a physical package
					// extension only set the property if extensionID is not
					// set.
					file.setPersistentProperty(sPackageExtensionPropertyKey,
							ilpe.getID());
				}
				catch (CoreException e) {
					// This is only for caching, so it should be safe to
					// ignore.
				}
			}
		}
		return ilpe;
	}

	public IConfigurablePackageExtension getConfigurablePackageExtension(IFile file) {
		IConfigurablePackageExtension icpe = null;
		String extensionID = null;
		try {
			extensionID = file
					.getPersistentProperty(sPackageExtensionPropertyKey);
			if (extensionID != null) {
				icpe = getConfigurablePackageExtension(extensionID);
			}
		}
		catch (CoreException e) {
			// This is just a cached value. Should be safe to ignore.
		}
		if (icpe == null) {
			String fileExt = file.getFileExtension();
			if (fileExt != null) {
				List extensions = getConfigurableExtensions(fileExt);
				for (Iterator it = extensions.iterator(); icpe == null
						&& it.hasNext();) {
					IConfigurablePackageExtension extension = (IConfigurablePackageExtension) it
							.next();
					if (extension.supportsFile(file)) {
						icpe = extension;
					}
				}
			}
			if (icpe != null && extensionID == null) {
				try {
					// the extensionID may represent a physical package
					// extension only set the property if extensionID is not
					// set.
					file.setPersistentProperty(sPackageExtensionPropertyKey,
							icpe.getID());
				}
				catch (CoreException e) {
					// This is only for caching, so it should be safe to
					// ignore.
				}
			}
		}
		return icpe;
	}

	public IPhysicalPackageExtension getPhysicalPackageExtension(IFile file) {
		IPhysicalPackageExtension ippe = null;
		String extensionID = null;
		try {
			extensionID = file
					.getPersistentProperty(sPackageExtensionPropertyKey);
			if (extensionID != null) {
				ippe = getPhysicalPackageExtension(extensionID);
			}
		}
		catch (CoreException e) {
			// This is just a cached value. Should be safe to ignore.
		}
		if (ippe == null) {
			String fileExt = file.getFileExtension();
			List extensions = getPhysicalPackageExtensions(fileExt);
			for (Iterator it = extensions.iterator(); ippe == null
					&& it.hasNext();) {
				IPhysicalPackageExtension extension = (IPhysicalPackageExtension) it
						.next();
				if (extension.supportsFile(file)) {
					ippe = extension;
				}
			}
			if (ippe != null) {
				try {
					file.setPersistentProperty(sPackageExtensionPropertyKey,
							ippe.getID());
				}
				catch (CoreException e) {
					// This is only for caching, so it should be safe to
					// ignore.
				}
			}
		}
		return ippe;
	}

	public IServiceDescriptorExtension getLogicalPackageExtension(String id) {
		IServiceDescriptorExtension ilpe;
		if (id == null || !mIdToLogicalExtension.containsKey(id)) {
			ilpe = null;
		}
		else {
			ilpe = (IServiceDescriptorExtension) mIdToLogicalExtension.get(id);
		}
		return ilpe;
	}

	public List getLogicalExtensions(String fileExtension) {
		return new ArrayList(mLogicalExtensionByFileExtension.subSet(
				fileExtension, fileExtension + '\0'));
	}

	public IConfigurablePackageExtension getConfigurablePackageExtension(String id) {
		IConfigurablePackageExtension icpe;
		if (id == null || !mIdToLogicalExtension.containsKey(id)) {
			icpe = null;
		}
		else {
			icpe = (IConfigurablePackageExtension) mIdToConfigurableExtension
					.get(id);
		}
		return icpe;
	}

	public List getConfigurableExtensions(String fileExtension) {
		return new ArrayList(mConfigurableExtensionByFileExtension.subSet(
				fileExtension, fileExtension + '\0'));
	}

	public IPhysicalPackageExtension getPhysicalPackageExtension(String id) {
		IPhysicalPackageExtension ippe;
		if (id == null || !mIdToPhysicalExtension.containsKey(id)) {
			ippe = null;
		}
		else {
			ippe = (IPhysicalPackageExtension) mIdToPhysicalExtension.get(id);
		}
		return ippe;
	}

	public List getPhysicalPackageExtensions(String fileExtension) {
		List retVal;
		if (fileExtension == null) {
			retVal = new ArrayList();
		}
		else {
	        if(mPhysicalExtensionByFileExtension.containsKey(fileExtension)){
	            retVal = (List)mPhysicalExtensionByFileExtension.get(fileExtension);
	        }
	        else{
	            retVal = new ArrayList();
	        }
		}
		return retVal;
	}

}