/*******************************************************************************
 * Copyright (c) 2003-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.wizards;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.stp.soas.internal.deploy.util.DeployUtilPlugin;
import org.eclipse.swt.widgets.Composite;

/**
 * This class is the wizard page used to prompt the user to select a project and
 * a file name for the new map.
 * 
 * A single IWizardPageCompleteListener registered with this page will be called
 * for every event to let it know whether or not this page is complete.
 * 
 * The Workspace used for detecting duplicate files may be set by calling
 * setWorkspace(IWorkspace).
 */
public class WizardNewFileCreationPage extends
		org.eclipse.ui.dialogs.WizardNewFileCreationPage implements
		ISummaryDataSource {

	/** The parent Workspace of this wizard page. */
	private IWorkspace mWorkspace = DeployUtilPlugin.getWorkspace();
	private IContainer mSelectedContainer = null;
	private IFile mSelectedFile = null;

	/** The extension for the new file. */
	private String smFileExtension;

	public WizardNewFileCreationPage(String pageName) {
		super(pageName, StructuredSelection.EMPTY);
	}

	public WizardNewFileCreationPage(String pageName, String title,
										ImageDescriptor titleImage) {
		super(pageName, StructuredSelection.EMPTY);
		setTitle(title);
		setImageDescriptor(titleImage);
	}

	public void init(String fileExtension, ISelection selection) {
		smFileExtension = fileExtension;
		if (!selection.isEmpty()) {
			Object o = ((IStructuredSelection) selection).getFirstElement();
			if (o instanceof IResource) {
				IResource res = (IResource) o;
				if (res.getType() == IResource.FOLDER
						|| res.getType() == IResource.PROJECT) {
					mSelectedContainer = (IContainer) res;
				}
				else if (res.getType() == IResource.FILE) {
					mSelectedContainer = res.getParent();
					if (res instanceof IFile) {
						mSelectedFile = (IFile) res;
					}
				}
			}
		}
	}

	public String getFullFileName() {
		// get current value of file name string and append the required
		// file name extension to it if it does not already have it.
		String fileName = getFileName();
		if (smFileExtension != null && !fileName.endsWith('.' + smFileExtension)) {
			fileName += '.' + smFileExtension;
		}
		return fileName;
	}

	public IFile getOutputFile() {		
		IContainer container = (IContainer) mWorkspace.getRoot().findMember(
				getContainerFullPath());
		return mWorkspace.getRoot().getFile(
				container.getFullPath().append(getFullFileName()));
	}

	/**
	 * Checks whether the user can advance to the next wizard page. In this
	 * case, it just checks whether file name entered by the user is unique in
	 * the current workspace.
	 * 
	 * @return <tt>true</tt> if the user can flip to the next page;
	 *         <tt>false</tt> otherwise
	 */
	public boolean canFlipToNextPage() {
		String fileName = getFullFileName();
		// check to see if the system file already exists
		//CR336500 - getFolder() does not work when the container path is the
		// root project folder.
		IResource container;
		IPath localContainerPath = getContainerFullPath();
		if (localContainerPath != null) {
			if (localContainerPath.segmentCount() == 1) {
				// Must use getProject()
				container = mWorkspace.getRoot().getProject(
						localContainerPath.lastSegment());
			}
			else {
				// OK to use getFolder()
				container = mWorkspace.getRoot().getFolder(
						getContainerFullPath());
			}
			//CR319958 - The proper way to get the full path for all files
			// (including files in project folders that are not directly in the
			// workspace folder, e.g. imported projects).
			IPath absoluteContainerPath = container.getLocation();
			File systemFile = new File(absoluteContainerPath.toOSString(),
					fileName);
			if (systemFile.exists()) {
				// display an error message and disable "Next" button on wizard
				setErrorMessage(DeployUtilPlugin
						.getDefault()
						.getResourceString(
								"WizardNewFileCreationPage.ERROR.FileExists", new Object[] { container.getFullPath().append(fileName)})); //$NON-NLS-1$
				return false;
			}
		}
		return super.canFlipToNextPage();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		super.createControl(parent);
		if (mSelectedContainer != null) {
			setContainerFullPath(mSelectedContainer.getFullPath());
		}
		if (mSelectedFile != null) {
			int idx = mSelectedFile.getName().lastIndexOf('.');

			if (idx == -1) {
				setFileName(mSelectedFile.getName());
			}
			else {
				setFileName(mSelectedFile.getName().substring(0, idx));
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#createAdvancedControls(org.eclipse.swt.widgets.Composite)
	 */
	protected void createAdvancedControls(Composite parent) {
		// We don't want any...
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#validateLinkedResource()
	 */
	protected IStatus validateLinkedResource() {
		return Status.OK_STATUS;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.datatools.connectivity.internal.ui.wizards.ISummaryDataSource#getSummaryData()
	 */
	public List getSummaryData() {
		List summaryData = new ArrayList(1);

		summaryData.add(new String[] {
				DeployUtilPlugin.getDefault().getResourceString(
						"WizardNewFileCreationPage.summary.fileName"), //$NON-NLS-1$
				getOutputFile().getFullPath().toString()});

		return summaryData;
	}

}