package org.eclipse.stp.core.infrastructure.emf;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.emf.common.notify.NotificationChain;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.ecore.util.InternalEList;
import org.eclipse.stp.core.infrastructure.assertion.Assert;

/**
 * This is a specialized list which is used for dealing with substitution groups
 * when serializing to XML. This list works in conjunction with
 * {@link org.eclipse.stp.core.infrastructure.emf.FeatureAdapter}.
 * 
 * @since 1.0
 * @see org.eclipse.stp.core.infrastructure.emf.FeatureAdapter
 * 
 */
public class FeatureAwareList implements InternalEList {

   private final EStructuralFeature  abstractFeature;

   private final FeatureMap.Internal featureMap;

   private final InternalEList       modelList;

   /**
    * Create a specialized {@link EList} which can is used to automatically
    * detect and use the proper {@link EStructuralFeature} defined for a
    * substitution group element for any given {@link EObject}.
    * 
    * @param anAbstractFeature -
    *           This is the EStructuralFeature which is the base feature for the
    *           substitution group.
    * @param aFeatureMap -
    *           This is the FeatureMap for managing the substitution group.
    */
   public FeatureAwareList(EStructuralFeature anAbstractFeature,
         FeatureMap.Internal aFeatureMap) {
      abstractFeature = anAbstractFeature;
      featureMap = aFeatureMap;
      modelList = (InternalEList) featureMap.list(abstractFeature);
   }

   public void add(int index, Object o) {

      Assert.isTrue(o instanceof EObject);
      EObject eObj = (EObject) o;

      EStructuralFeature subfeature = FeatureAdapter.getFeature(eObj);
      Assert.isNotNull(subfeature);

      featureMap.list(subfeature).add(index, eObj);
   }

   public boolean add(Object o) {
      Assert.isTrue(o instanceof EObject);
      EObject eObj = (EObject) o;

      EStructuralFeature subfeature = FeatureAdapter.getFeature(eObj);
      Assert.isNotNull(subfeature);

      return featureMap.list(subfeature).add(eObj);
   }

   public boolean addAll(Collection c) {
      return addAll(0, c);
   }

   public boolean addAll(int index, Collection c) {
      int initialSize = featureMap.size();
      // Object[] collection = c.toArray();
      for (Iterator itr = c.iterator(); itr.hasNext();) {
         Object o = itr.next();
         Assert.isTrue(o instanceof EObject);
         EObject eObj = (EObject) o;

         EStructuralFeature subfeature = FeatureAdapter.getFeature(eObj);
         Assert.isNotNull(subfeature);

         featureMap.list(subfeature)
               .add(index, Collections.singletonList(eObj));
      }
      return featureMap.size() > initialSize;
   }

   public void clear() {
      modelList.clear();
   }

   public boolean contains(Object o) {
      return modelList.contains(o);
   }

   public boolean containsAll(Collection c) {
      return modelList.containsAll(c);
   }

   public boolean equals(Object o) {
      return modelList.equals(o);
   }

   public Object get(int index) {
      return modelList.get(index);
   }

   public int hashCode() {
      return modelList.hashCode();
   }

   public int indexOf(Object o) {
      return modelList.indexOf(o);
   }

   public boolean isEmpty() {
      return modelList.isEmpty();
   }

   public Iterator iterator() {
      return modelList.iterator();
   }

   public int lastIndexOf(Object o) {
      return modelList.lastIndexOf(o);
   }

   public ListIterator listIterator() {
      return modelList.listIterator();
   }

   public ListIterator listIterator(int index) {
      return modelList.listIterator(index);
   }

   public Object remove(int index) {
      return modelList.remove(index);
   }

   public boolean remove(Object o) {
      return modelList.remove(o);
   }

   public boolean removeAll(Collection c) {
      return modelList.removeAll(c);
   }

   public boolean retainAll(Collection c) {
      return modelList.retainAll(c);
   }

   public Object set(int index, Object element) {
      return modelList.set(index, element);
   }

   public int size() {
      return modelList.size();
   }

   public List subList(int fromIndex, int toIndex) {
      return modelList.subList(fromIndex, toIndex);
   }

   public Object[] toArray() {
      return modelList.toArray();
   }

   public Object[] toArray(Object[] a) {
      return modelList.toArray(a);
   }

   public void move(int newPosition, Object object) {
      modelList.move(newPosition, object);
   }

   public Object move(int newPosition, int oldPosition) {
      return modelList.move(newPosition, oldPosition);
   }

   public Object basicGet(int index) {
      return modelList.basicGet(index);
   }

   public List basicList() {
      return modelList.basicList();
   }

   public Iterator basicIterator() {
      return modelList.basicIterator();
   }

   public ListIterator basicListIterator() {
      return modelList.basicListIterator();
   }

   public ListIterator basicListIterator(int index) {
      return modelList.basicListIterator(index);
   }

   public NotificationChain basicRemove(Object object,
         NotificationChain notifications) {
      return modelList.basicRemove(object, notifications);
   }

   public NotificationChain basicAdd(Object object,
         NotificationChain notifications) {
      return modelList.basicAdd(object, notifications);
   }

   public void addUnique(Object object) {
      modelList.addUnique(object);
   }

   public void addUnique(int index, Object object) {
      modelList.addUnique(index, object);
   }

   public Object setUnique(int index, Object object) {
      return modelList.setUnique(index, object);
   }

   public boolean addAllUnique(Collection arg0) {
      return modelList.addAllUnique(arg0);
   }

   public boolean addAllUnique(int arg0, Collection arg1) {
      return modelList.addAllUnique(arg0, arg1);
   }

   public boolean basicContains(Object object) {
      return modelList.basicContains(object);
   }

   public boolean basicContainsAll(Collection arg0) {
      return modelList.basicContainsAll(arg0);
   }

   public int basicIndexOf(Object object) {
      return modelList.basicIndexOf(object);
   }

   public int basicLastIndexOf(Object object) {
      return modelList.basicLastIndexOf(object);
   }

   public Object[] basicToArray() {
      return modelList.basicToArray();
   }

   public Object[] basicToArray(Object[] arg0) {
      return modelList.basicToArray(arg0);
   }
}
