/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.tests.infrastructure.emf;

import java.util.HashSet;

import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.jst.j2ee.ejb.EJBJar;
import org.eclipse.stp.core.infrastructure.emf.EditModelEvent;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModel;
import org.eclipse.stp.core.infrastructure.emf.IEditModelFactory;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;
import org.eclipse.stp.core.internal.infrastructure.emf.EditModel;
import org.eclipse.stp.core.internal.infrastructure.emf.UnreleasedScribblerException;
import org.eclipse.stp.core.internal.infrastructure.emfworkbench.EMFWorkbenchContext;
import org.eclipse.stp.core.tests.infrastructure.util.TestDomain;
import org.eclipse.stp.core.tests.infrastructure.util.TestEditModelListener;
import org.eclipse.stp.core.tests.infrastructure.util.TestErrorHandler;
import org.eclipse.stp.core.tests.infrastructure.util.TestProgressMonitor;
import org.eclipse.stp.core.tests.infrastructure.util.TestWorkspace;

/**
 * The following unit test will ensure that the API methods on IEditModel behave
 * properly. Some methods begin with the name of the method under test, followed
 * by the patter "underscore+type of parameter" for all paramters in the method.
 * Some methods may end in "V#" indicating a varation on the test. Other methods
 * that test a grouping of API methods have a descriptive name.
 */
public class IEditModelTest extends TestCase {

   private static final String TEST_EDITMODEL_ID = "test.ejb.editmodel"; //$NON-NLS-1$

   public IEditModelTest(String name) {
      super(name);
   }

   public static TestSuite suite() {
      TestSuite suite = new TestSuite();
      suite.addTestSuite(IEditModelTest.class);
      // suite.addTest(new IEditModelTest("testDispose"));
      return suite;
   }

   /**
    * (1) Initialize the test workspace (2) Invoke a System.gc() (3) Cleanup the
    * scribbler listeners attached to already created EditModels. These
    * autocorrect whenever an interesting resource event occurs.
    */
   protected void setUp() throws Exception {
      super.setUp();
      TestWorkspace.init();

      // We go through the SOA EMFWorkbenchContext here to make sure
      // that the EMFResourceFactoryRegistry.INSTANCE is registered as
      // the resource factory against this project's ProjectResourceSet
      EMFWorkbenchContext context = EMFWorkbenchContext
            .getInstance(TestWorkspace.getTestProject());
      ResourceSet resourceSet = context.getResourceSet();

      assertTrue("Ensure there are zero resources loaded.", //$NON-NLS-1$
            resourceSet.getResources().size() == 0);

      // Some of the tests ensure that listeners are empty at the beginning
      TestWorkspace.encourageGC();
   }

   protected void tearDown() throws Exception {
      super.tearDown();

      // Some of the tests ensure that listeners are empty at the beginning
      TestWorkspace.encourageGC();

      TestDomain ejbDomain = TestDomain.createTestDomain(TestWorkspace
            .getTestProject());

      // We go through the SOA EMFWorkbenchContext here to make sure
      // that the EMFResourceFactoryRegistry.INSTANCE is registered as
      // the resource factory against this project's ProjectResourceSet
      EMFWorkbenchContext context = EMFWorkbenchContext
            .getInstance(TestWorkspace.getTestProject());
      ResourceSet resourceSet = context.getResourceSet();

      assertTrue("Ensure there are zero resources loaded.", //$NON-NLS-1$
            resourceSet.getResources().size() == 0);

      // Force the load of a resource which should prompt a
      // notification to the stale EditModelScribbler's listener
      Resource r = resourceSet.getResource(ejbDomain.getTestURI(), true);
      // We did this to trip notifications to validate that the listener
      // removes itself. So just unload and remove the resource
      if (r != null)
         r.unload();
      resourceSet.getResources().remove(r);

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), "test.ejb.editmodel"); //$NON-NLS-1$

      // This should be true after the last few lines execute.
      // The notification from loading the resource should
      // trip the stale listeners to remove themselves and
      // autocorrect the resource reference counts
      assertTrue("Ensure there are zero EditModelListeners.", //$NON-NLS-1$
            ((EditModel) editModel).getListeners().size() == 0);
   }

   /**
    * Preconditions: Valid Edit Model, no created scribblers
    * 
    * Postconditions: A scribbler must be created and tracked by the edit model
    * (the reference count must increase by one). The scribbler should then be
    * properly released and the reference count on the IEditModel should drop to
    * 0.
    * 
    * Methods under test:
    * {@link IEditModel#createScribbler(IScribblerDomain[], boolean)};
    */
   public void testCreateScribbler_LIScribblerDomain_Boolean_V1() {

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), "test.ejb.editmodel"); //$NON-NLS-1$

      try {
         IEditModelScribbler scribbler = editModel
               .createScribbler(new IScribblerDomain[] { TestDomain
                     .createTestDomain(TestWorkspace.getTestProject()) }, false);

         // ensure the reference count has increased
         // NOT API!
         assertTrue("Ensure the referece count increased.", //$NON-NLS-1$
               ((EditModel) editModel).getReferenceCount() == 1);

         scribbler.close(new NullProgressMonitor());

         // NOT API!
         assertTrue("Ensure the referece count decreased.", //$NON-NLS-1$
               ((EditModel) editModel).getReferenceCount() == 0);

      } catch (EditModelException e) {
         System.err.println("Could not create scribbler"); //$NON-NLS-1$
         e.printStackTrace();
      }
   }

   /**
    * Preconditions: Valid Edit Model, no created scribblers
    * 
    * Postconditions: A scribbler must be created and tracked by the edit model
    * (the reference count must increase by one). The scribbler should then be
    * properly released and the reference count on the IEditModel should drop to
    * 0.
    * 
    * V2: Verify that the scribbler is created, allow it to be garbage
    * collected, and ensure that the reference count auto-corrects.
    * 
    * Methods under test:
    * {@link IEditModel#createScribbler(IScribblerDomain[], boolean)};
    */
   public void testCreateScribbler_LIScribblerDomain_Boolean_V2() {

      UnreleasedScribblerException.exceptions = new HashSet();

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), "test.ejb.editmodel"); //$NON-NLS-1$

      try {

         // ensure the reference count is zero
         // NOT API!
         assertTrue("Ensure the reference count increased.", //$NON-NLS-1$
               ((EditModel) editModel).getReferenceCount() == 0);

         // we are going to create the scribbler and let it get nulled out
         IEditModelScribbler scribbler = editModel
               .createScribbler(new IScribblerDomain[] { TestDomain
                     .createTestDomain(TestWorkspace.getTestProject()) }, false);

         // The following line is useful but primarily removes the compiler
         // warning: "The local variable scribbler is not used"
         assertTrue(
               "Ensure the scribbler is NOT created dirty.", !scribbler.isDirty()); //$NON-NLS-1$

         // ensure the reference count has increased
         // NOT API!
         assertTrue("Ensure the referece count increased.", //$NON-NLS-1$
               ((EditModel) editModel).getReferenceCount() == 1);

         // explicitly null out the scribbler
         scribbler = null;

         System.out
               .println("---------BEGIN EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$

         TestWorkspace.encourageGC();
         // The GC cycle is not deterministic enough for this assertion
         // assertTrue("Ensure the reference count decreased.", ((EditModel)
         // editModel).getReferenceCount() == 0);//$NON-NLS-1$

         EMFWorkbenchContext context = EMFWorkbenchContext
               .getInstance(TestWorkspace.getTestProject());
         ResourceSet resourceSet = context.getResourceSet();

         // Force the load of a resource which should prompt a
         // notification to the stale EditModelScribbler's listener
         Resource r = resourceSet.getResource(TestDomain.createTestDomain(
               TestWorkspace.getTestProject()).getTestURI(), true);
         // We did this to trip notifications to validate that the listener
         // removes itself. So just unload and remove the resource
         if (r != null)
            r.unload();
         resourceSet.getResources().remove(r);

         assertTrue(
               "There should have been one unreleased scribbler exception thrown.", UnreleasedScribblerException.exceptions.size() == 1); //$NON-NLS-1$

         System.out
               .println("---------END EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$

      } catch (EditModelException e) {
         System.err.println("Could not create scribbler"); //$NON-NLS-1$
         e.printStackTrace();
      } finally {
         UnreleasedScribblerException.exceptions.clear();
      }
   }

   /**
    * Preconditions: Valid Edit Model, no created scribblers
    * 
    * Postconditions: A scribbler must be created and tracked by the edit model
    * (the reference count must increase by one). The scribbler should then be
    * properly released and the reference count on the IEditModel should drop to
    * 0.
    * 
    * V2: Verify that the scribbler is created, allow it to be garbage
    * collected, and ensure that the reference count auto-corrects.
    * 
    * V3: Ensure that the scribbler's internal EditModelListener is correctly
    * removed after the scribbler is gc'd.
    * 
    * Methods under test:
    * {@link IEditModel#createScribbler(IScribblerDomain[], boolean)};
    */
   public void testCreateScribbler_LIScribblerDomain_Boolean_V3() {

      UnreleasedScribblerException.exceptions = new HashSet();

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), "test.ejb.editmodel"); //$NON-NLS-1$

      try {

         TestDomain ejbDomain = TestDomain.createTestDomain(TestWorkspace
               .getTestProject());

         // This should be true after the last few lines execute.
         // The notification from loading the resource should
         // trip the stale listeners to remove themselves and
         // autocorrect the resource reference counts
         assertTrue("Ensure there are zero EditModelListeners.", //$NON-NLS-1$
               ((EditModel) editModel).getListeners().size() == 0);

         int initialRefCount = ((EditModel) editModel).getReferenceCount();

         // we are going to create the scribbler and let it get nulled out
         IEditModelScribbler scribbler = editModel.createScribbler(
               new IScribblerDomain[] { ejbDomain }, false);

         // The following line is useful but primarily removes the compiler
         // warning: "The local variable scribbler is not used"
         assertTrue(
               "Ensure the scribbler is NOT created dirty.", !scribbler.isDirty()); //$NON-NLS-1$

         // ensure the reference count has increased
         // NOT API!
         assertEquals(
               "Ensure the referece count increased by one.", initialRefCount + 1, ((EditModel) editModel).getReferenceCount()); //$NON-NLS-1$

         // the Scribbler should add a listener
         assertTrue(
               "Ensure there is exactly one EditModelListener (from the Scribbler).", //$NON-NLS-1$
               ((EditModel) editModel).getListeners().size() == 1);

         // explicitly null out the scribbler
         scribbler = null;

         System.out
               .println("---------BEGIN EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$

         TestWorkspace.encourageGC();

         // The GC cycle is not deterministic enough for this assertion
         // assertEquals("Ensure the referece count decreased by one.",
         // initialRefCount, ((EditModel) editModel).getReferenceCount() );
         // //$NON-NLS-1$

         // We go through the SOA EMFWorkbenchContext here to make sure
         // that the EMFResourceFactoryRegistry.INSTANCE is registered as
         // the resource factory against this project's ProjectResourceSet
         EMFWorkbenchContext context = EMFWorkbenchContext
               .getInstance(TestWorkspace.getTestProject());
         ResourceSet resourceSet = context.getResourceSet();

         assertTrue("Ensure there are zero resources loaded.", //$NON-NLS-1$
               resourceSet.getResources().size() == 0);

         // Force the load of a resource which should prompt a
         // notification to the stale EditModelScribbler's listener
         Resource resource = resourceSet.getResource(ejbDomain.getTestURI(),
               true);
         resource.unload();
         resourceSet.getResources().remove(resource);

         // the Scribbler's listener should remove itself
         // with the notification just above
         assertTrue("Ensure there are zero EditModelListeners.", //$NON-NLS-1$
               ((EditModel) editModel).getListeners().size() == 0);

         assertTrue(
               "There should have been one unreleased scribbler exception thrown.", UnreleasedScribblerException.exceptions.size() == 1); //$NON-NLS-1$

         System.out
               .println("---------END EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$

      } catch (EditModelException e) {
         System.err.println("Could not create scribbler"); //$NON-NLS-1$
         e.printStackTrace();
      } finally {
         UnreleasedScribblerException.exceptions.clear();
      }
   }

   /**
    * Preconditions: Valid Edit Model, no created scribblers
    * 
    * Postconditions: A scribbler must be created and tracked by the edit model
    * (the reference count must increase by one). The scribbler should then be
    * properly released and the reference count on the IEditModel should drop to
    * 0.
    * 
    * V4: Verify that when two scribblers are created, allowed to be garbage
    * collected, that the reference count auto-corrects.
    * 
    * Methods under test:
    * {@link IEditModel#createScribbler(IScribblerDomain[], boolean)};
    */
   public void testCreateScribbler_LIScribblerDomain_Boolean_V4() {

      UnreleasedScribblerException.exceptions = new HashSet();

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), "test.ejb.editmodel"); //$NON-NLS-1$

      try {

         int initialRefCount = ((EditModel) editModel).getReferenceCount();

         // we are going to create the scribbler and let it get nulled out
         IEditModelScribbler scribbler = editModel
               .createScribbler(new IScribblerDomain[] { TestDomain
                     .createTestDomain(TestWorkspace.getTestProject()) }, false);

         // we are going to create the scribbler and let it get nulled out
         IEditModelScribbler scribbler2 = editModel
               .createScribbler(new IScribblerDomain[] { TestDomain
                     .createTestDomain(TestWorkspace.getTestProject()) }, false);

         // The following line is useful but primarily removes the compiler
         // warning: "The local variable scribbler is not used"
         assertTrue(
               "Ensure the scribbler is NOT created dirty.", !scribbler.isDirty()); //$NON-NLS-1$

         // The following line is useful but primarily removes the compiler
         // warning: "The local variable scribbler is not used"
         assertTrue(
               "Ensure the scribbler is NOT created dirty.", !scribbler2.isDirty()); //$NON-NLS-1$

         // ensure the reference count has increased
         // NOT API!
         assertEquals(
               "Ensure the referece count increased by two.", initialRefCount + 2, //$NON-NLS-1$
               ((EditModel) editModel).getReferenceCount());

         System.out
               .println("---------BEGIN EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$

         // explicitly null out the scribblers
         scribbler = null;
         scribbler2 = null;

         TestWorkspace.encourageGC();
         // The GC cycle is not deterministic enough for this assertion
         // assertTrue("Ensure the referece count decreased.", ((EditModel)
         // editModel).getReferenceCount() == 0);//$NON-NLS-1$

         EMFWorkbenchContext context = EMFWorkbenchContext
               .getInstance(TestWorkspace.getTestProject());
         ResourceSet resourceSet = context.getResourceSet();

         // Force the load of a resource which should prompt a
         // notification to the stale EditModelScribbler's listener
         Resource r = resourceSet.getResource(TestDomain.createTestDomain(
               TestWorkspace.getTestProject()).getTestURI(), true);
         // We did this to trip notifications to validate that the listener
         // removes itself. So just unload and remove the resource
         if (r != null)
            r.unload();
         resourceSet.getResources().remove(r);

         assertTrue(
               "There should have been one unreleased scribbler exception thrown.", UnreleasedScribblerException.exceptions.size() == 2); //$NON-NLS-1$

         System.out
               .println("---------END EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$

      } catch (EditModelException e) {
         System.err.println("Could not create scribbler"); //$NON-NLS-1$
         e.printStackTrace();
      } finally {
         UnreleasedScribblerException.exceptions.clear();
      }

   }

   /**
    * 
    * The test should: create an edit model and create a scribbler. An
    * interesting resource should be loaded through a means other than the
    * scribbler (such as the EMF Workbench Context Resource Set). The edit model
    * should detect the load of the resource and notify its listeners. An event
    * of type "save" should be received with exactly one resource in its list of
    * changed resources.
    * 
    * Methods under test:
    * {@link IEditModel#addListener(org.eclipse.stp.core.infrastructure.emf.IEditModelListener)},
    * {@link IEditModel#removeListener(org.eclipse.stp.core.infrastructure.emf.IEditModelListener)}.
    * 
    */
   public void testNotifiactionOnLoad() throws Exception {

      TestDomain ejbDomain = TestDomain.createTestDomain(TestWorkspace
            .getTestProject());

      // We go through the SOA EMFWorkbenchContext here to make sure
      // that the EMFResourceFactoryRegistry.INSTANCE is registered as
      // the resource factory against this project's ProjectResourceSet
      EMFWorkbenchContext context = EMFWorkbenchContext
            .getInstance(TestWorkspace.getTestProject());
      ResourceSet resourceSet = context.getResourceSet();

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID);

      IEditModelScribbler scribbler = editModel.createScribbler(
            new IScribblerDomain[] { ejbDomain }, false);

      assertTrue("Ensure there are zero resources loaded.", //$NON-NLS-1$
            resourceSet.getResources().size() == 0);

      // add a test listener
      TestEditModelListener testListener = new TestEditModelListener(
            EditModelEvent.LOADED_RESOURCE);
      editModel.addListener(testListener);

      // Force the load of a resource which should prompt a
      // notification to the stale EditModelScribbler's listener
      Resource resource = resourceSet.getResource(ejbDomain.getTestURI(), true);

      editModel.removeListener(testListener);
      assertTrue(
            "A notification of the loaded resource should have been received.", testListener.getEvents().length == 1); //$NON-NLS-1$

      EditModelEvent event = testListener.getEvents()[0];
      assertTrue(
            "A notification of the loaded resource should have been received.", event.getEventCode() == EditModelEvent.LOADED_RESOURCE); //$NON-NLS-1$
      assertTrue(
            "The resources in the changed list should match the above resource.", event.getChangedResources().size() == 1 && event.getChangedResources().contains(resource)); //$NON-NLS-1$

      scribbler.close(null);

      resource.unload();
      resourceSet.getResources().remove(resource);
   }

   /**
    * 
    * The test should: create an edit model, create a scribbler, fetch a
    * resource from the scribbler, modify it, and invoke save. An event of type
    * "save" should be received with exactly one resource in its list of changed
    * resources.
    * 
    * Methods under test:
    * {@link IEditModel#addListener(org.eclipse.stp.core.infrastructure.emf.IEditModelListener)},
    * {@link IEditModel#removeListener(org.eclipse.stp.core.infrastructure.emf.IEditModelListener)}.
    * 
    */
   public void testNotifiactionOnSave() throws Exception {

      TestDomain ejbDomain = TestDomain.createTestDomain(TestWorkspace
            .getTestProject());

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID);

      IEditModelScribbler scribbler = editModel.createScribbler(
            new IScribblerDomain[] { ejbDomain }, false);

      // Force the load of a resource which should prompt a
      // notification to the stale EditModelScribbler's listener
      Resource resource = scribbler.getResource(ejbDomain.getTestURI());

      EJBJar jar = (EJBJar) resource.getContents().get(0);
      jar.setDisplayName("TestDisplayName"); //$NON-NLS-1$
      assertTrue(
            "Ensure the resource is marked as modified.", resource.isModified()); //$NON-NLS-1$

      TestProgressMonitor testMonitor = new TestProgressMonitor(3);

      // add a test listener
      TestEditModelListener testListener = new TestEditModelListener(
            EditModelEvent.SAVE);
      editModel.addListener(testListener);

      scribbler.save(true, testMonitor);

      editModel.removeListener(testListener);

      assertTrue(
            "A notification should have been received.", testListener.getEvents().length == 1); //$NON-NLS-1$

      EditModelEvent event = testListener.getEvents()[0];
      assertTrue(
            "A notification of the type 'save' should have been received.", event.getEventCode() == EditModelEvent.SAVE); //$NON-NLS-1$
      assertTrue(
            "The resources in the changed list should match the above resource.", event.getChangedResources().size() == 1 && event.getChangedResources().contains(resource)); //$NON-NLS-1$

      scribbler.close(null);

   }

   /**
    * 
    * The test should: create an edit model, create a scribbler, fetch a
    * resource from the scribbler, modify it, and invoke revert. An event of
    * type "revert" should be received with exactly one resource in its list of
    * changed resources.
    * 
    * Methods under test:
    * {@link IEditModel#addListener(org.eclipse.stp.core.infrastructure.emf.IEditModelListener)},
    * {@link IEditModel#removeListener(org.eclipse.stp.core.infrastructure.emf.IEditModelListener)}.
    * 
    */
   public void testNotifiactionOnRevert() throws Exception {

      TestDomain ejbDomain = TestDomain.createTestDomain(TestWorkspace
            .getTestProject());

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID);

      IEditModelScribbler scribbler = editModel.createScribbler(
            new IScribblerDomain[] { ejbDomain }, false);

      // Force the load of a resource which should prompt a
      // notification to the stale EditModelScribbler's listener
      Resource resource = scribbler.getResource(ejbDomain.getTestURI());

      EJBJar jar = (EJBJar) resource.getContents().get(0);
      jar.setDisplayName("TestDisplayName"); //$NON-NLS-1$
      assertTrue(
            "Ensure the resource is marked as modified.", resource.isModified()); //$NON-NLS-1$

      TestProgressMonitor testMonitor = new TestProgressMonitor(3);

      // add a test listener
      TestEditModelListener testListener = new TestEditModelListener(
            EditModelEvent.REVERT);
      editModel.addListener(testListener);
      scribbler.revert(testMonitor);
      editModel.removeListener(testListener);

      assertTrue(
            "A notification should have been received.", testListener.getEvents().length == 1); //$NON-NLS-1$

      EditModelEvent event = testListener.getEvents()[0];
      assertTrue(
            "A notification of the type 'revert' should have been received.", event.getEventCode() == EditModelEvent.REVERT); //$NON-NLS-1$
      assertTrue(
            "The resources in the changed list should match the above resource.", event.getChangedResources().size() == 1 && event.getChangedResources().contains(resource)); //$NON-NLS-1$

      scribbler.close(null);

   }

   /**
    * 
    * Precond: Two valid resources and a valid edit model
    * 
    * Test: Verify that the edit model declares that it is interested in one
    * resource but not the other. Resources are declared as interesting through
    * IScribblerDomains, so creating an IEditModelScribbler around the
    * 
    */
   public void testIsInterrestedInResourceResource() {

   }

   /*
    * Test method for
    * 'com.ibm.soa.infrastructure.emf.internal.EditModel.getOperationHistory()'
    */
   public void testGetOperationHistory() {
      // TODO Auto-generated method stub

   }

   /**
    * Ensure that the Error Handler can be overridden at any point in time.
    * 
    * Also ensure that a value of 'null' cannot be supplied.
    * 
    * Methods under test:
    * {@link IEditModel#setErrorHandler(org.eclipse.stp.core.infrastructure.emf.IEditModelErrorHandler)};
    */
   public void testSetErrorHandler() throws EditModelException {

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID);
      TestDomain ejbDomain = TestDomain.createTestDomain(TestWorkspace
            .getTestProject());
      IEditModelScribbler scribbler = editModel.createScribbler(
            new IScribblerDomain[] { ejbDomain }, false);

      // add a test error handler
      TestErrorHandler testErrorHandler = new TestErrorHandler();
      editModel.setErrorHandler(testErrorHandler);

      Resource resource = scribbler.getResource(new Path("nullpath")); //$NON-NLS-1$

      assertNull("The returned resource should be null.", resource); //$NON-NLS-1$
      assertTrue(
            "Ensure that a load error occurred for the resource.", testErrorHandler.handledLoadFailed); //$NON-NLS-1$

      // ensure that when we try to force an error, that no
      // runtime exceptions occur with a null error handler
      editModel.setErrorHandler(null);
      System.out
            .println("---------BEGIN EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$
      resource = scribbler.getResource(new Path("nullpath1")); //$NON-NLS-1$
      System.out
            .println("---------END EXPECTED ERROR LOG [" + getClass().getName() + "]--------------"); //$NON-NLS-1$//$NON-NLS-2$

      scribbler.close(null);

      // we must explicitly remove the resource because even
      // though the Edit Model won't support it, the Resource is
      // still created by EMF in the Resource Set
      EMFWorkbenchContext context = EMFWorkbenchContext
            .getInstance(TestWorkspace.getTestProject());
      ResourceSet resourceSet = context.getResourceSet();
      resourceSet.getResources().clear();

   }

   /**
    * 
    * Methods under test: {@link IEditModel#isDisposed()};
    */
   public void testDispose() throws Exception {

      IProject project = TestWorkspace.getTestProject();
      try {
         IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
               project, TEST_EDITMODEL_ID);
         TestEditModelListener testListener = new TestEditModelListener(
               EditModelEvent.PRE_DISPOSE);
         editModel.addListener(testListener);

         project.close(null);

         assertTrue(
               "The edit model should be disposed.", editModel.isDisposed()); //$NON-NLS-1$
         assertEquals(
               "The set of resources for the edit model should be empty.", 0, ((EditModel) editModel).getResources().size()); //$NON-NLS-1$

         assertEquals(
               "There should have been one event.", 1, testListener.getEvents().length); //$NON-NLS-1$
         EditModelEvent[] events = testListener.getEvents();
         assertEquals(
               "The event should have been of type pre_dispose", EditModelEvent.PRE_DISPOSE, events[0].getEventCode()); //$NON-NLS-1$

         project.open(null);

         EMFWorkbenchContext context = EMFWorkbenchContext
               .getInstance(TestWorkspace.getTestProject());
         ResourceSet resourceSet = context.getResourceSet();
         assertEquals(
               "The set of resources for the resource set should be empty.", 0, resourceSet.getResources().size()); //$NON-NLS-1$
      } finally {
         if (!project.isAccessible())
            project.open(null);
      }

   }

}
