/******************************************************************************
 * Copyright (c) 2007, Intalio Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Intalio Inc. - initial API and implementation
 *******************************************************************************
 * Dates       		 Author              Changes
 * Jan 24, 2007      Antoine Toulm   Creation
 */
package org.eclipse.stp.bpmn.diagram.generation.impl;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IPath;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gmf.runtime.diagram.core.services.ViewService;
import org.eclipse.gmf.runtime.diagram.core.util.ViewUtil;
import org.eclipse.gmf.runtime.emf.core.util.EObjectAdapter;
import org.eclipse.gmf.runtime.notation.Bounds;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.gmf.runtime.notation.Edge;
import org.eclipse.gmf.runtime.notation.Node;
import org.eclipse.gmf.runtime.notation.NotationFactory;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.stp.bpmn.Activity;
import org.eclipse.stp.bpmn.ActivityType;
import org.eclipse.stp.bpmn.Graph;
import org.eclipse.stp.bpmn.Lane;
import org.eclipse.stp.bpmn.MessagingEdge;
import org.eclipse.stp.bpmn.Pool;
import org.eclipse.stp.bpmn.SequenceEdge;
import org.eclipse.stp.bpmn.SubProcess;
import org.eclipse.stp.bpmn.diagram.edit.parts.ActivityEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.LaneEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.MessagingEdgeEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.PoolEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.PoolPoolCompartmentEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.SequenceEdgeEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.SubProcessEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.SubProcessSubProcessBodyCompartmentEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.SubProcessSubProcessBorderCompartmentEditPart;
import org.eclipse.stp.bpmn.diagram.part.BpmnDiagramEditorPlugin;
import org.eclipse.stp.bpmn.diagram.part.BpmnVisualIDRegistry;

/**
 * This class extends BPMNProcessGenerator by 
 * creating semantic elements, associating them optionally with a 
 * location and/or a size.
 * Then it is able to generate all the Views at once, 
 * and returning them to the user for further use.
 * @author <a href="mailto:atoulme@intalio.com">Antoine Toulm</a>
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class BPMNVisual2ProcessGenerator extends BPMNProcessGenerator {

	/**
	 * Default constructor, will create a virtual diagram.
	 */
	public BPMNVisual2ProcessGenerator() {
		
	}
	/**
	 * Will create a new diagram on the new file.
	 * @param path the path - it'd better be ending with bpmn_diagram.
	 */
	public BPMNVisual2ProcessGenerator(IPath path) {
		super(path);
	}
	/**
	 * Appends to the given diagram.
	 * @param diagram
	 */
	public BPMNVisual2ProcessGenerator(Diagram diagram) {
		super(diagram);
	}
	
	/**
	 * Keeps track of the positions, so that it creates constraints 
	 * when generating the semantic elements' views.
	 */
	private Map<EObject, Rectangle> positions = 
		new HashMap<EObject, Rectangle>();
	/**
	 * Map populated during the generateViews method. Keeps track of the
	 * link between semantic and notational elements,
	 * so that it is possible to select the shape to drop on the editor next.
	 */
	private Map<EObject, View> semantic2notationMap = 
		new HashMap<EObject, View>();
	/**
	 * List of sequence edges collected during the view generation,
	 * so that their views can be generated after.
	 */
	private List<SequenceEdge> collectedSequenceEdges = 
		new LinkedList<SequenceEdge>();
	/**
	 * List of messaging edges collected during the view generation,
	 * so that their views can be generated after.
	 */
	private List<MessagingEdge> collectedMessagingEdges = 
		new LinkedList<MessagingEdge>();
	
	/**
	 * Creates an activity, keep the bounds in memory to apply 
	 * them when generating the view.
	 * @param parent the parent or null
	 * @param name the name of the activity, ie the label of the shape
	 * @param activityType the type of the activity, 
	 * one of the values of ActivityType
	 * @param x the int corresponding to 
	 * the x position of the shape in the parent
	 * @param y the int corresponding to 
	 * the y position of the shape in the parent
	 * @param width the width of the shape
	 * @param height the height of the shape
	 * @return
	 */
	public Activity addActivity(Graph parent, String name, 
			int activityType, int x, int y, int width, int height) {
		Rectangle rect = new Rectangle(x, y, width, height);
		Activity act = super.addActivity(parent, name, activityType);
		positions.put(act, rect);
		return act;
	}
	
	/**
	 * Creates a semantic object represnting a lane and associate it with a pool
	 * @param pool the parent pool
	 * @param name the name of the lane
	 * @param x the x position in parent
	 * @param y the y position in parent
	 * @param width width of the lane
	 * @param height height of the lane
	 * @return
	 */
	public Lane addLane(Pool pool, String name, 
			int x, int y, int width, int height) {
		Rectangle rect = new Rectangle(x, y, width, height);
		Lane lane = super.addLane(pool, name);
		positions.put(lane, rect);
		return lane;
	}
	/**
	 * Creates the semantic element representing a pool
	 * @param name the name of the pool
	 * @param x its x position
	 * @param y its y position
	 * @param width its width 
	 * @param height its height
	 */
	public Pool addPool(String name, int x, int y, int width, int height) {
		Rectangle rect = new Rectangle(x, y, width, height);
		Pool pool = addPool(name);
		positions.put(pool,rect);
		return pool;
	}
	
	/**
	 * Generate the views corresponding to all the semantic elements contained
	 * in the diagram hold by the generator.
	 */
	public void generateViews() {
		
		semantic2notationMap.clear();
		
		recursiveGenerate(getSemanticModel().eContents());
		
		for (MessagingEdge mesEdge : collectedMessagingEdges) {
			semantic2notationMap.put(mesEdge, 
					generateMessagingEdgeView(mesEdge));
		}
		
		for (SequenceEdge seqEdge : collectedSequenceEdges) {
			semantic2notationMap.put(seqEdge, 
			generateSequenceEdgeView(seqEdge));
		}
	}
	
	@SuppressWarnings("unchecked")
	private void recursiveGenerate(List elements) {
		for (Object elt : elements) {
			if (elt instanceof Pool) {
				Node pool = generatePoolView((Pool) elt);
				semantic2notationMap.put((EObject) elt, pool);
				recursiveGenerate(((Pool) elt).eContents());
			} else if (elt instanceof SubProcess) {
				Node sp = generateActivityView((SubProcess) elt);
				semantic2notationMap.put((EObject) elt, sp);
				recursiveGenerate(((SubProcess) elt).eContents());
			} else if (elt instanceof Activity) {
				semantic2notationMap.put((EObject) elt, 
						generateActivityView((Activity) elt));
			} else if (elt instanceof SequenceEdge) {
				collectedSequenceEdges.add((SequenceEdge) elt);
			} else if (elt instanceof MessagingEdge) {
				collectedMessagingEdges.add((MessagingEdge) elt);
			} else if (elt instanceof Lane) {
				semantic2notationMap.put((EObject) elt, 
						generateLaneView((Lane) elt));
			} else if (elt instanceof EAnnotation) {
				// i won't generate a view for an EAnnotation.
			} else {
				throw new IllegalArgumentException(
						"Don't know how to generate a view for " + elt);
			}
		}
	}
	
	/**
	 * Generates the view for a given activity.
	 * @param activity
	 * @return Node
	 */
	private Node generateActivityView(final Activity activity) {
		InternalRecordingCommand command = 
			new InternalRecordingCommand() {

			@Override
			protected void doExecute() {

				Node parent = null;
				if (activity.getGraph() != null) {
					for (Object  child : semantic2notationMap.
							get(activity.eContainer()).getChildren()) {
						if (child instanceof Node && 
								String.valueOf(PoolPoolCompartmentEditPart.VISUAL_ID).
								equals(String.valueOf(((Node)child).getType()))) {
							parent = (Node) child;
						} else if (child instanceof Node && 
								String.valueOf(SubProcessSubProcessBodyCompartmentEditPart.VISUAL_ID).
								equals(String.valueOf(((Node)child).getType()))) {
							parent = (Node) child;
						} 
					}
				} else {
					// then the activity is certainly attached to a subprocess
					for (Object  child : semantic2notationMap.
							get(activity.getEventHandlerFor()).getChildren()) {
						if (child instanceof Node && 
								String.valueOf(SubProcessSubProcessBorderCompartmentEditPart.VISUAL_ID).
								equals(String.valueOf(((Node)child).getType()))) {
							parent = (Node) child;
						}
					}
				}
				if (parent == null) {
					throw new IllegalArgumentException(
						"Unable to find the compartment view for this graph " + 
						activity.eContainer());
				}

				Node aNode = null;

				if (activity.getActivityType().getValue()
						!= ActivityType.SUB_PROCESS) {
					
					String semanticHints2 = BpmnVisualIDRegistry.getType(
							ActivityEditPart.VISUAL_ID);
					aNode =
						ViewService.getInstance().createNode(
								new EObjectAdapter(activity),
								parent,
								semanticHints2, ViewUtil.APPEND,
								BpmnDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
					
				} else {
					String semanticHintsForSP = BpmnVisualIDRegistry.getType(
							SubProcessEditPart.VISUAL_ID);

					aNode =
						ViewService.getInstance().createNode(
								new EObjectAdapter(activity),
								parent,
								semanticHintsForSP, 
								ViewUtil.APPEND,
								BpmnDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
				}
				generateLayoutConstraint(aNode);
				
				setReturnedObject(aNode);
			}

		};
		
		_editingDomain.getCommandStack().execute(command);
		return (Node) command.getReturnedObject();
	}
	
	/**
	 * Generates a view for the given pool
	 * @param pool
	 * @return
	 */
	private Node generatePoolView(final Pool pool) {
		InternalRecordingCommand command = 
			new InternalRecordingCommand() {

			@Override
			protected void doExecute() {
				String semanticHints = BpmnVisualIDRegistry.getType(
						PoolEditPart.VISUAL_ID);
				Node poolNotationNode = 
					ViewService.getInstance().createNode(
							new EObjectAdapter(pool),
							getGraphicalModel(), 
							semanticHints, ViewUtil.APPEND,
							BpmnDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
				
				generateLayoutConstraint(poolNotationNode);
				setReturnedObject(poolNotationNode);
			}

		};
		_editingDomain.getCommandStack().execute(command);
		return (Node) command.getReturnedObject();
	}
	
	/**
	 * To be used only in a command.
	 * Generates the layout constraint associated to a semantic element,
	 * then applies to the node.
	 * @param node
	 */
	private void generateLayoutConstraint(Node node) {
		if (positions.get(node.getElement()) != null) {
			Bounds bounds = NotationFactory.eINSTANCE.createBounds();
			Rectangle rect = positions.get(node.getElement());
			bounds.setHeight(rect.height);
			bounds.setWidth(rect.width);
			bounds.setX(rect.x);
			bounds.setY(rect.y);
			node.setLayoutConstraint(bounds);
		}
	}
	
	/**
	 * Generates the view for the lane
	 * @param lane
	 * @return a Node representing the lane
	 */
	private Node generateLaneView(final Lane lane) {
		InternalRecordingCommand command = 
			new InternalRecordingCommand() {

			@Override
			protected void doExecute() {

				Node parent = null;
				for (Object  child : semantic2notationMap.
						get(lane.eContainer()).getChildren()) {
					if (child instanceof Node &&
				String.valueOf(PoolPoolCompartmentEditPart.VISUAL_ID).equals(
									String.valueOf(((Node)child).getType()))) {
						parent = (Node) child;
					} 
				}

				if (parent == null) {
					throw new IllegalArgumentException(
						"Unable to find the compartment view for this graph " + 
						lane.eContainer());
				}
				String semanticHints = BpmnVisualIDRegistry.getType(
						LaneEditPart.VISUAL_ID);
				Node laneNotationNode = 
					ViewService.getInstance().createNode(new EObjectAdapter(lane),
							parent, 
							semanticHints, ViewUtil.APPEND,
							BpmnDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
				generateLayoutConstraint(laneNotationNode);
				setReturnedObject(laneNotationNode);
			}

		};
		
		_editingDomain.getCommandStack().execute(command);
		return (Node) command.getReturnedObject();
	}
	
	/**
	 * Generates the view for the sequence edge.
	 * Note that the source and the target of the view are expected to be found
	 * through the semantic2notationsMap, so it needs to be populated first.
	 * @param seqEdge
	 * @return
	 */
	private Edge generateSequenceEdgeView(final SequenceEdge seqEdge) {
		InternalRecordingCommand command = 
			new InternalRecordingCommand() {

			@Override
			protected void doExecute() {
				
				String semanticHints = BpmnVisualIDRegistry.getType(
						SequenceEdgeEditPart.VISUAL_ID);
				Edge anEdge =
					(Edge) ViewService.getInstance().createEdge(
							new EObjectAdapter(seqEdge),
							getGraphicalModel(), 
							semanticHints, ViewUtil.APPEND,
							BpmnDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
				anEdge.setSource(semantic2notationMap.get(seqEdge.getSource()));
				anEdge.setTarget(semantic2notationMap.get(seqEdge.getTarget()));
				setReturnedObject(anEdge);
			}};
		_editingDomain.getCommandStack().execute(command);
			
		return (Edge) command.getReturnedObject();
	}
	
	/**
	 * Generates a view for the messaging edge
	 * Note that the source and the target of the view are expected to be found
	 * through the semantic2notationsMap, so it needs to be populated first.
	 * @param mesEdge
	 * @return
	 */
	private Edge generateMessagingEdgeView(final MessagingEdge mesEdge) {
		InternalRecordingCommand command = 
			new InternalRecordingCommand() {

			@Override
			protected void doExecute() {
				
				String semanticHints = BpmnVisualIDRegistry.getType(
						MessagingEdgeEditPart.VISUAL_ID);
				Edge anEdge =
					(Edge) ViewService.getInstance().createEdge(
							new EObjectAdapter(mesEdge),
							getGraphicalModel(), 
							semanticHints, ViewUtil.APPEND,
						BpmnDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
				anEdge.setSource(semantic2notationMap.get(mesEdge.getSource()));
				anEdge.setTarget(semantic2notationMap.get(mesEdge.getTarget()));
				setReturnedObject(anEdge);
			}};
		_editingDomain.getCommandStack().execute(command);

		return (Edge) command.getReturnedObject();
	}

	/**
	 * @return the semantic2notationMap
	 */
	public Map<EObject, View> getSemantic2notationMap() {
		return semantic2notationMap;
	}
	
	/**
	 * Sets the feature of the given element to the value.
	 * @param elt
	 * @param feature
	 * @param value
	 */
	public void setEObjectFeature(final EObject elt, 
			final EStructuralFeature feature, final Object value) {
		InternalRecordingCommand command = new InternalRecordingCommand() {

			@Override
			protected void doExecute() {
				elt.eSet(feature, value);
			}
			
		};
		_editingDomain.getCommandStack().execute(command);
	}
}
