/******************************************************************************
 * Copyright (c) 2006, Intalio Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Intalio Inc. - initial API and implementation
 *******************************************************************************
 * Dates       		 Author              Changes
 * Nov 28, 2006      Antoine Toulm   Creation
 */
package org.eclipse.stp.bpmn.policies;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.commands.UnexecutableCommand;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.ui.util.CustomData;
import org.eclipse.gmf.runtime.common.ui.util.ICustomData;
import org.eclipse.gmf.runtime.diagram.ui.commands.ICommandProxy;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editpolicies.DragDropEditPolicy;
import org.eclipse.gmf.runtime.diagram.ui.internal.commands.ClipboardCommand;
import org.eclipse.gmf.runtime.diagram.ui.internal.requests.PasteViewRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.DropObjectsRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.RequestConstants;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.notation.Bounds;
import org.eclipse.gmf.runtime.notation.Edge;
import org.eclipse.gmf.runtime.notation.LayoutConstraint;
import org.eclipse.gmf.runtime.notation.Location;
import org.eclipse.gmf.runtime.notation.Node;
import org.eclipse.gmf.runtime.notation.NotationFactory;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.stp.bpmn.Activity;
import org.eclipse.stp.bpmn.BpmnDiagram;
import org.eclipse.stp.bpmn.Graph;
import org.eclipse.stp.bpmn.MessagingEdge;
import org.eclipse.stp.bpmn.Pool;
import org.eclipse.stp.bpmn.SequenceEdge;
import org.eclipse.stp.bpmn.Vertex;
import org.eclipse.stp.bpmn.diagram.actions.ArrangeSelectionRecursivelyAction;

/**
 * Extending the diagram drag and drop edit policy so that
 * the diagram tries to understand the objects dropped upon it.
 * If it is just a single String, then it is used as 
 * the data of a clipboard fragment. If it is a bunch of EObjects,
 * it tries to adapt them into EObjects using a IContextAwareAdapterFactory. 
 * It finally returns a paste command with the data of the objects.
 * 
 * This implies that the pasted EObjects must be instances of View,
 * since the pasteCommand can only handle those.
 * 
 * @author <a href="mailto:atoulme@intalio.com">Antoine Toulm</a>
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class DiagramDragDropEditPolicyEx extends DragDropEditPolicy {

	/**
     * A straight forward cache.
     */
    private static class FileAndTimeStamp {
        public Object request;
        public List<? extends View> result;
        public String errorWhenNull;
        public int size;
    }
    
    private static WeakHashMap<Request, FileAndTimeStamp> CACHE = new WeakHashMap<Request, FileAndTimeStamp>();
    
	/* (non-Javadoc)
	 * @see org.eclipse.gmf.runtime.diagram.ui.editpolicies.DiagramDragDropEditPolicy#getDropObjectsCommand(org.eclipse.gmf.runtime.diagram.ui.requests.DropObjectsRequest)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Command getDropObjectsCommand(DropObjectsRequest dropRequest) {
		List objects = dropRequest.getObjects();
		CustomData data = null;
		if (objects.size() == 1 && objects.get(0) instanceof String) {
			if (((String) objects.get(0)).length() == 0) {
				return UnexecutableCommand.INSTANCE;
			}
			// we might just have a String representing a fragment there
			data = new CustomData(
						ClipboardCommand.DRAWING_SURFACE,
						((String) objects.get(0)).getBytes());
			
			PasteViewRequest pasteReq = 
				new PasteViewRequest(new ICustomData[] {data});
			
			return getHost().getCommand(pasteReq);
			
		} else {
			List<? extends View> views = null;
			synchronized(dropRequest) {
                FileAndTimeStamp ft = CACHE.get(dropRequest);
                if (ft != null) {
                    if (ft.size != dropRequest.getObjects().size()) {
                        CACHE.remove(dropRequest);
                        ft = null;
                    }
                }
                if (ft != null) {
                    views = ft.result;
                }
                ft = new FileAndTimeStamp();
                CACHE.put(dropRequest, ft);
                views = retrieveViews(objects);
                ft.request = dropRequest;
                ft.size = dropRequest.getObjects().size();
                ft.result = views;

			}
			final List eObjects = views;
			if (eObjects == null||eObjects.isEmpty()) {
				return UnexecutableCommand.INSTANCE;
			}
			
			CompoundCommand compound = new CompoundCommand();
			// first, for eachview, we create a dropCommand.
			for (Object view : eObjects) {
				compound.add(getDropViewCommand((View) view, dropRequest));
			}
			
			// then we add a refresh command.
			compound.add(getHost().getCommand(new Request(RequestConstants.REQ_REFRESH)));
			
			// finally we arrange the paerts selectively.
			// We try to arrange the parts more concerned, 
			// ie activities connected together.
			final List previousEditParts = new LinkedList(getHost().getChildren());
			AbstractTransactionalCommand command = 
				new AbstractTransactionalCommand(
						((IGraphicalEditPart) getHost()).getEditingDomain(), 
						"Arranging elements", 
						null) {

				@Override
				protected CommandResult doExecuteWithResult(
						IProgressMonitor m, IAdaptable info) {

					

					List<IGraphicalEditPart> newEditParts = 
						new LinkedList<IGraphicalEditPart>();
					for (Object editPart : getHost().getChildren()) {
						if (!previousEditParts.contains(editPart)) {
							newEditParts.add((IGraphicalEditPart) editPart);
						}
					}
					
					ArrangeSelectionRecursivelyAction.arrange(newEditParts);
					return CommandResult.newOKCommandResult();
				}
			};
			compound.add(new ICommandProxy(command));
			return compound;
		}
	}

	/**
	 * @notgenerated
	 * Retrieves a list of EObjects by adapting the objects
	 * with a custom factory
	 * @param objects
	 * @return a List of Eobjects.
	 */
	@SuppressWarnings("unchecked")
	private List retrieveViews(List objects) {
		List eObjects = new LinkedList();

		// now iterate over the objects and try to adapt them
		// as usual.
		for (Object input : objects) {
			if (input instanceof View) {
				eObjects.add(input);
				continue;
			}
			// try to adapt stuff with the factory
			Object result = Platform.getAdapterManager().
				getAdapter(input, View.class);
			if (result instanceof Collection) {
				eObjects.addAll((Collection) result);
			} else if (result instanceof View) {
				eObjects.add(result);
			}
		}

		if (!eObjects.isEmpty()) {
			return eObjects;
		}
		return null;
	}

	/**
	 * This method takes care of dropping a view on the host of the edit policy.
	 * It has some semantic rules installed in the canExecute method of
	 * the command it creates. It also takes care of the children, 
	 * by copying the references. Finally it inspects the children 
	 * to see if there are any edges remaining and 
	 * copy them to the new children structure.
	 * @param view
	 * @param request
	 * @return a command that enables the drop of a view 
	 * on the edit policy host.
	 */
	protected Command getDropViewCommand(final View view,
			final DropObjectsRequest request) {
		
		AbstractTransactionalCommand viewCommand = 
			new AbstractTransactionalCommand(
					((IGraphicalEditPart) getHost()).getEditingDomain(), 
					"Dropping a view", 
					null) {

			@Override
			public boolean canExecute() {
				EObject parent = ((IGraphicalEditPart) getHost()).
					resolveSemanticElement();
				EObject child = view.getElement();
				if (parent instanceof BpmnDiagram) {
					if (child instanceof Pool) {
						return true;
					} else if (child instanceof MessagingEdge) {
						return true;
					}
				} else if (parent instanceof Graph) {
					if (child instanceof Activity) {
						return true;
					} else if (child instanceof SequenceEdge) {
						return true;
					}
				}
				return false;
			}
			@Override
			protected CommandResult doExecuteWithResult(
					IProgressMonitor monitor, IAdaptable info)
			throws ExecutionException {
				
				// access the other domain. The view might come from an other
				// domain, so we need to have its go for a write transaction.
				TransactionalEditingDomain domain = 
					(TransactionalEditingDomain) AdapterFactoryEditingDomain.
					getEditingDomainFor(view);
				RecordingCommand command = 
					new RecordingCommand(
							((IGraphicalEditPart) getHost()).getEditingDomain() 
					) {
					@Override
					protected void doExecute() {
						// first copy the view and its semantic element.
						EcoreUtil.Copier copier = new EcoreUtil.Copier();
						View newView = (View) copier.copy(view);
						copier.copyReferences();
						// create the copy of the semantic element.
						EObject newSemantic = copier.copy(view.getElement());
						
						// insert the new view into the host diagram.
						((IGraphicalEditPart) getHost()).getNotationView().
						insertChild(newView);
						// now insert the semantic 
						// element into the new container.
						final EObject parentElement = 
							((IGraphicalEditPart) getHost()).
								resolveSemanticElement();
						if (newSemantic instanceof EAnnotation) {
							((EAnnotation) newSemantic).setEModelElement(
									(EModelElement) parentElement);
						} else if (parentElement instanceof BpmnDiagram) {
							if (newSemantic instanceof Pool) {
								((Pool) newSemantic).
								setBpmnDiagram((BpmnDiagram) parentElement);
							} else if (newSemantic instanceof MessagingEdge) {
								((MessagingEdge) newSemantic).setBpmnDiagram(
										(BpmnDiagram) parentElement);
							}
						} else if (parentElement instanceof Graph) {
							if (newSemantic instanceof Activity) {
								((Activity) newSemantic).
									setGraph((Graph) parentElement);
							} else if (newSemantic instanceof SequenceEdge) {
								((SequenceEdge) newSemantic).
								setGraph((Graph) parentElement);
							}
						}
						if (view.getElement() instanceof Vertex) {
							for (Object edge : ((Vertex) view.getElement()).
									getIncomingEdges()) {
								((SequenceEdge) edge).
									setTarget((Vertex) newSemantic);
								((SequenceEdge) edge).
									setGraph(((Vertex) newSemantic).getGraph());
							}
							for (Object edge : ((Vertex) view.getElement()).
									getOutgoingEdges()) {
								((SequenceEdge) edge).
									setSource((Vertex) newSemantic);
								((SequenceEdge) edge).
									setGraph(((Vertex) newSemantic).getGraph());
							}
						}
						copier.copyReferences();

						// inspect the edges and register 
						// their new sources and targets into Maps.
						TreeIterator treeIter = view.eAllContents();
						TreeIterator newTreeIter = newView.eAllContents();
						Map<Edge,Node> sources = new HashMap<Edge,Node>();
						Map<Edge,Node> targets = new HashMap<Edge,Node>();
						while (treeIter.hasNext()) {
							Object oldNode = treeIter.next();
							Object newNode = newTreeIter.next();
							if (oldNode instanceof Node) {
								for (Object edge : ((Node) oldNode).
										getSourceEdges()) {
									sources.put((Edge) edge, (Node) newNode);
								}

								for (Object edge : ((Node) oldNode).
										getTargetEdges()) {
									targets.put((Edge) edge, (Node) newNode);
								}
							}
						}

						// now iterate over the edges, 
						// copy them and set the copy's properties
						for (Edge oldEdge : sources.keySet()) {
							if (targets.get(oldEdge) == null) {
								// invalid edge, should not be copied.
								// we may raise an exception here.
								continue;
							}

							Edge newEdge = (Edge) copier.copy(oldEdge);
							copier.copyReferences();
							((View) getHost().getModel()).getDiagram().
								insertEdge(newEdge);
							
						}
						
						for (Edge oldEdge : sources.keySet()) {
							EcoreUtil.remove(oldEdge);
						}
						// finally (?) 
						// sets the location of the view to be the drop point.
						// this point is just a starting point that 
						// will be changed during the arrange operations.
						if (newView instanceof Node) {
							Object bounds = ((Node) newView).
								getLayoutConstraint();
							if (bounds == null) {
								bounds = NotationFactory.
									eINSTANCE.createBounds();
							}
							Point dropLocation = request.getLocation();
							Rectangle hostBounds = 
								((IGraphicalEditPart) getHost()).
									getFigure().getBounds();
							((IGraphicalEditPart) getHost()).
							getFigure().translateToRelative(dropLocation);
							dropLocation.x = dropLocation.x - hostBounds.x;
							dropLocation.y = dropLocation.y - hostBounds.y;
							
							if (bounds instanceof Bounds) {
								((Bounds) bounds).setX(dropLocation.x);
								((Bounds) bounds).setY(dropLocation.y);
							} else if (bounds instanceof Location) {
								((Location) bounds).setX(dropLocation.x);
								((Location) bounds).setY(dropLocation.y);
							}
							((Node) newView).setLayoutConstraint(
									(LayoutConstraint) bounds);
						}
					}

				};
				domain.getCommandStack().execute(command);
				return CommandResult.newOKCommandResult();
			}

		};

		CompoundCommand comp = new CompoundCommand();
		comp.add(new ICommandProxy(viewCommand));
		return comp;
	}
	
}
