/******************************************************************************
 * Copyright (c) 2006, Intalio Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Intalio Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.bpmn.policies;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.editpolicies.GraphicalEditPolicy;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.diagram.ui.commands.ICommandProxy;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editpolicies.EditPolicyRoles;
import org.eclipse.gmf.runtime.diagram.ui.requests.DropObjectsRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.RequestConstants;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.stp.bpmn.BpmnDiagram;
import org.eclipse.stp.bpmn.Graph;
import org.eclipse.stp.bpmn.MessagingEdge;
import org.eclipse.stp.bpmn.Pool;
import org.eclipse.stp.bpmn.SequenceEdge;
import org.eclipse.stp.bpmn.Vertex;
import org.eclipse.stp.bpmn.diagram.part.BpmnDiagramEditorPlugin;
import org.eclipse.stp.bpmn.diagram.ui.PopupBalloon;
import org.eclipse.stp.bpmn.dnd.IGenericEAnnotationDndHandler;

/**
 * Policy to handle the drop of EAnnotations on a EModelElement.
 * 
 * @author atoulme
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class GenericEAnnotationDropPolicy extends GraphicalEditPolicy {

	private EditPolicy previousEditPolicy;
	
	public GenericEAnnotationDropPolicy(IGraphicalEditPart part) {
		EditPolicy policy = part.getEditPolicy(EditPolicyRoles.DRAG_DROP_ROLE);
		if (policy != null) {
			previousEditPolicy = policy;
		}
	}
	/**
	 * HashMap holding the objects for the request,
	 * so that they are not computed several times.
	 * Note that this is a WeakHashMap, 
	 * so the requests are disposed by the garbage collector.
	 */
	@SuppressWarnings("unchecked")
	private Map<Request,List> _requestObjects = 
		new WeakHashMap<Request,List>(); 
	/**
	 * The balloon used for the feedback.
	 */
	private static PopupBalloon _balloon;
	
	/**
	 * the messages carried as popup descriptors.
	 */
	private List<IStatus> _messages = 
		new LinkedList<IStatus>();
	/* (non-Javadoc)
	 * @see org.eclipse.gef.editpolicies.AbstractEditPolicy#getTargetEditPart(org.eclipse.gef.Request)
	 */
	@Override
	public EditPart getTargetEditPart(Request request) {
		if (understandsRequest(request)) {
			return getHost();
		}
		return null;
	}

    /**
     * Only understands DRAG, DROP and DROP_ELEMENTS requests.
     *
     * @return whether the request is supported
     */
    @Override
    public boolean understandsRequest(Request request) {
        boolean r = RequestConstants.REQ_MOVE.equals(request.getType())
            || RequestConstants.REQ_DRAG.equals(request.getType())
            || RequestConstants.REQ_DROP.equals(request.getType());
        boolean dropObj = RequestConstants.REQ_DROP_OBJECTS.equals(request.getType());
        if (r || dropObj) {
            if (delegateUnderstandsRequest(request)) {
                return true;
            }
        }
        if (dropObj && request instanceof DropObjectsRequest) {
            if (internalUnderstandsRequest((DropObjectsRequest) request)) {
                return true;
            }
        }
        return false;
    }

	/**
	 * tries to use the drop objects as annotations, either directly or 
	 * by adapting them.
	 * @param req
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private boolean internalUnderstandsRequest(DropObjectsRequest req) {
		if (req instanceof DropObjectsRequest) {
			List objects = ((DropObjectsRequest) req).getObjects();
			
			List<EAnnotation> annotations = retrieveAnnotations(objects);
			if (annotations == null) {
				return false;
			}
			boolean accepted = accept(annotations,
					(DropObjectsRequest) req);
			if (accepted) {
				_requestObjects.put(req,annotations);
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Delegates the request to the previously installed edit policy.
	 * @param req
	 * @return true if the request is interesting for the edit policy
	 */
	private boolean delegateUnderstandsRequest(Request req) {
		if (previousEditPolicy != null) {
            if (previousEditPolicy.getHost() == null) {
                //just in case it had been removed.
                previousEditPolicy.setHost(getHost());
            }
			return previousEditPolicy.understandsRequest(req);
		}
		return false;
	}
	
	private Command delegateGetCommand(Request req) {
		if (previousEditPolicy != null) {
			return previousEditPolicy.getCommand(req);
		}
		return null;
	}
	/**
	 * retrieves the annotations by adapting them through a declared
	 * adapter factory.
	 * @param objects
	 * @return a list of EAnnotations or null.
	 */
	@SuppressWarnings("unchecked")
	private List<EAnnotation> retrieveAnnotations(List objects) {
		List<EAnnotation> annotations = new LinkedList<EAnnotation>();
		for (Object dropped : objects) {
			Object ann =Platform.getAdapterManager().
				getAdapter(dropped, EAnnotation.class);
			if (dropped instanceof EAnnotation) {
				ann = dropped;
			}
			if (ann == null || (!(ann instanceof EAnnotation))) {
				return null;
			}
			annotations.add((EAnnotation) ann);
		}
		return annotations;
	}
	
	@SuppressWarnings("unchecked")
	private boolean accept(List<EAnnotation> annotations,
			DropObjectsRequest request) {
		_messages.clear();
		boolean doit = true;
		for (EAnnotation annotation:annotations) {
			IGenericEAnnotationDndHandler handler =BpmnDiagramEditorPlugin.
			getInstance().
			getEAnnotationDragAndDropHandler(annotation.getSource());
			if (handler == null) {
				throw new IllegalArgumentException("No extension point is " +
						"defined for the annotation source " + 
						annotation.getSource());
			}
			IStatus status = handler.accept(annotation, 
					(IGraphicalEditPart) getHost());

			if (status.getSeverity() != IStatus.OK) {
				doit=false;

			} 

			if (status.getMessage()!= null&&status.getMessage().length() != 0) {
				_messages.add(status);
			}
		}
		if (!_messages.isEmpty()) {
			if (_balloon == null) {
				_balloon = new PopupBalloon(
						(IGraphicalEditPart) getHost());
			} 
			if (!_balloon.isShowing() &&
					(_balloon.showsOnThisEditPart(getHost()))) {
				// the message has already been shown once
				// on this edit part.
				return doit;
			}
			if (_balloon.isShowing() &&
					!(_balloon.showsOnThisEditPart(getHost()))) {
				_balloon.hide();
			}
			if (!_balloon.isShowing()) {
				for (IStatus descriptor:_messages){

					_balloon.addPopupBarDescriptor(descriptor.getMessage(),
							descriptor.getSeverity());
				}
				
				_balloon.showBalloon(
						request.getLocation(),
								getHost());
			}
		}
		return doit;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gef.editpolicies.AbstractEditPolicy#getCommand(org.eclipse.gef.Request)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Command getCommand(Request request) {
		if (!(request instanceof DropObjectsRequest)) {
			return null;
		}
			
		if (delegateUnderstandsRequest((DropObjectsRequest) request)) {
			Command command = delegateGetCommand(request);
			if (command != null && command.canExecute()) {
				return command;
			}
		}
		
		if (!internalUnderstandsRequest((DropObjectsRequest) request)) {
			return null;
		}
		
		final List objects = _requestObjects.get(request);
		if (!objects.isEmpty() && objects.get(0) instanceof EObject && 
				!(objects.get(0) instanceof EAnnotation)) {
			// all of a sudden we are dropping objects
			// we drop every object on the parent.
			// we only drop vertices on Graphs and Graphs on BpmnDiagram.
			final EObject parent = ((IGraphicalEditPart) getHost()).
				resolveSemanticElement();
			
			AddObjectsCommand command = 
				new AddObjectsCommand(parent) {

						@Override
						protected CommandResult doExecuteWithResult(
								IProgressMonitor monitor, IAdaptable info)
								throws ExecutionException {
							for (Object toDrop : objects) {
								if (parent instanceof BpmnDiagram) {
									if (toDrop instanceof Pool) {
										((Pool) toDrop).
											setBpmnDiagram((BpmnDiagram) parent);
									}
									if (toDrop instanceof MessagingEdge) {
										((MessagingEdge) toDrop).
										setBpmnDiagram((BpmnDiagram) parent);
									}
								}
								if (parent instanceof Graph) {
									if (toDrop instanceof Vertex) {
										((Vertex) toDrop).
											setGraph((Graph) parent);
									} else if (toDrop instanceof SequenceEdge) {
										((SequenceEdge) toDrop).
											setGraph((Graph) parent);
									}
								} 
							}
							((IGraphicalEditPart) getHost()).refresh();
							return CommandResult.newOKCommandResult();
						}};
			return new ICommandProxy(command);
		}
		// we don't drop objects. We drop annotations.
		List<EAnnotation> annotations = objects;
		if (getHost().getModel() instanceof View &&
				((View)getHost().getModel()).getElement()
				instanceof EModelElement) {
			EModelElement modelElement =
				(EModelElement) ((View)getHost().getModel()).getElement();

			final ICommand gmfco = new AddAnnotationCommand(
					modelElement,annotations);
			
			return new ICommandProxy(gmfco);
		} else {
			// the view is associated with something that is not an EModelElement.
			// does nothing.
		}
		return delegateGetCommand(request);
	}

	/**
	 * Command to wrap the EMF command, with redo and undo ability.
	 * @author <a href="mailto:atoulme@intalio.com">Antoine Toulm</a>
	 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
	 */
	private class AddAnnotationCommand extends AbstractTransactionalCommand {

		private EModelElement element;
		private List<EAnnotation> annotationsToAdd;
		private CompoundCommand emfCommand;
		
		public AddAnnotationCommand(
				EModelElement elt,List<EAnnotation> annotationsToAdd) {
			super((TransactionalEditingDomain) AdapterFactoryEditingDomain.
					getEditingDomainFor(elt),
					"DnD",
					getWorkspaceFiles(elt));
			element = elt;
			this.annotationsToAdd = annotationsToAdd;
			initCommand();
		}
		
		private void initCommand() {
			emfCommand = new CompoundCommand();
			EditingDomain domain = AdapterFactoryEditingDomain.
			getEditingDomainFor(element);
			for (EAnnotation annotation : annotationsToAdd) {
				IGenericEAnnotationDndHandler handler =
					BpmnDiagramEditorPlugin.getInstance().
					getEAnnotationDragAndDropHandler(
							((EAnnotation) annotation).getSource());
				if (handler == null) {
					throw new IllegalArgumentException(
							"No extension point is defined for the " +
							"annotation source " + 
							((EAnnotation) annotation).getSource());
				}
				if (handler.doPerformDrop(
								emfCommand,(EAnnotation) annotation, element)) {

					emfCommand.append(
							SetCommand.create(domain, annotation, 
									EcorePackage.
									eINSTANCE.getEAnnotation_EModelElement(),
									element));
				}
			}
			
			
		}
		protected CommandResult doExecuteWithResult(
				IProgressMonitor monitor, IAdaptable info)
		throws ExecutionException {

			emfCommand.execute();
			if (_balloon != null && _balloon.isShowing()) {
				_balloon.hide();
			}
			return CommandResult.newOKCommandResult();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand#doUndo(org.eclipse.core.runtime.IProgressMonitor, org.eclipse.core.runtime.IAdaptable)
		 */
		@Override
		protected IStatus doUndo(IProgressMonitor monitor, IAdaptable info) 
		throws ExecutionException {
			emfCommand.undo();
			return super.doUndo(monitor, info);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand#doRedo(org.eclipse.core.runtime.IProgressMonitor, org.eclipse.core.runtime.IAdaptable)
		 */
		@Override
		protected IStatus doRedo(IProgressMonitor arg0, IAdaptable arg1) 
		throws ExecutionException {
			emfCommand.redo();
			return super.doRedo(arg0, arg1);
		}
	}
	
	private abstract class AddObjectsCommand 
		extends AbstractTransactionalCommand {

		public AddObjectsCommand(EObject element) {
			super((TransactionalEditingDomain) AdapterFactoryEditingDomain.
					getEditingDomainFor(element), "Dropped objects", 
					getWorkspaceFiles(element));
		}

		
		
	}
}
