/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.common.annotations.ext;

import java.lang.annotation.Annotation;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MemberValuePair;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.annotations.ext.IAnnotationInitializer;

public class AnnotationSupportProxy {

	private static final String EXT_ATT_SC_ANN_CLASS_ONLY = "classOnly";
	private static final String EXT_ATT_SC_ANN_INTERFACE_ONLY = "interfaceOnly";
	private static final String EXT_ATT_SC_ANN_NATURE = "nature";
    private static final String EXT_ATT_SC_ANN_INIT_CLASS = "initializerClass";
    private static final String EXT_ATT_SC_ANN_CLASS = "annotationClass";
    private static final String EXT_ELT_SC_ANN = "annotation";
    
    private static Map<String, Class<? extends Annotation> > JAXWS_ANN_DECL;
    private static Map<Class <? extends Annotation>, IAnnotationInitializer > JAXWS_ANN_INIT;
    private static Map<String, String> ANN_NATURES;
    private static Map<String, Boolean> ANN_CLASS_ONLY;
    private static Map<String, Boolean> ANN_INTERFACE_ONLY;

    private static final LoggingProxy LOG = LoggingProxy.getlogger(AnnotationSupportProxy.class);
    public static final String EXT_POINT_SC_ANNOTATION_SUPPORT = "org.eclipse.stp.sc.common.AnnotationSupport";

    private AnnotationSupportProxy() {}
    
    public static void loadAnnotationSupportExtensions() {

        String extPointId = EXT_POINT_SC_ANNOTATION_SUPPORT;
        IExtensionRegistry reg = Platform.getExtensionRegistry();
        IConfigurationElement[] extElts = reg.getConfigurationElementsFor(extPointId);
        
        JAXWS_ANN_DECL = new HashMap<String, Class <? extends Annotation>>(extElts.length);
        JAXWS_ANN_INIT = new HashMap<Class <? extends Annotation>, IAnnotationInitializer>(extElts.length);
        ANN_NATURES = new HashMap<String, String>();
        ANN_CLASS_ONLY = new HashMap<String, Boolean>();
        ANN_INTERFACE_ONLY = new HashMap<String, Boolean>();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            if (extElts[extIndex].getName().equals(EXT_ELT_SC_ANN)) {
                String annotationClassName = extElts[extIndex].getAttribute(EXT_ATT_SC_ANN_CLASS);
                LOG.debug("Load Annoation support:" + annotationClassName);
                Class <? extends Annotation> annotationClass = null;
                try {
                    // NOTE : the next line was commented and broken down in 2 so as to avoid a compilation
                    //        warning (doesn't change much in the end but it looks better in my IDE ;) ) 
                    //annotationClass = (Class <? extends Annotation>)Class.forName(annotationClassName);
                    Class tmpUntypedClass = Class.forName(annotationClassName);
                    annotationClass = (Class <? extends Annotation>)tmpUntypedClass;
                    
                    // validation: the next line will thraugh ClassCastException if the class specifid in the
                    // extension isn't an annotation
                    annotationClass.asSubclass(Annotation.class);
    
                    JAXWS_ANN_DECL.put(annotationClass.getSimpleName(), annotationClass);
                    LOG.debug("annotation added: " + annotationClassName);
                    
                } catch (ClassNotFoundException e) {
                    // invalid extension definition (not a class name)... ignore...
                    // TODO : could it be that the necessary libs aren't loaded? there may be a need to inform the user... ??? ... 
                    StringBuffer msg = new StringBuffer("Invalid ")
                                            .append(EXT_POINT_SC_ANNOTATION_SUPPORT)
                                            .append(" extension definition: '")
                                            .append(annotationClassName)
                                            .append("' is not a valid class name");
                    LOG.error(msg.toString() , e);
                } catch (ClassCastException e) {
                    // invalid extension definition (not an annotation class)... ignore...
                    StringBuffer msg = new StringBuffer("Invalid ")
                                            .append(EXT_POINT_SC_ANNOTATION_SUPPORT)
                                            .append(" extension definition: the class '")
                                            .append(annotationClassName)
                                            .append("' is not a <Annotation> class");
                    LOG.debug(msg.toString(), e);
                }
                
                String intializerClassName = extElts[extIndex].getAttribute(EXT_ATT_SC_ANN_INIT_CLASS);
                try {
                    // no point of looking for the initializer if the annotation class wasn't resolved in 
                    // the 1st place
                    if (annotationClass != null && intializerClassName != null) {

                        IAnnotationInitializer initializerObj = (IAnnotationInitializer) extElts[extIndex]
                            .createExecutableExtension(EXT_ATT_SC_ANN_INIT_CLASS);
                        JAXWS_ANN_INIT.put(annotationClass, initializerObj);
                        LOG.debug("annotation initializer added: "
                                  + initializerObj.getClass().getCanonicalName());
                    }                    
                    
                } catch (CoreException e) {
                    // invalid extension definition (not a class name)... ignore...
                    StringBuffer msg = new StringBuffer("Invalid ")
                                            .append(EXT_POINT_SC_ANNOTATION_SUPPORT)
                                            .append(" extension definition: '")
                                            .append(intializerClassName)
                                            .append("' is not a valid class name");
                    LOG.error(msg.toString() , e);
                } catch (ClassCastException e) {
                    // invalid extension definition (not an IAnnotationInitializer class)... ignore...
                    StringBuffer msg = new StringBuffer("Invalid ")
                                            .append(EXT_POINT_SC_ANNOTATION_SUPPORT)
                                            .append(" extension definition: the class '")
                                            .append(intializerClassName)
                                            .append("' is not a <IAnnotationInitializer> class");
                    LOG.debug(msg.toString(), e);
                }
                
                //process nature
                String natureID = extElts[extIndex].getAttribute(EXT_ATT_SC_ANN_NATURE);
                if (natureID != null) {
                    ANN_NATURES.put(annotationClassName, natureID);
                }
                
                //process class only
                Boolean classOnly = new Boolean(extElts[extIndex].getAttribute(EXT_ATT_SC_ANN_CLASS_ONLY));
                ANN_CLASS_ONLY.put(annotationClassName, classOnly);
                
                //process interface only
                Boolean interfaceOnly = new Boolean(extElts[extIndex].getAttribute(EXT_ATT_SC_ANN_INTERFACE_ONLY));
                ANN_INTERFACE_ONLY.put(annotationClassName, interfaceOnly);
            }
        }
    }
    
    public static boolean verifyNature(String annClassName, String[] natures) {
    	String definedNature = ANN_NATURES.get(annClassName);
    	if (definedNature == null) {
    		return true;
    	}
    	for (String nature : natures) {
    		if (nature.equals(definedNature)) {
    			return true;
    		}
    	}
    	return false;
    }

    public static boolean isClassOnly(String annotationName) {
        if (JAXWS_ANN_DECL == null || JAXWS_ANN_DECL.size() == 0) {
            loadAnnotationSupportExtensions();
        }
        return ANN_CLASS_ONLY.get(annotationName).booleanValue();
    }

    public static boolean isInterfaceOnly(String annotationName) {
        if (JAXWS_ANN_DECL == null || JAXWS_ANN_DECL.size() == 0) {
            loadAnnotationSupportExtensions();
        }
        return ANN_INTERFACE_ONLY.get(annotationName).booleanValue();
    }

    public static Map<String, Class <? extends Annotation>> getAllAvailableAnnotations() {
        if (JAXWS_ANN_DECL == null || JAXWS_ANN_DECL.size() == 0) {
            loadAnnotationSupportExtensions();
        }
        return JAXWS_ANN_DECL;
    }

    public static Class <? extends Annotation> getAvailableAnnotation(String annClassSimpleName) {
        return getAllAvailableAnnotations().get(annClassSimpleName);
    }

    public static Class <? extends Annotation> getAvailableAnnotation(Class annClass) {
        return getAllAvailableAnnotations().get(annClass.getSimpleName());
    }

    public static Map<Class <? extends Annotation>, IAnnotationInitializer> getAllAvailableInitializers() {
        if (JAXWS_ANN_INIT == null || JAXWS_ANN_INIT.size() == 0) {
            loadAnnotationSupportExtensions();
        }
        return JAXWS_ANN_INIT;
    }

    public static IAnnotationInitializer getAvailableAnnotationInitializer(String annClassSimpleName) {
        Class<? extends Annotation> annClass = getAllAvailableAnnotations().get(annClassSimpleName);
        return annClass == null ? null : getAllAvailableInitializers().get(annClass);
    }

    public static IAnnotationInitializer getAvailableAnnotationInitializer(Class annClass) {
        return getAllAvailableInitializers().get(annClass);
    }

    public static List<MemberValuePair> getDefaultAttributes(Class<? extends Annotation> annotationClass,
                                                             CompilationUnit astRoot,
                                                             IMember jdtMember,
                                                             SingleVariableDeclaration jdtMemberParam) {
        
        IAnnotationInitializer init = getAvailableAnnotationInitializer(annotationClass);
        if (init != null) {
            return init.getDefaultAttributes(annotationClass, astRoot, jdtMember, jdtMemberParam);
        }
        return null;
    }


}
