/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.common.logging;

import org.eclipse.core.runtime.Platform;


/**
 * utility to handle development logging. 
 * This simple a <code>LogWritter</code> with added support for:
 * <br> - the eclipse tracing framework (the default tracing switch only include "debug", "info" and "error"
 * but custom switches can be added via the 2nd param of the factory methods) 
 * <br> - delegating the actual logging to any number writers (log4j, java.util.logging, eclipse logging,...)
 * via the registration mechanism.   
 */
public final class LoggingProxy implements ILogWriter {

    public static final String LOG_LEVEL_DEBUG = "org.eclipse.stp.common/debug";
    public static final String LOG_LEVEL_INFO = "org.eclipse.stp.common/info";
    public static final String LOG_LEVEL_ERROR = "org.eclipse.stp.common/error";

    private static final String OPTION_ENABLED = "true";

    private String[] tracingOptions;
    private ILogWriter writer;

    
    private LoggingProxy(Class aLogRoot, String[] someTracingOptions) {
        tracingOptions = someTracingOptions;
        writer = new SystemOutWriter(aLogRoot);
    }
    
    public static LoggingProxy getlogger(Class aLogRoot, String[] someTracingOptions) {
        return new LoggingProxy(aLogRoot, someTracingOptions);
    }
    
    public static LoggingProxy getlogger(Class aLogRoot, String aTracingOption) {
        return new LoggingProxy(aLogRoot, new String[]{aTracingOption});
    }

    public static LoggingProxy getlogger(Class aLogRoot) {
        return new LoggingProxy(aLogRoot, null);
    }
    
    private boolean checkLogOptions(String[] extraTracingOptions) {
        
        for (int i = 0; tracingOptions != null && i < tracingOptions.length; i++) {
            if (!OPTION_ENABLED.equals(Platform.getDebugOption(tracingOptions[i]))) {
                return false;
            }
        }

        for (int i = 0; extraTracingOptions != null && i < extraTracingOptions.length; i++) {
            if (!OPTION_ENABLED.equals(Platform.getDebugOption(extraTracingOptions[i]))) {
                return false;
            }
        }

        return true;
    }
    
    /* (non-Javadoc)
     * @see com.iona.stp.common.util.logging.ILogger#debug(java.lang.Object, java.lang.Throwable)
     */
    public void debug(Object message, Throwable t) {
        if (checkLogOptions(new String[] {LOG_LEVEL_DEBUG})) {
            writer.debug(message, t);
        }
    }

    /* (non-Javadoc)
     * @see com.iona.stp.common.util.logging.ILogger#debug(java.lang.Object)
     */
    public void debug(Object message) {
        if (checkLogOptions(new String[] {LOG_LEVEL_DEBUG})) {
            writer.debug(message);
        }
    }

    /* (non-Javadoc)
     * @see com.iona.stp.common.util.logging.ILogger#info(java.lang.Object, java.lang.Throwable)
     */
    public void info(Object message, Throwable t) {
        if (checkLogOptions(new String[] {LOG_LEVEL_INFO})) {
            writer.info(message, t);
        }
    }

    /* (non-Javadoc)
     * @see com.iona.stp.common.util.logging.ILogger#info(java.lang.Object)
     */
    public void info(Object message) {
        if (checkLogOptions(new String[] {LOG_LEVEL_INFO})) {
            writer.info(message);
        }
    }
    
    /* (non-Javadoc)
     * @see com.iona.stp.common.util.logging.ILogger#error(java.lang.Object, java.lang.Throwable)
     */
    public void error(Object message, Throwable t) {
        if (checkLogOptions(new String[] {LOG_LEVEL_ERROR})) {
            writer.error(message, t);
        }
    }

    /* (non-Javadoc)
     * @see com.iona.stp.common.util.logging.ILogger#error(java.lang.Object)
     */
    public void error(Object message) {
        if (checkLogOptions(new String[] {LOG_LEVEL_ERROR})) {
            writer.error(message);
        }
    }
}
