/*******************************************************************************
 * Copyright (c) 2008, 2009 SOPERA GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SOPERA GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.swordfish.tooling.ui.wizards;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.cxf.tools.common.ToolContext;
import org.apache.cxf.tools.wsdlto.WSDLToJava;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swordfish.tooling.ui.Messages;
import org.eclipse.swordfish.tooling.ui.helper.ServiceModel;
import org.eclipse.swordfish.tooling.ui.helper.Wsdl4jModel;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

/**
 * The java code from WSDL generation operation. 
 * @author yyurkin
 */
public class PluginContentWsdlGenerationOperation extends WorkspaceModifyOperation {
	
	private String pathToWSDL;
	private IProject newProject;
	private IFolder srcFolder;
	private boolean isConsumer;
	
	private String serviceName;
	private String serviceUrl;
	private String targetNamespace;
	private String implementorName;

	private List<String> warnings = new ArrayList<String>();

	/**
	 * The constructor.
	 * @param pathToWsdl - full path to WSDL on file system
	 * @param newProject - the <code>IProject</code> instance, pointing to the created project 
	 * @param srcFolder - the <code>IFolder</code> instance, pointing to the sources
	 * @param isConsumer - consumer code generation
	 */
	public PluginContentWsdlGenerationOperation(String pathToWsdl, IProject newProject, IFolder srcFolder, boolean isConsumer) {
		super();
		this.pathToWSDL = pathToWsdl;
		this.newProject = newProject;
		this.srcFolder = srcFolder;
		this.isConsumer = isConsumer;
	}

	@Override
	protected void execute(IProgressMonitor monitor) throws CoreException,
			InvocationTargetException, InterruptedException {
		try {
			String type;
			String wsdlLocation;
			ServiceModel sm = new Wsdl4jModel(pathToWSDL);

			serviceName = sm.getServiceName();
			targetNamespace = sm.getTargetNamespace();
			serviceUrl = sm.getServiceUrl();
			String impl = sm.getImplementorClass();
			implementorName = impl.substring(0, impl.lastIndexOf('.') + 1) + capitalize(sm.getSoapPortType());
			
			if (isConsumer) {
				type = "-client";
				wsdlLocation = new File(pathToWSDL).toURI().toString();
			} else {
				type = "-impl";
				implementorName += "Impl";
				wsdlLocation = "";
			}
			
			if (serviceName == null) {
				throw new Exception(Messages.ERROR_NO_SERVICE_CLASSES);
			}
			
		    WSDLToJava w2j = new WSDLToJava(new String[]{
		    		type,
		    		"-wsdlLocation", wsdlLocation,
		    		"-d", srcFolder.getLocation().toOSString(),
		    		pathToWSDL});
			ToolContext context = new ToolContext();
			w2j.run(context);
 			
		    newProject.refreshLocal(IResource.DEPTH_INFINITE, monitor);

		} catch (CoreException e) {
			throw e;
		} catch (Exception e) {
			throw new InvocationTargetException(e);
		}
	}
	

	private String capitalize(String s) {
		return s.substring(0, 1).toUpperCase() + s.substring(1);
	}
	
	/**
	 * @return the warnings or <code>null</code> if there was no warnings
	 */
	public List<String> getWarnings() {
		return (null == warnings) ? null : Collections.unmodifiableList(warnings);
	}

	/**
	 * @return the serviceUrl
	 */
	public String getServiceUrl() {
		return serviceUrl;
	}

	/**
	 * @return the serviceName
	 */
	public String getServiceName() {
		return serviceName;
	}

	/**
	 * @return the targetNamespace
	 */
	public String getNameSpace() {
		return targetNamespace;
	}

	/**
	 * @return the implementorName
	 */
	public String getImplementorName() {
		return implementorName;
	}
	
}