/*******************************************************************************
 * Copyright (c) 2008, 2009 SOPERA GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SOPERA GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.swordfish.tooling.ui.helper;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swordfish.tooling.ui.Messages;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;

/**
 * Utility class to manage Dialog forms
 * 
 */
public class DialogSupport {
	
	/**
	 * Provides workspace file selection dialog. 
	 * @param shell - the shell hosting the dialog
	 * @param extension - file extension to filter
	 * @return the String containing full path to file selected 
	 */
	public static String selectResourceFromWorkspace(Shell shell,
			final String extension) {
		
		ILabelProvider labelProvider = new LabelProvider() {

		    public Image getImage(final Object element) {
		    	String imageType = ISharedImages.IMG_OBJ_FOLDER;
				if (((IResource) element).getType() == IResource.FILE) {
					imageType = ISharedImages.IMG_OBJ_ELEMENT;
				} 

				return PlatformUI.getWorkbench().getSharedImages().getImage(imageType);
		    }

		    public String getText(final Object element) {
		    	return ((IResource) element).getName();
		    }
		};

		ITreeContentProvider contentProvider = new ITreeContentProvider() {
			/**
			 * {@inheritDoc}
			 */
		    public Object[] getChildren(final Object parentElement) {
		    	Object[] result = null;
		    	if (!(parentElement instanceof IContainer)) {
		    		return result;
		    	}		    		
		    	try {
		    		List<IResource> arrFolder = new ArrayList<IResource>();
		    		List<IResource> arrFile = new ArrayList<IResource>();
		    		IResource[] res = ((IContainer) parentElement).members();
		    		for (int i = 0; i < res.length; ++i) {
		    			if (res[i].getType() == IResource.FILE) {
		    				if (extension.contains(".") && extension.equalsIgnoreCase(res[i].getName())) {
		    					arrFile.add(res[i]);
		    				} else	if (res[i].getName().endsWith("." + extension)) {
		    					arrFile.add(res[i]);
		    				}
		    			} else {
		    				arrFolder.add(res[i]);
		    			}
		    		}
		    			
		    		List<IResource> arr = new ArrayList<IResource>();
		    		arr.addAll(arrFolder);
		    		arr.addAll(arrFile);
					result = arr.toArray();
		    	} catch (CoreException e) {
						// TODO: handle exception
				}
		    	
		    	return result;
		    }
		    
			/**
			 * {@inheritDoc}
			 */
		    public Object getParent(Object element) {
		    	return ((IResource) element).getParent();
		    }
		    
			/**
			 * {@inheritDoc}
			 */
		    public boolean hasChildren(Object element) {
				if (((IResource) element).getType() == IResource.FILE) {
					return false;
				}
				return true;
		    }
		    
			/**
			 * {@inheritDoc}
			 */
		    public Object[] getElements(Object inputElement) {
		    	return getChildren(inputElement);
		    }
		    
			/**
			 * {@inheritDoc}
			 */
		    public void dispose() { }
		    
			/**
			 * {@inheritDoc}
			 */
		    public void inputChanged(Viewer viewer, Object oldInput, Object newInput) { }
		};
		
		ElementTreeSelectionDialog dialog =
			new ElementTreeSelectionDialog(shell, labelProvider, contentProvider);
		dialog.setTitle(Messages.UI_TITLE_SELECT_INPUT_RESOURCE);
		dialog.setAllowMultiple(false);
		dialog.setDoubleClickSelects(true);
		dialog.setInput(ResourcesPlugin.getWorkspace().getRoot());
		if (dialog.open() == Window.OK) {
			IResource sel = (IResource) dialog.getFirstResult();
			if (sel.getType() == IResource.FILE) {
				return sel.getLocation().toOSString();
			}
		}

		return null;
	}

	/**
	 * Prepares the file selection dialog
	 * @param shell - the shell hosting the dialog
	 * @return prepared FileDialog object
	 */
	public static FileDialog getFileSelectionDialog(final Shell shell) {
		FileDialog fileDialog = new FileDialog(shell, SWT.PRIMARY_MODAL | SWT.OPEN);
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		String strDefaultProject = (projects.length > 0) ? projects[0].getFullPath().toOSString() : "";
		int index = strDefaultProject.indexOf("/"); //$NON-NLS-1$
		if (index == -1) {
			index = strDefaultProject.indexOf("\\"); //$NON-NLS-1$
		}
		if (index != -1) {
			strDefaultProject = strDefaultProject.substring(index + 1);
		}
		fileDialog.setFilterPath(strDefaultProject);
		return fileDialog;
	}

	/**
	 * Provides file open dialog 
	 * @param shell - the shell hosting the dialog
	 * @param fileFilter - fileFilter to filter
	 * @return the String containing full path to file selected 
	 */
	public static String openFileSelectionDialog(Shell shell, String fileFilter) {		
		FileDialog fileDialog = getFileSelectionDialog(shell);
		if (fileFilter.contains(".")) {
			fileDialog.setFilterExtensions(new String[] { fileFilter });
			fileDialog.setFilterNames(new String[] {fileFilter});
		} else {
			fileDialog.setFilterExtensions(new String[] { "*." + fileFilter });
			fileDialog.setFilterNames(new String[] {"*." + fileFilter});
		}
		
		fileDialog.setText(Messages.UI_TITLE_SELECT_INPUT_FILE);
		return fileDialog.open();
	}

}
