/*******************************************************************************
 * Copyright (c) 2009 SOPERA GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SOPERA GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.swordfish.tooling.ui.helper;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * A simple processor that replaces marked place holders in a template by values
 * Place holders consist of a prefix, the place holder name and a suffix.
 */
public class SimpleTemplateProcessor {
	private static final String DEFAULT_MARKER = "$";
	private String prefix;
	private String suffix;
	private String template;
	private Map<String, String> replacements;


	/**
	 * Construct a processor with empty replacement map
	 * @param template - the template to be processed.
	 */
	public SimpleTemplateProcessor(String template) {
		this(template, new HashMap<String, String>(), DEFAULT_MARKER, DEFAULT_MARKER);
	}


	/**
	 * Construct a processor
	 * @param template - the template to be processed.
	 * @param replacements - the replacement mapping
	 */
	public SimpleTemplateProcessor(String template, Map<String, String> replacements) {
		this(template, replacements, DEFAULT_MARKER, DEFAULT_MARKER);
	}

	/**
	 * Construct a processor with custom prefix and suffix
	 * @param template - the template to be processed.
	 * @param prefix - the prefix of place holders in the template
	 * @param suffix - the suffix of place holders in the template
	 */
	SimpleTemplateProcessor(String template, String prefix, String suffix) {
		this(template, new HashMap<String, String>(), prefix, suffix);
	}

	/**
	 * Construct a processor with custom prefix and suffix
	 * @param template - the template to be processed.
	 * @param replacements - the replacement mapping
	 * @param prefix - the prefix of place holders in the template
	 * @param suffix - the suffix of place holders in the template
	 */
	SimpleTemplateProcessor(String template, Map<String, String> replacements, String prefix, String suffix) {
		super();
		this.template = template;
		this.prefix = prefix;
		this.suffix = suffix;
		this.replacements = replacements;
	}

	/**
	 * Replace all occurrences of keys in the replacement map by the associated values
	 * @param template - the template text
	 * @param replacements - a map of key - value pairs
	 * @return - the template after processing
	 */
	String replace(String template, Map<String, String> replacements) {
		if (replacements == null) {
			throw new IllegalStateException("Replacements must not be null!");
		}

		String result = template;

		for (Iterator<String> it = replacements.keySet().iterator(); it.hasNext();) {
			String key = it.next();
			String placeHolder = prefix + key + suffix;
			result = result.replace(placeHolder, replacements.get(key));
		}

		return result;
	}

	
	/**
	 * Process template using the replacements set before
	 * @return the template after processing
	 */
	public String process() {
		return replace(template, this.replacements);
	}
	
	/**
	 * Process the template using the replacements
	 * @param replacements - a mapping of place holder names to values
	 * @return the template after processing
	 */
	public String process(Map<String, String> replacements) {		
		this.replacements = replacements;
		return replace(template, this.replacements);
	}

	
	/**
	 * Put a replacement for processing into the replacement mapping.
	 * @param key - the place holder to be found in the template (without prefix & suffix)
	 * @param value - the value that should replace the place holder
	 * @throws IllegalArgumentException if value item is null
	 */
	public void putReplacement(String key, String value) {
		if (value == null) {
			throw new IllegalArgumentException("Value must not be null");
		}
		replacements.put(key, value);
	}
}
