package org.eclipse.swordfish.tooling.ui.helper;

import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import javax.xml.XMLConstants;
import javax.xml.namespace.NamespaceContext;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathException;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.swordfish.tooling.ui.wizards.generators.data.ConsumerProjectInformation;
import org.eclipse.swordfish.tooling.ui.wizards.generators.data.JaxWsClientInformation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

public class ConsumerProjectInformationUtil {
	private static final Log LOG = LogFactory.getLog(ConsumerProjectInformationUtil.class);
	private static final String SPRING_BEANS_TAG_NAME = "spring:beans";
	public static final String JAXWS_CLIENT_TAG_NAME = "jaxws:client";

	public static final String JAX_WS_CONSUMER_XML = "META-INF/spring/jaxws-consumer.xml";
	public static final String METAINF_MF = "META-INF/spring/jaxws-consumer.xml";

	public static final String SPRING_CONFIG_DIR = "META-INF/spring/";

	public static final String XMLNS_PREFIX = "xmlns:";
	public static final String SERVICE_NAMESPACE_ATTRIBUTE = "xmlns:serviceNamespace";

	public static final String SERVICE_NAME_ATTRIBUTE = "serviceName";

	public static List<File> getSpringOSGIFiles(IProject project) {
		File[] files = new File[0];
		final File springDir = FileUtil.toFile(project.getFile(SPRING_CONFIG_DIR));

		if (springDir.exists() && springDir.isDirectory()) {
			files = springDir.listFiles(new FilenameFilter() {
				public boolean accept(File dir, String name) {
					return dir.equals(springDir) && name.endsWith(".xml");
				}
			});
		}
		return Arrays.asList(files);
	}

	public static IFile getProjectManifest(IProject project) {
		return project.getFile("META-INF/MANIFEST.MF");
	}

	public static boolean isJaxWsConsumerProject(IProject project) {

		if (!getProjectManifest(project).exists()) {
			return false;
		}
		try {
			List<File> xmlFiles = getSpringOSGIFiles(project);

			for (File file : xmlFiles) {
				Document doc = getDocument(new FileInputStream(file));

				if (isJaxWsConsumerSpringConfig(doc)) {
					return true;
				}
			}
		} catch (Exception e) {
			LOG.error(e);
		}
		return false;
	}

	public static ConsumerProjectInformation getConsumerProjectInfo(IProject project) {
		try {
			List<File> xmlFiles = getSpringOSGIFiles(project);

			ConsumerProjectInformation projectInfo = new ConsumerProjectInformation();
			projectInfo.setProjectName(project.getName());

			OSGIManifest manifest = new OSGIManifest(
					getProjectManifest(project).getContents());
			projectInfo.setPackagesToImport(manifest.getExports());

			for (File file : xmlFiles) {

				Document doc = getDocument(new FileInputStream(file));
				if (doc != null) {
					List<JaxWsClientInformation> clientInfoList = getJaxWsClientInformation(doc);

					if (clientInfoList.size() > 0) {
						projectInfo.addClientRefList(clientInfoList);
					}
				}

			}

			if (projectInfo.getClientRefs().size() > 0) {
				return projectInfo;
			}
		} catch (Exception e) {
			LOG.error(e);
		}
		return null;
	}

	public static List<JaxWsClientInformation> getJaxWsClientInformation(
			Document doc) throws XPathException {

		List<JaxWsClientInformation> consumerInfoList = new ArrayList<JaxWsClientInformation>();
		if (doc != null) {
			List<Element> jaxWsClientElements = findJaxWsClientElements(doc);

			for (Element jaxWsClient : jaxWsClientElements) {
				JaxWsClientInformation consumerInfo = new JaxWsClientInformation();
				consumerInfo.setJaxWsClientElement(jaxWsClient);

				// looking for class package
				String serviceNameAttribute = jaxWsClient
						.getAttribute(SERVICE_NAME_ATTRIBUTE);
				String namespaceAttributeName = XMLNS_PREFIX
						+ serviceNameAttribute.substring(0,
								serviceNameAttribute.indexOf(':'));

				// info.setPackagesToImport(packagesToImport)
				// now lets extract the service name
				Element springBeans = findSpringBeansElement(doc);
				String serviceNamespace = springBeans
						.getAttribute(namespaceAttributeName);
				consumerInfo.setServiceNamespace(serviceNamespace);

				consumerInfoList.add(consumerInfo);
			}
		}
		return consumerInfoList;
	}

	public static Document getDocument(InputStream inputStream) {

		DocumentBuilderFactory documentFactory = DocumentBuilderFactory
				.newInstance();
		documentFactory.setNamespaceAware(true);
		DocumentBuilder builder;
		try {
			builder = documentFactory.newDocumentBuilder();
			return builder.parse(inputStream);
		} catch (Exception e) {
			LOG.error(e);
		}
		return null;
	}

	public static boolean isJaxWsConsumerSpringConfig(Document doc) {
		boolean result = false;

		try {
			List<Element> clientElements = findJaxWsClientElements(doc);
			result = clientElements.size() > 0;

		} catch (XPathException ex) {
			LOG.error(ex);
		}
		return result;
	}

	public static List<Element> findJaxWsClientElements(Document doc)
			throws XPathException {
		return findElementsUsingXpath(doc, "//" + JAXWS_CLIENT_TAG_NAME);
	}

	public static Element findSpringBeansElement(Document doc)
			throws XPathException {
		List<Element> elements = findElementsUsingXpath(doc, "//"
				+ SPRING_BEANS_TAG_NAME);
		if (elements.size() == 1) {
			return elements.get(0);
		}
		return null;
	}

	private static List<Element> findElementsUsingXpath(Document doc,
			String xpathExpression) throws XPathException {

		List<Element> result = new ArrayList<Element>();

		XPathFactory factory = XPathFactory.newInstance();
		XPath xpath = factory.newXPath();
		xpath.setNamespaceContext(new PersonalNamespaceContext());
		XPathExpression expression;

		expression = xpath.compile(xpathExpression);

		NodeList nodes = (NodeList) expression.evaluate(doc, XPathConstants.NODESET);

		for (int i = 0; i < nodes.getLength(); i++) {
			result.add((Element) nodes.item(i));
		}

		return result;
	}

	private static class PersonalNamespaceContext implements NamespaceContext {

		public String getNamespaceURI(String prefix) {
			if (prefix == null)
				throw new NullPointerException("Null prefix");
			else if ("jaxws".equals(prefix))
				return "http://cxf.apache.org/jaxws";
			else if ("spring".equals(prefix))
				return "http://www.springframework.org/schema/beans";
			else if ("xml".equals(prefix))
				return XMLConstants.XML_NS_URI;

			return XMLConstants.NULL_NS_URI;
		}

		// This method isn't necessary for XPath processing.
		public String getPrefix(String uri) {
			throw new UnsupportedOperationException();
		}

		// This method isn't necessary for XPath processing either.
		@SuppressWarnings("unchecked")
		public Iterator getPrefixes(String uri) {
			throw new UnsupportedOperationException();
		}
	}
}
