/*******************************************************************************
 * Copyright (c) 2009 SOPERA GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SOPERA GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.swordfish.tooling.ui.wizards.generators;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.swordfish.tooling.ui.Logger;
import org.eclipse.swordfish.tooling.ui.helper.TemplateProcessor;
import org.eclipse.swordfish.tooling.ui.wizards.TemplateConstants;

public class CxfProviderSpringEndpointGenerator implements CodeGenerator {
	private static final String TEMPLATE_LOCATION = "/templates/service/META-INF/spring/cxf-endpoint.xml"; //$NON-NLS-1$
	private String name;
	private String targetNameSpace;
	private String implementor;
	private String url;
	private String clientRefs;
	private String springRefs;
	private InputStreamProvider templateProvider;

	/**
	 * Contruct a generator for cxf-endpoint.xml
	 * 
	 * @param serviceName - name of the service
	 * @param targetNamespace - name of the target
	 * @param implementor - fully qualified name of the class, implementing the service interface
	 * @param serviceURL - url, on which the service is accessible
	 * @param clientRefs - a list of jaxws client references (xml tags)
	 * @param springRefs - a list of spring property settings to inject clients into service (xml tags)
	 */
	public CxfProviderSpringEndpointGenerator(String serviceName, String targetNamespace, String implementor,
												String serviceURL, String clientRefs, String springRefs) {
		this.name = serviceName;
		this.targetNameSpace = targetNamespace;
		this.implementor = implementor;
		this.url = serviceURL;
		this.clientRefs = clientRefs;
		this.springRefs = springRefs;
		this.templateProvider = new DefaultTemplateStreamProvider();
	}

	
	/* (non-Javadoc)
	 * @see org.eclipse.swordfish.tooling.ui.wizards.actions.CodeGenerator#generate()
	 */
	public InputStream generate() {
		InputStream template = null;
		InputStream result = null;

		try {
			template = templateProvider.getInputStream(TEMPLATE_LOCATION);
			Map<String, String> replacements = new HashMap<String, String>();
			replacements.put(TemplateConstants.KEY_SERVICE_ID, Character.toLowerCase(name.charAt(0))
					+ name.substring(1));
			replacements.put(TemplateConstants.KEY_CLIENT_REFS, clientRefs);
			replacements.put(TemplateConstants.KEY_CLIENT_SPRING_REFS, springRefs);
			replacements.put(TemplateConstants.KEY_SERVICE_NAME, name);
			replacements.put(TemplateConstants.KEY_SERVICE_TARGET_NAMESPACE, targetNameSpace);

			int index = implementor.lastIndexOf('.');
			replacements.put(TemplateConstants.KEY_SERVICE_PACKAGE, implementor.substring(0, index));
			replacements.put(TemplateConstants.KEY_SERVICE_IMPL, implementor.substring(index + 1));

			replacements.put(TemplateConstants.KEY_SERVICE_URL, url);
			result = new TemplateProcessor(template, replacements).process();

		} catch (IOException ioe) {
			Logger.error("Unable to process template " + TEMPLATE_LOCATION, ioe);

		} finally {
			if (template != null) {
				try {
					template.close();
				} catch (IOException ioe2) {
					// ignore
				}
			}
		}
		return result;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.swordfish.tooling.ui.wizards.generators.CodeGenerator#setTemplateStreamProvider(org.eclipse.swordfish.tooling.ui.wizards.generators.InputStreamProvider)
	 */
	public void setTemplateStreamProvider(InputStreamProvider templateProvider) {
		this.templateProvider = templateProvider;
	}
}
