/*******************************************************************************
 * Copyright (c) 2009 SOPERA GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SOPERA GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.swordfish.tooling.ui.wizards.generators;

import java.io.StringWriter;
import java.util.List;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.swordfish.tooling.ui.helper.ConsumerProjectInformationUtil;
import org.eclipse.swordfish.tooling.ui.wizards.generators.data.ConsumerProjectInformation;
import org.eclipse.swordfish.tooling.ui.wizards.generators.data.JaxWsClientInformation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * Generates a list of CXF client references (consumer referenes used by provider)
 */
public class JaxWsClientReferenceGenerator {
	private static String TAB = "\t";
	private static String NEW_LINE = System.getProperty("line.separator");
	public static String PROPERTY_SEPARATOR =  NEW_LINE + TAB + TAB;
	private TransformerFactory tFactory;
	private Transformer transformer;

	public JaxWsClientReferenceGenerator() {
		tFactory = TransformerFactory.newInstance();
	}

	public String generate(List<ConsumerProjectInformation> infos) {
		StringBuffer sb = new StringBuffer();

		if (infos != null) {
			for (ConsumerProjectInformation info : infos) {
				sb.append(generate(info));
			}
		}

		return sb.toString();
	}

	public String generate(ConsumerProjectInformation info) {
		StringBuffer result = new StringBuffer();

		if (info != null && info.getClientRefs() != null) {
			List<JaxWsClientInformation> refList = info.getClientRefs();

			for (JaxWsClientInformation ref : refList) {
				if (hasGenerationContent(ref)) {
					try {
						String entry = generate(ref.getJaxWsClientElement(), ref.getServiceNamespace());
						result.append(TAB);
						result.append(entry.replace(" ", PROPERTY_SEPARATOR));
						result.append(NEW_LINE);

					} catch (TransformerException e) {
						throw new IllegalArgumentException("JAXWS client info in " + info.getProjectName()
								+ " invalid " + info, e);
					}
				}
			}
		}

		return result.toString();
	}

	private String generate(Element jaxWsClientElement, String serviceNamespace) throws TransformerException {
		createTransformer();
		Node modNode = jaxWsClientElement.cloneNode(true);
		Document doc = modNode.getOwnerDocument();

		Node serviceNs = doc.createAttribute(ConsumerProjectInformationUtil.SERVICE_NAMESPACE_ATTRIBUTE);
		serviceNs.setNodeValue(serviceNamespace);
		modNode.getAttributes().setNamedItem(serviceNs);

		return convertNodeToString(modNode);
	}

	boolean hasGenerationContent(JaxWsClientInformation clientInfo) {
		return (clientInfo != null)
				&& (clientInfo.getJaxWsClientElement() != null)
				&& (ConsumerProjectInformationUtil.JAXWS_CLIENT_TAG_NAME.equals(clientInfo.getJaxWsClientElement()
						.getNodeName())
						&& (clientInfo.getServiceNamespace() != null) && !("".equals(clientInfo.getServiceNamespace())));
	}

	private void createTransformer() throws TransformerConfigurationException {
		if (transformer == null) {
			transformer = tFactory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
		}
	}

	private String convertNodeToString(Node n) throws TransformerException {
		DOMSource source = new DOMSource(n);
		StringWriter sw = new StringWriter();
		StreamResult result = new StreamResult(sw);
		transformer.transform(source, result);
		return sw.toString();
	}
}
