/*******************************************************************************
 * Copyright (c) 2008, 2009 SOPERA GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SOPERA GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.swordfish.tooling.ui.helper;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swordfish.tooling.ui.Activator;
import org.eclipse.ui.ide.undo.CreateFileOperation;

/**
 * Contains utility methods for the file handling
 *
 */
public class FileUtil {
	
	public static void createFileFromTemplate(IProgressMonitor monitor, IProject project,  String fileName, String fileTemplate, Map<String, String> templateSubstituteMapping)
		throws IOException, ExecutionException, CoreException {
		IPath path = project.getFullPath().append(fileName);
		final IFile newFileHandle = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
		if (newFileHandle.exists()) {
			newFileHandle.delete(true, monitor);
		}
		InputStream inputStream = Activator.getDefault().openStream(fileTemplate);
		if (templateSubstituteMapping != null) {
			inputStream = new TemplateProcessor(inputStream, templateSubstituteMapping).process();
		}
		CreateFileOperation createFileOperation = new CreateFileOperation(newFileHandle,
				null, inputStream,
				"File creation: " + fileName);
		createFileOperation.execute(monitor, null);
	}
	public static void createFileFromTemplate(IProgressMonitor monitor, IProject project,  String fileName, InputStream fileTemplate, Map<String, String> templateSubstituteMapping)
		throws IOException, ExecutionException, CoreException {
		IPath path = project.getFullPath().append(fileName);
		final IFile newFileHandle = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
		if (newFileHandle.exists()) {
			newFileHandle.delete(true, monitor);
		}
		InputStream inputStream = fileTemplate;
		if (templateSubstituteMapping != null) {
			inputStream = new TemplateProcessor(inputStream, templateSubstituteMapping).process();
		}
		CreateFileOperation createFileOperation = new CreateFileOperation(newFileHandle,
				null, inputStream,
				"File creation: " + fileName);
		createFileOperation.execute(monitor, null);
}
	public static File toFile(IResource iResource) {
		Assert.isNotNull(iResource);
		return new File(iResource.getLocationURI());
	}
	/**
	 * Copies specified file to the target directory
	 * @param fileToCopy  - copied file
	 * @param targetDir - the directory, which the file will be copied to
	 */
	public static void copyFile(File fileToCopy , File targetDir) {
		if (!fileToCopy.exists()) {
            throw new IllegalArgumentException();
        }
		if (!targetDir.isDirectory()) {
			throw new IllegalArgumentException();
		}
		copyDirectory(fileToCopy, new File(targetDir, fileToCopy.getName()), null);
	}
	
	/**
	 * Copies the directory from the source location to the target one
	 * if fileExtensions is null all subsequent files will be copied 
	 * @param sourceLocation - source directory
	 * @param targetLocation - target directory
	 * @param fileExtensions the list of the file name filters, that specify which files to copy  
	 */
	public static void copyDirectory(File sourceLocation, File targetLocation, String[] fileExtensions)  {
        try {
        if (sourceLocation.isDirectory()) {
            if (!targetLocation.exists()) {
                org.eclipse.core.runtime.Assert.isTrue(targetLocation.mkdir());
            }
            
            String[] children = sourceLocation.list();
            for (String child : children) {
                
            	boolean eligibleByExtension = false;
                if (fileExtensions != null) {
                	for (String fileExtension : fileExtensions) {
                		if (child.toLowerCase().endsWith(fileExtension.toLowerCase())) {
                			eligibleByExtension = true;
                			break;
                		}
                	}
                } else {
                	eligibleByExtension = true;
                }
                
            	if (eligibleByExtension) {
            		copyDirectory(new File(sourceLocation, child),
                        new File(targetLocation, child), fileExtensions);
            	}
            }
        } else {
        	InputStream in = null;
            OutputStream out = null;
            try {
            	in = new FileInputStream(sourceLocation);
            	out = new FileOutputStream(targetLocation);
            
	            // Copy the bits from instream to outstream
	            byte[] buf = new byte[1024];
	            int len;
	            while ((len = in.read(buf)) > 0) {
	                out.write(buf, 0, len);
	            }
            } finally {
            	if (in != null) {
            		in.close();
            	}
            	if (out != null) {
            		out.close();
            	}
            }
        }
        } catch (Exception ex) {
        	throw new IllegalStateException(ex.getMessage(), ex);
        }
    }
}
