/*******************************************************************************
 * Copyright (c) 2009 SOPERA GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SOPERA GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.swordfish.tooling.ui.wizards.generators;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swordfish.tooling.ui.helper.ConsumerProjectInformationUtil;
import org.eclipse.swordfish.tooling.ui.helper.SimpleTemplateProcessor;
import org.eclipse.swordfish.tooling.ui.wizards.generators.data.ConsumerProjectInformation;
import org.eclipse.swordfish.tooling.ui.wizards.generators.data.JaxWsClientInformation;

/**
 * Generates code pieces needed to manipulate the provider service
 * implementation to deal with consumer client injection
 */
public class JaxWsClientJavaCodeGenerator {
	private static String CLIENT_TAG = ConsumerProjectInformationUtil.JAXWS_CLIENT_TAG_NAME;
	private static String CLIENT_TYPE = "type";
	private static String PROPERTY = "property";
	private static String MEMBER_NAME = "member";
	private static String CLIENT_PACKAGE = "package";
	private SimpleTemplateProcessor importProc;
	private SimpleTemplateProcessor fieldProc;
	private SimpleTemplateProcessor getProc;
	private SimpleTemplateProcessor setProc;

	private List<String> imports;
	private List<String> fields;
	private List<String> accessors;

	public JaxWsClientJavaCodeGenerator() {
		super();
		importProc = new SimpleTemplateProcessor("$package$.$type$");
		fieldProc = new SimpleTemplateProcessor("private $type$ $member$;");
		getProc = new SimpleTemplateProcessor("public $type$ get$property$() {\n\treturn $member$;\n}");
		setProc = new SimpleTemplateProcessor("public void set$property$($type$ newVal) {\n\t$member$ = newVal;\n}");
	}

	/**
	 * Generates imports, fields and field accessor methods for a list of
	 * consumer info. Use the getters to retrieve the generated code pieces.
	 * 
	 * @param consumerInfo
	 *            - a list of ConsumerProjectInformation instances
	 */
	public void generate(List<ConsumerProjectInformation> consumerInfo) {
		imports = new ArrayList<String>();
		fields = new ArrayList<String>();
		accessors = new ArrayList<String>();

		if (consumerInfo != null) {
			for (ConsumerProjectInformation ci : consumerInfo) {
				for (JaxWsClientInformation clientInfo : ci.getClientRefs()) {
					if (hasGenerationContent(clientInfo)) {
						imports.add(generateImport(clientInfo));
						fields.add(generateField(clientInfo));
						accessors.add(generateGetter(clientInfo));
						accessors.add(generateSetter(clientInfo));
					}
				}
			}
		}
	}

	String generateImport(JaxWsClientInformation clientInfo) {
		importProc.putReplacement(CLIENT_PACKAGE, clientInfo.getServiceClassPackage());
		importProc.putReplacement(CLIENT_TYPE, clientInfo.getServiceClass());
		return importProc.process();
	}

	String generateField(JaxWsClientInformation clientInfo) {
		fieldProc.putReplacement(CLIENT_TYPE, clientInfo.getServiceClass());
		fieldProc.putReplacement(MEMBER_NAME, deCaptitalize(clientInfo.getClientID()));
		return fieldProc.process();
	}

	String generateGetter(JaxWsClientInformation clientInfo) {
		getProc.putReplacement(CLIENT_TYPE, clientInfo.getServiceClass());
		getProc.putReplacement(PROPERTY, clientInfo.getClientID());
		getProc.putReplacement(MEMBER_NAME, deCaptitalize(clientInfo.getClientID()));
		return getProc.process();
	}

	String generateSetter(JaxWsClientInformation clientInfo) {
		setProc.putReplacement(CLIENT_TYPE, clientInfo.getServiceClass());
		setProc.putReplacement(PROPERTY, clientInfo.getClientID());
		setProc.putReplacement(MEMBER_NAME, deCaptitalize(clientInfo.getClientID()));
		return setProc.process();
	}

	/**
	 * @return list of imports needed for the client references
	 */
	public List<String> getImports() {
		return imports;
	}

	/**
	 * @return list of field definitions needed for the client references
	 */
	public List<String> getFields() {
		return fields;
	}

	/**
	 * @return list of getter and setter methods to access the client references
	 */
	public List<String> getAccessorMethods() {
		return accessors;
	}

	private String deCaptitalize(String str) {
		return str.substring(0, 1).toLowerCase() + str.substring(1);
	}

	boolean hasGenerationContent(JaxWsClientInformation clientInfo) {
		return (clientInfo != null)
				&& (clientInfo.getJaxWsClientElement() != null)
				&& (CLIENT_TAG.equals(clientInfo.getJaxWsClientElement().getNodeName())
				&& (clientInfo.getServiceClassAttribute() != null)
				&& (!"".equals(clientInfo.getServiceClass())) && (clientInfo.getClientID()) != null)
				&& (!"".equals(clientInfo.getClientID()));
	}
}
