/*******************************************************************************
 * Copyright (c) 2011, 2012 Wind River Systems, Inc. and others. All rights reserved.
 * This program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Wind River Systems - initial API and implementation
 *******************************************************************************/
package org.eclipse.tcf.te.tcf.locator.listener;

import org.eclipse.core.runtime.Assert;
import org.eclipse.tcf.protocol.IChannel;
import org.eclipse.tcf.protocol.IPeer;
import org.eclipse.tcf.protocol.Protocol;
import org.eclipse.tcf.te.tcf.core.listeners.interfaces.IChannelStateChangeListener;
import org.eclipse.tcf.te.tcf.locator.activator.CoreBundleActivator;
import org.eclipse.tcf.te.tcf.locator.interfaces.ITracing;
import org.eclipse.tcf.te.tcf.locator.interfaces.nodes.ILocatorModel;
import org.eclipse.tcf.te.tcf.locator.interfaces.nodes.IPeerModel;
import org.eclipse.tcf.te.tcf.locator.interfaces.nodes.IPeerModelProperties;
import org.eclipse.tcf.te.tcf.locator.interfaces.services.ILocatorModelLookupService;


/**
 * Channel state change listener implementation.
 */
public class ChannelStateChangeListener implements IChannelStateChangeListener {
	// Reference to the parent model
	private final ILocatorModel model;

	/**
	 * Constructor.
	 *
	 * @param model The parent locator model. Must not be <code>null</code>.
	 */
	public ChannelStateChangeListener(ILocatorModel model) {
		Assert.isNotNull(model);
		this.model = model;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tcf.te.tcf.tcf.core.interfaces.listeners.IChannelStateChangeListener#stateChanged(org.eclipse.tcf.protocol.IChannel, int)
	 */
	@Override
	public void stateChanged(IChannel channel, int state) {
		Assert.isNotNull(channel);
		Assert.isTrue(Protocol.isDispatchThread());

		if (CoreBundleActivator.getTraceHandler().isSlotEnabled(0, ITracing.ID_TRACE_CHANNEL_STATE_CHANGE_LISTENER)) {
			CoreBundleActivator.getTraceHandler().trace("ChannelStateChangeListener.stateChanged( " + channel + ", " + (state == IChannel.STATE_OPEN ? "OPEN" : "CLOSED") + " )", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
														ITracing.ID_TRACE_CHANNEL_STATE_CHANGE_LISTENER, this);
		}

		switch (state) {
			case IChannel.STATE_OPEN:
				IPeer peer = channel.getRemotePeer();
				if (peer != null && peer.getID() != null) {
					// Find the corresponding model node
					IPeerModel node = model.getService(ILocatorModelLookupService.class).lkupPeerModelById(peer.getID());
					if (node != null) {
						// Increase the channel reference counter by 1
						int counter = node.getIntProperty(IPeerModelProperties.PROP_CHANNEL_REF_COUNTER);
						if (counter < 0) counter = 0;
						counter++;
						node.setProperty(IPeerModelProperties.PROP_CHANNEL_REF_COUNTER, counter);
						if (counter > 0) node.setProperty(IPeerModelProperties.PROP_STATE, IPeerModelProperties.STATE_CONNECTED);
					}
				}
				break;
			case IChannel.STATE_CLOSED:
				peer = channel.getRemotePeer();
				if (peer != null && peer.getID() != null) {
					// Find the corresponding model node
					IPeerModel node = model.getService(ILocatorModelLookupService.class).lkupPeerModelById(peer.getID());
					if (node != null) {
						// Decrease the channel reference counter by 1
						int counter = node.getIntProperty(IPeerModelProperties.PROP_CHANNEL_REF_COUNTER);
						counter--;
						if (counter < 0) counter = 0;
						node.setProperty(IPeerModelProperties.PROP_CHANNEL_REF_COUNTER, counter);
						if (counter == 0 && node.isProperty(IPeerModelProperties.PROP_STATE, IPeerModelProperties.STATE_CONNECTED)) {
							node.setProperty(IPeerModelProperties.PROP_STATE, IPeerModelProperties.STATE_REACHABLE);
						}
					}
				}
				break;
		}
	}
}
