/*******************************************************************************
 * Copyright (c) 2011, 2012 Wind River Systems, Inc. and others. All rights reserved.
 * This program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Wind River Systems - initial API and implementation
 *******************************************************************************/
package org.eclipse.tcf.te.tcf.ui.editor.sections;

import java.util.concurrent.atomic.AtomicBoolean;

import org.eclipse.core.runtime.Assert;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tcf.protocol.Protocol;
import org.eclipse.tcf.te.runtime.interfaces.properties.IPropertiesContainer;
import org.eclipse.tcf.te.runtime.properties.PropertiesContainer;
import org.eclipse.tcf.te.tcf.locator.interfaces.nodes.IPeerModel;
import org.eclipse.tcf.te.tcf.locator.interfaces.nodes.IPeerModelProperties;
import org.eclipse.tcf.te.tcf.locator.interfaces.services.ILocatorModelPeerNodeQueryService;
import org.eclipse.tcf.te.tcf.ui.nls.Messages;
import org.eclipse.tcf.te.ui.forms.parts.AbstractSection;
import org.eclipse.tcf.te.ui.swt.DisplayUtil;
import org.eclipse.tcf.te.ui.swt.SWTControlUtil;
import org.eclipse.tcf.te.ui.views.editor.pages.AbstractEditorPage;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;

/**
 * Peer services section implementation.
 */
public class ServicesSection extends AbstractSection {
	// The section sub controls
	/* default */ Text local;
	/* default */ Text remote;

	// Reference to the original data object
	/* default */ IPeerModel od;
	// Reference to a copy of the original data
	/* default */ final IPropertiesContainer odc = new PropertiesContainer();

	/**
	 * Constructor.
	 *
	 * @param form The parent managed form. Must not be <code>null</code>.
	 * @param parent The parent composite. Must not be <code>null</code>.
	 */
	public ServicesSection(IManagedForm form, Composite parent) {
		super(form, parent, Section.DESCRIPTION | ExpandableComposite.TWISTIE);
		createClient(getSection(), form.getToolkit());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tcf.te.ui.forms.parts.AbstractSection#createClient(org.eclipse.ui.forms.widgets.Section, org.eclipse.ui.forms.widgets.FormToolkit)
	 */
	@Override
	protected void createClient(Section section, FormToolkit toolkit) {
		Assert.isNotNull(section);
		Assert.isNotNull(toolkit);

		// Configure the section
		section.setText(Messages.ServicesSection_title);
		section.setDescription(Messages.ServicesSection_description);

		if (section.getParent().getLayout() instanceof GridLayout) {
			section.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		}

		// Create the section client
		Composite client = createClientContainer(section, 1, toolkit);
		Assert.isNotNull(client);
		section.setClient(client);

		Group group = new Group(client, SWT.NONE);
		group.setText(Messages.ServicesSection_group_local_title);
		group.setLayout(new GridLayout());
		group.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

		local = new Text(group, SWT.READ_ONLY | SWT.WRAP | SWT.MULTI);
		GridData layoutData = new GridData(SWT.FILL, SWT.CENTER, true, false);
		layoutData.widthHint = SWTControlUtil.convertWidthInCharsToPixels(local, 20);
		layoutData.heightHint = SWTControlUtil.convertHeightInCharsToPixels(local, 5);
		local.setLayoutData(layoutData);

		group = new Group(client, SWT.NONE);
		group.setText(Messages.ServicesSection_group_remote_title);
		group.setLayout(new GridLayout());
		group.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

		remote = new Text(group, SWT.READ_ONLY | SWT.WRAP | SWT.MULTI);
		layoutData = new GridData(SWT.FILL, SWT.CENTER, true, false);
		layoutData.widthHint = SWTControlUtil.convertWidthInCharsToPixels(local, 20);
		layoutData.heightHint = SWTControlUtil.convertHeightInCharsToPixels(remote, 5);
		remote.setLayoutData(layoutData);

		// Mark the control update as completed now
		setIsUpdating(false);
	}

	/**
	 * Indicates whether the sections parent page has become the active in the editor.
	 *
	 * @param active <code>True</code> if the parent page should be visible, <code>false</code> otherwise.
	 */
	public void setActive(boolean active) {
		// If the parent page has become the active and it does not contain
		// unsaved data, than fill in the data from the selected node
		if (active) {
			// Leave everything unchanged if the page is in dirty state
			if (getManagedForm().getContainer() instanceof AbstractEditorPage
					&& !((AbstractEditorPage)getManagedForm().getContainer()).isDirty()) {
				Object node = ((AbstractEditorPage)getManagedForm().getContainer()).getEditorInputNode();
				if (node instanceof IPeerModel) {
					setupData((IPeerModel)node);
				}
			}
		}
	}

	// Flag to mark that the services query had been done for the current peer model node
	private boolean servicesQueryTriggered = false;

	/**
	 * Initialize the page widgets based of the data from the given peer node.
	 * <p>
	 * This method may called multiple times during the lifetime of the page and
	 * the given configuration node might be even <code>null</code>.
	 *
	 * @param node The peer node or <code>null</code>.
	 */
	public void setupData(final IPeerModel node) {
		// Reset the services query triggered flag if we setup for a new peer model node
		if (od != node) servicesQueryTriggered = false;

		// Besides the node itself, we need to look at the node data to determine
		// if the widgets needs to be updated. For the comparisation, keep the
		// current properties of the original data copy in a temporary container.
		final IPropertiesContainer previousOdc = new PropertiesContainer();
		previousOdc.setProperties(odc.getProperties());

		// Store a reference to the original data
		od = node;
		// Clean the original data copy
		odc.clearProperties();

		// If no data is available, we are done
		if (node == null) return;

		// Thread access to the model is limited to the executors thread.
		// Copy the data over to the working copy to ease the access.
		Protocol.invokeAndWait(new Runnable() {
			@Override
			public void run() {
				// Copy over the properties
				odc.setProperties(od.getProperties());
			}
		});

		boolean forceQuery = false;

		// If the original data copy does not match the previous original
		// data copy, the services needs to be queried again.
		if (!previousOdc.getProperties().equals(odc.getProperties())) {
			servicesQueryTriggered = false;
			forceQuery = true;
		}

		// Create the UI runnable
		final AtomicBoolean fireRefreshTabs = new AtomicBoolean();
		final Runnable uiRunnable = new Runnable() {

			@Override
			public void run() {
				boolean fireNotification = fireRefreshTabs.get();

				String value = odc.getStringProperty(IPeerModelProperties.PROP_LOCAL_SERVICES);
				fireNotification |= value != null && !value.equals(SWTControlUtil.getText(local));
				SWTControlUtil.setText(local, value != null ? value : ""); //$NON-NLS-1$
				value = odc.getStringProperty(IPeerModelProperties.PROP_REMOTE_SERVICES);
				fireNotification |= value != null && !value.equals(SWTControlUtil.getText(remote));
				SWTControlUtil.setText(remote, value != null ? value : ""); //$NON-NLS-1$

				if (fireNotification) {
					// Fire a change event to trigger the editor refresh
					od.fireChangeEvent("editor.refreshTab", Boolean.FALSE, Boolean.TRUE); //$NON-NLS-1$
				}
			}
		};

		// If not yet triggered or if forced, run the service query
		if (!servicesQueryTriggered) {
			// Mark the services query as triggered
			servicesQueryTriggered = true;

			final boolean finForceQuery = forceQuery;

			Runnable runnable = new Runnable() {

				@Override
				public void run() {
					// Check if we have to run the query at all
					boolean doQuery = finForceQuery ||
										(!node.containsKey(IPeerModelProperties.PROP_REMOTE_SERVICES)
											&& !node.containsKey(IPeerModelProperties.PROP_LOCAL_SERVICES));

					if (doQuery) {
						ILocatorModelPeerNodeQueryService service = node.getModel().getService(ILocatorModelPeerNodeQueryService.class);
						if (service != null) {
							service.queryServicesAsync(node, new ILocatorModelPeerNodeQueryService.DoneQueryServices() {
								@Override
								public void doneQueryServices(Throwable error) {
									// Copy over the service properties
									odc.setProperty(IPeerModelProperties.PROP_REMOTE_SERVICES, node.getProperty(IPeerModelProperties.PROP_REMOTE_SERVICES));
									odc.setProperty(IPeerModelProperties.PROP_LOCAL_SERVICES, node.getProperty(IPeerModelProperties.PROP_LOCAL_SERVICES));

									// Setup the data within the UI controls and fire the change notification
									fireRefreshTabs.set(true);
									DisplayUtil.safeAsyncExec(uiRunnable);
								}
							});
						}
					} else {
						// Copy over the properties
						odc.setProperties(node.getProperties());
						// Setup the data within the UI controls
						DisplayUtil.safeAsyncExec(uiRunnable);
					}
				}
			};

			Protocol.invokeLater(runnable);
		} else {
			// Setup the data within the UI controls
			uiRunnable.run();
		}
	}
}
