/*******************************************************************************
 * Copyright (c) 2011, 2014 Wind River Systems, Inc. and others. All rights reserved.
 * This program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Wind River Systems - initial API and implementation
 * Max Weninger (Wind River) - [361352] [TERMINALS][SSH] Add SSH terminal support
 *******************************************************************************/
package org.eclipse.tcf.te.ui.terminals.internal.dialogs;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogPage;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.TypedEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tcf.te.runtime.interfaces.properties.IPropertiesContainer;
import org.eclipse.tcf.te.runtime.properties.PropertiesContainer;
import org.eclipse.tcf.te.runtime.services.interfaces.constants.ITerminalsConnectorConstants;
import org.eclipse.tcf.te.ui.controls.BaseDialogPageControl;
import org.eclipse.tcf.te.ui.controls.BaseWizardConfigurationPanelControl;
import org.eclipse.tcf.te.ui.controls.interfaces.IWizardConfigurationPanel;
import org.eclipse.tcf.te.ui.interfaces.data.IDataExchangeNode;
import org.eclipse.tcf.te.ui.jface.dialogs.CustomTrayDialog;
import org.eclipse.tcf.te.ui.jface.interfaces.IValidatingContainer;
import org.eclipse.tcf.te.ui.swt.SWTControlUtil;
import org.eclipse.tcf.te.ui.terminals.activator.UIPlugin;
import org.eclipse.tcf.te.ui.terminals.help.IContextHelpIds;
import org.eclipse.tcf.te.ui.terminals.interfaces.IConfigurationPanel;
import org.eclipse.tcf.te.ui.terminals.interfaces.ILauncherDelegate;
import org.eclipse.tcf.te.ui.terminals.interfaces.tracing.ITraceIds;
import org.eclipse.tcf.te.ui.terminals.launcher.LauncherDelegateManager;
import org.eclipse.tcf.te.ui.terminals.nls.Messages;
import org.eclipse.tcf.te.ui.terminals.panels.AbstractConfigurationPanel;
import org.eclipse.ui.forms.widgets.FormToolkit;

/**
 * Launch terminal settings dialog implementation.
 */
public class LaunchTerminalSettingsDialog extends CustomTrayDialog implements IValidatingContainer {
	// The parent selection
	private ISelection selection = null;

	// The subcontrols
	/* default */ Combo terminals;
	/* default */ SettingsPanelControl settings;

	private FormToolkit toolkit = null;

	// Map the label added to the combobox to the corresponding launcher delegate.
	/* default */ final Map<String, ILauncherDelegate> label2delegate = new HashMap<String, ILauncherDelegate>();

	// The data object containing the currently selected settings
	private IPropertiesContainer data = null;

	/**
	 * The control managing the terminal setting panels.
	 */
	protected class SettingsPanelControl extends BaseWizardConfigurationPanelControl implements IValidatingContainer {

		/**
		 * Constructor.
		 *
		 * @param parentPage The parent dialog page this control is embedded in.
		 *                   Might be <code>null</code> if the control is not associated with a page.
		 */
        public SettingsPanelControl(IDialogPage parentPage) {
	        super(parentPage);
	        setPanelIsGroup(true);
        }

        /* (non-Javadoc)
         * @see org.eclipse.tcf.te.ui.controls.BaseWizardConfigurationPanelControl#getGroupLabel()
         */
        @Override
        public String getGroupLabel() {
            return Messages.LaunchTerminalSettingsDialog_group_label;
        }

        /* (non-Javadoc)
         * @see org.eclipse.tcf.te.ui.controls.BaseWizardConfigurationPanelControl#showConfigurationPanel(java.lang.String)
         */
        @Override
        public void showConfigurationPanel(String key) {
        	// Check if we have to create the panel first
    		IWizardConfigurationPanel configPanel = getConfigurationPanel(key);
    		if (configPanel == null) {
           		// Get the corresponding delegate
           		ILauncherDelegate delegate = label2delegate.get(key);
           		Assert.isNotNull(delegate);
           		// Get the wizard configuration panel instance
           		configPanel = delegate.getPanel(this);
            	if (configPanel == null) configPanel = new EmptySettingsPanel(this);
            	// Push the selection to the configuration panel
            	((IConfigurationPanel)configPanel).setSelection(getSelection());
            	// Add it
            	addConfigurationPanel(key, configPanel);
            	// Create the panel controls
            	configPanel.setupPanel(getPanel(), getFormToolkit());
    		}

            super.showConfigurationPanel(key);

            validate();
        }

        /* (non-Javadoc)
         * @see org.eclipse.tcf.te.ui.jface.interfaces.IValidatingContainer#validate()
         */
        @Override
        public void validate() {
        	LaunchTerminalSettingsDialog.this.validate();
        }
	}

	/**
	 * An empty terminal settings panel.
	 */
	protected class EmptySettingsPanel extends AbstractConfigurationPanel {

		/**
	     * Constructor.
	     *
		 * @param parentControl The parent control. Must not be <code>null</code>!
	     */
	    public EmptySettingsPanel(BaseDialogPageControl parentControl) {
		    super(parentControl);
	    }

		/* (non-Javadoc)
	     * @see org.eclipse.tcf.te.ui.controls.interfaces.IWizardConfigurationPanel#setupPanel(org.eclipse.swt.widgets.Composite, org.eclipse.tcf.te.ui.controls.interfaces.FormToolkit)
	     */
	    @SuppressWarnings("synthetic-access")
        @Override
	    public void setupPanel(Composite parent, FormToolkit toolkit) {
	    	Composite panel = new Composite(parent, SWT.NONE);
	    	panel.setLayout(new GridLayout());
	    	panel.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

	    	Label label = new Label(panel, SWT.HORIZONTAL);
	    	GridData layoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
			layoutData.widthHint = convertWidthInCharsToPixels(30);
			layoutData.heightHint = convertHeightInCharsToPixels(5);
			label.setLayoutData(layoutData);

	    	setControl(panel);
	    }

	    /* (non-Javadoc)
	     * @see org.eclipse.tcf.te.ui.controls.interfaces.IWizardConfigurationPanel#dataChanged(org.eclipse.tcf.te.runtime.interfaces.properties.IPropertiesContainer, org.eclipse.swt.events.TypedEvent)
	     */
	    @Override
	    public boolean dataChanged(IPropertiesContainer data, TypedEvent e) {
	        return false;
	    }

		@Override
        protected void saveSettingsForHost(boolean add) {
        }

		@Override
        protected void fillSettingsForHost(String host) {
        }

		@Override
        protected String getHostFromSettings() {
			return null;
        }

		@Override
	    public void doRestoreWidgetValues(IDialogSettings settings, String idPrefix) {
		}

		@Override
	    public void doSaveWidgetValues(IDialogSettings settings, String idPrefix) {
		}
	}

	/**
     * Constructor.
     *
	 * @param shell The parent shell or <code>null</code>.
     */
    public LaunchTerminalSettingsDialog(Shell shell) {
	    super(shell, IContextHelpIds.LAUNCH_TERMINAL_SETTINGS_DIALOG);
    }

    private long start = 0;

	/**
     * Constructor.
     *
	 * @param shell The parent shell or <code>null</code>.
     */
    public LaunchTerminalSettingsDialog(Shell shell, long start) {
	    super(shell, IContextHelpIds.LAUNCH_TERMINAL_SETTINGS_DIALOG);
	    this.start = start;
    }

    /**
     * Sets the parent selection.
     *
     * @param selection The parent selection or <code>null</code>.
     */
    public void setSelection(ISelection selection) {
    	this.selection = selection;
    }

    /**
     * Returns the parent selection.
     *
     * @return The parent selection or <code>null</code>.
     */
    public ISelection getSelection() {
    	return selection;
    }

    /* (non-Javadoc)
     * @see org.eclipse.tcf.te.ui.jface.dialogs.CustomTrayDialog#dispose()
     */
    @Override
    protected void dispose() {
    	if (settings != null) { settings.dispose(); settings = null; }
    	if (toolkit != null) { toolkit.dispose(); toolkit = null; }

        super.dispose();
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.dialogs.Dialog#isResizable()
     */
    @Override
    protected boolean isResizable() {
        return true;
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.dialogs.Dialog#createContents(org.eclipse.swt.widgets.Composite)
     */
    @Override
    protected Control createContents(Composite parent) {
        Control composite = super.createContents(parent);

        // Validate the dialog after having created all the content
        validate();

        return composite;
    }

    /* (non-Javadoc)
     * @see org.eclipse.tcf.te.ui.jface.dialogs.CustomTrayDialog#createDialogAreaContent(org.eclipse.swt.widgets.Composite)
     */
    @Override
    protected void createDialogAreaContent(Composite parent) {
        super.createDialogAreaContent(parent);

        if (UIPlugin.getTraceHandler().isSlotEnabled(0, ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER)) {
			UIPlugin.getTraceHandler().trace("Creating dialog area after " + (System.currentTimeMillis() - start) + " ms.", //$NON-NLS-1$ //$NON-NLS-2$
												ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER, LaunchTerminalSettingsDialog.this);
		}

    	setDialogTitle(Messages.LaunchTerminalSettingsDialog_title);

        Composite panel = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        layout.marginHeight = 0; layout.marginWidth = 0;
        panel.setLayout(layout);
        panel.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, true));

        Label label = new Label(panel, SWT.HORIZONTAL);
        label.setText(Messages.LaunchTerminalSettingsDialog_combo_label);

        terminals = new Combo(panel, SWT.READ_ONLY);
        terminals.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        terminals.addSelectionListener(new SelectionAdapter() {
        	@Override
        	public void widgetSelected(SelectionEvent e) {
        		// Get the old panel
        		IWizardConfigurationPanel oldPanel = settings.getActiveConfigurationPanel();
        		// Extract the current settings in an special properties container
        		IPropertiesContainer data = new PropertiesContainer();
        		if (oldPanel instanceof IDataExchangeNode) ((IDataExchangeNode)oldPanel).extractData(data);
        		// Clean out settings which are never passed between the panels
        		data.setProperty(ITerminalsConnectorConstants.PROP_IP_PORT, null);
        		data.setProperty(ITerminalsConnectorConstants.PROP_TIMEOUT, null);
            	data.setProperty(ITerminalsConnectorConstants.PROP_TERMINAL_CONNECTOR_ID, null);
            	data.setProperty(ITerminalsConnectorConstants.PROP_CONNECTOR_TYPE_ID, null);
        		// Switch to the new panel
        		settings.showConfigurationPanel(terminals.getText());
        		// Get the new panel
        		IWizardConfigurationPanel newPanel = settings.getActiveConfigurationPanel();
        		// Re-setup the relevant data
        		if (newPanel instanceof IDataExchangeNode) ((IDataExchangeNode)newPanel).setupData(data);

        		// resize the dialog if needed to show the complete panel
        		getShell().pack();
        	}
		});

        // fill the combo with content
        fillCombo(terminals);

        // Create the settings panel control
        settings = new SettingsPanelControl(null);

		// Create, initialize and add the first visible panel. All
        // other panels are created on demand only.
        String terminalLabel = SWTControlUtil.getItem(terminals, 0);
        if (terminalLabel != null) {
       		// Get the corresponding delegate
       		ILauncherDelegate delegate = label2delegate.get(terminalLabel);
       		Assert.isNotNull(delegate);
       		// Get the wizard configuration panel instance
       		IConfigurationPanel configPanel = delegate.getPanel(settings);
        	if (configPanel == null) configPanel = new EmptySettingsPanel(settings);
        	// Push the selection to the configuration panel
        	Assert.isNotNull(configPanel);
        	configPanel.setSelection(getSelection());
        	// Add it
        	settings.addConfigurationPanel(terminalLabel, configPanel);
        	// Attach the listener
        }

        // Create the toolkit
        toolkit = new FormToolkit(panel.getDisplay());
		// Setup the panel control
		settings.setupPanel(panel, terminals.getItems(), toolkit);
		GridData layoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		layoutData.horizontalSpan = 2;
		settings.getPanel().setLayoutData(layoutData);

		// Preselect the first terminal launcher
		terminals.select(0);
		settings.showConfigurationPanel(terminals.getText());

		SWTControlUtil.setEnabled(terminals, terminals.getItemCount() > 1);

		restoreWidgetValues();

        applyDialogFont(panel);

		if (UIPlugin.getTraceHandler().isSlotEnabled(0, ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER)) {
			UIPlugin.getTraceHandler().trace("Created dialog area after " + (System.currentTimeMillis() - start) + " ms.", //$NON-NLS-1$ //$NON-NLS-2$
												ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER, LaunchTerminalSettingsDialog.this);
		}
    }

    /**
     * Fill the given combo with content. The content are the terminal
     * launcher delegate labels.
     *
     * @param combo The combo. Must not be <code>null</code>.
     */
    protected void fillCombo(Combo combo) {
    	Assert.isNotNull(combo);

		if (UIPlugin.getTraceHandler().isSlotEnabled(0, ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER)) {
			UIPlugin.getTraceHandler().trace("Filling combo after " + (System.currentTimeMillis() - start) + " ms.", //$NON-NLS-1$ //$NON-NLS-2$
												ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER, LaunchTerminalSettingsDialog.this);
		}

    	List<String> items = new ArrayList<String>();

    	if(selection==null || selection.isEmpty()){
    		if (UIPlugin.getTraceHandler().isSlotEnabled(0, ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER)) {
    			UIPlugin.getTraceHandler().trace("Getting launcher delegates after " + (System.currentTimeMillis() - start) + " ms.", //$NON-NLS-1$ //$NON-NLS-2$
    												ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER, LaunchTerminalSettingsDialog.this);
    		}

			ILauncherDelegate[] delegates = LauncherDelegateManager.getInstance().getLauncherDelegates(false);

    		if (UIPlugin.getTraceHandler().isSlotEnabled(0, ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER)) {
    			UIPlugin.getTraceHandler().trace("Got launcher delegates after " + (System.currentTimeMillis() - start) + " ms.", //$NON-NLS-1$ //$NON-NLS-2$
    												ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER, LaunchTerminalSettingsDialog.this);
    		}

    		for (ILauncherDelegate delegate : delegates) {
    			if (delegate.isHidden()) continue;
    			String label = delegate.getLabel();
    			if (label == null || "".equals(label.trim())) label = delegate.getId(); //$NON-NLS-1$
    			label2delegate.put(label, delegate);
    			items.add(label);
    		}
    	} else {
    		if (UIPlugin.getTraceHandler().isSlotEnabled(0, ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER)) {
    			UIPlugin.getTraceHandler().trace("Getting applicable launcher delegates after " + (System.currentTimeMillis() - start) + " ms.", //$NON-NLS-1$ //$NON-NLS-2$
    												ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER, LaunchTerminalSettingsDialog.this);
    		}

    		ILauncherDelegate[] delegates = LauncherDelegateManager.getInstance().getApplicableLauncherDelegates(selection);

    		if (UIPlugin.getTraceHandler().isSlotEnabled(0, ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER)) {
    			UIPlugin.getTraceHandler().trace("Got applicable launcher delegates after " + (System.currentTimeMillis() - start) + " ms.", //$NON-NLS-1$ //$NON-NLS-2$
    												ITraceIds.TRACE_LAUNCH_TERMINAL_COMMAND_HANDLER, LaunchTerminalSettingsDialog.this);
    		}

    		for (ILauncherDelegate delegate : delegates) {
    			if (delegate.isHidden()) continue;
    			String label = delegate.getLabel();
    			if (label == null || "".equals(label.trim())) label = delegate.getId(); //$NON-NLS-1$
    			label2delegate.put(label, delegate);
    			items.add(label);
    		}
    	}
    	Collections.sort(items);
    	combo.setItems(items.toArray(new String[items.size()]));
    }

    /* (non-Javadoc)
     * @see org.eclipse.tcf.te.ui.jface.interfaces.IValidatingContainer#validate()
     */
    @Override
    public void validate() {
    	IWizardConfigurationPanel panel = this.settings.getActiveConfigurationPanel();
    	Button okButton = getButton(IDialogConstants.OK_ID);
    	SWTControlUtil.setEnabled(okButton, panel.isValid());
    }

    /* (non-Javadoc)
     * @see org.eclipse.tcf.te.ui.jface.interfaces.IValidatingContainer#setMessage(java.lang.String, int)
     */
    @Override
    public void setMessage(String message, int messageType) {
    	if (settings != null) {
    		settings.setMessage(message, messageType);
    	}
    }

    /* (non-Javadoc)
     * @see org.eclipse.tcf.te.ui.jface.dialogs.CustomTrayDialog#saveWidgetValues()
     */
    @Override
    protected void saveWidgetValues() {
    	IDialogSettings settings = getDialogSettings();
    	if (settings != null) {
    		settings.put("terminalLabel", SWTControlUtil.getText(terminals)); //$NON-NLS-1$
    		this.settings.saveWidgetValues(settings, null);
    	}
    }

    /* (non-Javadoc)
     * @see org.eclipse.tcf.te.ui.jface.dialogs.CustomTrayDialog#restoreWidgetValues()
     */
    @Override
    protected void restoreWidgetValues() {
    	IDialogSettings settings = getDialogSettings();
    	if (settings != null) {
    		String terminalLabel = settings.get("terminalLabel"); //$NON-NLS-1$
    		int index = terminalLabel != null ? Arrays.asList(terminals.getItems()).indexOf(terminalLabel) : -1;
    		if (index != -1) {
    			terminals.select(index);
    			this.settings.showConfigurationPanel(terminals.getText());
    		}

    		this.settings.restoreWidgetValues(settings, null);
    	}
    }

    /* (non-Javadoc)
     * @see org.eclipse.tcf.te.ui.jface.dialogs.CustomTrayDialog#okPressed()
     */
    @Override
    protected void okPressed() {
    	IWizardConfigurationPanel panel = this.settings.getActiveConfigurationPanel();

    	if(!panel.isValid()){
			MessageBox mb = new MessageBox(getShell(), SWT.ICON_ERROR | SWT.OK);
			mb.setText(Messages.LaunchTerminalSettingsDialog_error_title);
			mb.setMessage(NLS.bind(Messages.LaunchTerminalSettingsDialog_error_invalidSettings, panel.getMessage() != null ? panel.getMessage() : Messages.LaunchTerminalSettingsDialog_error_unknownReason));
			mb.open();
			return;
    	}
    	data = new PropertiesContainer();

    	// Store the id of the selected delegate
    	data.setProperty(ITerminalsConnectorConstants.PROP_DELEGATE_ID, label2delegate.get(terminals.getText()).getId());
    	// Store the selection
    	data.setProperty(ITerminalsConnectorConstants.PROP_SELECTION, selection);

    	// Store the delegate specific settings
    	if (panel instanceof IDataExchangeNode) {
    		((IDataExchangeNode)panel).extractData(data);
    	}

        super.okPressed();
    }

    /**
     * Returns the configured terminal launcher settings.
     * <p>
     * The settings are extracted from the UI widgets once
     * OK got pressed.
     *
     * @return The configured terminal launcher settings or <code>null</code>.
     */
    public IPropertiesContainer getSettings() {
    	return data;
    }
}
