"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.startNamedPipeServer = startNamedPipeServer;
const fs = require("fs");
const net = require("net");
const collectExtractProps_1 = require("./requests/collectExtractProps");
const componentInfos_1 = require("./requests/componentInfos");
const getImportPathForFile_1 = require("./requests/getImportPathForFile");
const getPropertiesAtLocation_1 = require("./requests/getPropertiesAtLocation");
const getQuickInfoAtPosition_1 = require("./requests/getQuickInfoAtPosition");
const utils_1 = require("./utils");
async function startNamedPipeServer(ts, info, language, projectKind) {
    const server = net.createServer(connection => {
        connection.on('data', data => {
            const text = data.toString();
            if (text === 'ping') {
                connection.write('pong');
                return;
            }
            const request = JSON.parse(text);
            const fileName = request.args[0];
            const requestContext = {
                typescript: ts,
                languageService: info.languageService,
                languageServiceHost: info.languageServiceHost,
                language: language,
                isTsPlugin: true,
                getFileId: (fileName) => fileName,
            };
            if (request.type === 'containsFile') {
                sendResponse(info.project.containsFile(ts.server.toNormalizedPath(fileName)));
            }
            else if (request.type === 'projectInfo') {
                sendResponse({
                    name: info.project.getProjectName(),
                    kind: info.project.projectKind,
                    currentDirectory: info.project.getCurrentDirectory(),
                });
            }
            else if (request.type === 'collectExtractProps') {
                const result = collectExtractProps_1.collectExtractProps.apply(requestContext, request.args);
                sendResponse(result);
            }
            else if (request.type === 'getImportPathForFile') {
                const result = getImportPathForFile_1.getImportPathForFile.apply(requestContext, request.args);
                sendResponse(result);
            }
            else if (request.type === 'getPropertiesAtLocation') {
                const result = getPropertiesAtLocation_1.getPropertiesAtLocation.apply(requestContext, request.args);
                sendResponse(result);
            }
            else if (request.type === 'getQuickInfoAtPosition') {
                const result = getQuickInfoAtPosition_1.getQuickInfoAtPosition.apply(requestContext, request.args);
                sendResponse(result);
            }
            // Component Infos
            else if (request.type === 'getComponentProps') {
                const result = componentInfos_1.getComponentProps.apply(requestContext, request.args);
                sendResponse(result);
            }
            else if (request.type === 'getComponentEvents') {
                const result = componentInfos_1.getComponentEvents.apply(requestContext, request.args);
                sendResponse(result);
            }
            else if (request.type === 'getTemplateContextProps') {
                const result = componentInfos_1.getTemplateContextProps.apply(requestContext, request.args);
                sendResponse(result);
            }
            else if (request.type === 'getComponentNames') {
                const result = componentInfos_1.getComponentNames.apply(requestContext, request.args);
                sendResponse(result);
            }
            else if (request.type === 'getElementAttrs') {
                const result = componentInfos_1.getElementAttrs.apply(requestContext, request.args);
                sendResponse(result);
            }
            else {
                console.warn('[Vue Named Pipe Server] Unknown request type:', request.type);
            }
        });
        connection.on('error', err => console.error('[Vue Named Pipe Server]', err.message));
        function sendResponse(data) {
            connection.write(JSON.stringify(data ?? null) + '\n\n');
        }
    });
    for (let i = 0; i < 20; i++) {
        const path = (0, utils_1.getNamedPipePath)(projectKind, i);
        const socket = await (0, utils_1.connect)(path, 100);
        if (typeof socket === 'object') {
            socket.end();
        }
        const namedPipeOccupied = typeof socket === 'object' || socket === 'timeout';
        if (namedPipeOccupied) {
            continue;
        }
        const success = await tryListen(server, path);
        if (success) {
            break;
        }
    }
}
function tryListen(server, namedPipePath) {
    return new Promise(resolve => {
        const onSuccess = () => {
            server.off('error', onError);
            resolve(true);
        };
        const onError = (err) => {
            if (err.code === 'ECONNREFUSED') {
                try {
                    console.log('[Vue Named Pipe Client] Deleting:', namedPipePath);
                    fs.promises.unlink(namedPipePath);
                }
                catch { }
            }
            server.off('error', onError);
            server.close();
            resolve(false);
        };
        server.listen(namedPipePath, onSuccess);
        server.on('error', onError);
    });
}
//# sourceMappingURL=server.js.map