/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: AnnotationManager.java,v 1.3 2010/03/07 19:16:06 mtaal Exp $
 */

package org.eclipse.emf.texo.generator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.texo.annotations.annotationsmodel.ENamedElementAnnotation;

/**
 * During the annotation process takes care of managing all annotations and annotators.
 * 
 * Can be called to request for an annotation of a certain type (EClass) for an
 * {@link ENamedElement}. Will retrieve the annotations which have manually been created and will
 * call an annotator to complete the values of the annotation.
 * 
 * @see ModelController
 * @see Annotator
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 */
public class AnnotationManager {

  private List<ENamedElementAnnotation> initialAnnotations = new ArrayList<ENamedElementAnnotation>();
  private final Map<EModelElement, List<ENamedElementAnnotation>> modelAnnotations = new HashMap<EModelElement, List<ENamedElementAnnotation>>();

  private Map<EClass, Annotator<? extends ENamedElementAnnotation>> annotators = new HashMap<EClass, Annotator<? extends ENamedElementAnnotation>>();

  @SuppressWarnings("unchecked")
  public ENamedElementAnnotation getAnnotation(ENamedElement annotatedModelElement,
      EClass annotationEClass) {
    if (modelAnnotations.containsKey(annotatedModelElement)) {
      final List<ENamedElementAnnotation> annotationList = modelAnnotations
          .get(annotatedModelElement);
      for (ENamedElementAnnotation annotation : annotationList) {
        if (annotationEClass.isSuperTypeOf(annotation.eClass())) {
          return annotation;
        }
      }
    }

    // not found create, see if the initial list contains some
    ENamedElementAnnotation initialAnnotation = null;
    for (ENamedElementAnnotation annotation : initialAnnotations) {
      if (annotationEClass.isSuperTypeOf(annotation.eClass())) {
        initialAnnotation = annotation;
        break;
      }
    }
    // create a new annotation
    final ENamedElementAnnotation newElementAnnotation = (ENamedElementAnnotation) annotationEClass
        .getEPackage().getEFactoryInstance().create(annotationEClass);

    // if there is an initial annotation copy the values over
    if (initialAnnotation != null) {
      for (EStructuralFeature eFeature : annotationEClass.getEAllStructuralFeatures()) {
        if (initialAnnotation.eIsSet(eFeature)) {
          newElementAnnotation.eSet(eFeature, initialAnnotation.eGet(eFeature));
        }
      }
    }
    newElementAnnotation.setENamedElement(annotatedModelElement);

    // set this new annotation in the overall list so it can be found
    addAnnotation(newElementAnnotation);

    // find an annotator to do the rest
    if (annotators.containsKey(annotationEClass)) {
      ((Annotator<ENamedElementAnnotation>) annotators.get(annotationEClass))
          .setAnnotationFeatures(newElementAnnotation);
    }

    return newElementAnnotation;
  }

  public List<ENamedElementAnnotation> getAnnotations(ENamedElement namedElement) {
    List<ENamedElementAnnotation> annotations = modelAnnotations.get(namedElement);
    if (annotations == null) {
      return Collections.emptyList();
    }
    return annotations;
  }

  protected void addAnnotation(ENamedElementAnnotation annotation) {
    List<ENamedElementAnnotation> annotations = modelAnnotations.get(annotation.getENamedElement());
    if (annotations == null) {
      annotations = new ArrayList<ENamedElementAnnotation>();
      modelAnnotations.put(annotation.getENamedElement(), annotations);
    }
    annotations.add(annotation);
  }

  public void addAnnotator(Annotator<? extends ENamedElementAnnotation> annotator) {
    annotators.put(annotator.getAnnotationEClass(), annotator);
    annotator.setAnnotationManager(this);
  }

  @SuppressWarnings("unchecked")
  public Annotator<ENamedElementAnnotation> getAnnotator(EClass annotationEClass) {
    return (Annotator<ENamedElementAnnotation>) annotators.get(annotationEClass);
  }

  public Map<EModelElement, List<ENamedElementAnnotation>> getModelAnnotations() {
    return modelAnnotations;
  }

  public List<ENamedElementAnnotation> getInitialAnnotations() {
    return initialAnnotations;
  }

  public void setInitialAnnotations(List<ENamedElementAnnotation> initialAnnotations) {
    this.initialAnnotations = initialAnnotations;
  }
}
