/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: WorkFlowStarter.java,v 1.10 2010/03/11 02:18:47 mtaal Exp $
 */

package org.eclipse.emf.texo.generator;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.mwe.core.WorkflowRunner;
import org.eclipse.emf.mwe.core.issues.Issues;
import org.eclipse.emf.mwe.core.issues.IssuesImpl;
import org.eclipse.emf.mwe.core.issues.MWEDiagnostic;
import org.eclipse.emf.mwe.core.monitor.ProgressMonitor;
import org.eclipse.emf.mwe.core.monitor.ProgressMonitorAdapter;

/**
 * Generates code for a specific Eclipse project and stores the result in that project..
 * 
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 */
public class WorkFlowStarter {

  public static final String MODEL_CONTROLLER_CONTEXT_SLOT = "modelController"; //$NON-NLS-1$
  public static final String PARAMETER_SOURCE_FOLDER = "defaultSourceFolder"; //$NON-NLS-1$
  public static final String PARAMETER_PROJECT_NAME = "projectName"; //$NON-NLS-1$

  private IProgressMonitor monitor = new org.eclipse.core.runtime.NullProgressMonitor();

  /**
   * Generate model code for a set of EPackages. The generated code is stored and merged with the
   * already present code in the project.
   * 
   * @param projectName
   *          the name of the project in which the generated code is stored
   * @param ePackages
   *          the ePackages to generate the model code for
   * @param modelController
   *          the object controlling models and annotations
   * @param defaultSourceFolder
   *          the defaultSourceFolder is used to set outlets which have a ${defaultSourceFolder}
   *          parameters, is the folder within the target project
   */
  public void generateModelCode(final String projectName, final List<EPackage> ePackages,
      final ModelController modelController, final String defaultSourceFolder) {
    final Map<String, String> parameters = new HashMap<String, String>();
    final Map<String, Object> slotContents = new HashMap<String, Object>();
    slotContents.put(MODEL_CONTROLLER_CONTEXT_SLOT, modelController);
    parameters.put(PARAMETER_SOURCE_FOLDER, defaultSourceFolder);

    // not very nice to branch like this to support 2 versions
    if (isXtendHelios()) {
      runWorkFlow(projectName, "/org/eclipse/emf/texo/generator/generate_model_36.oaw", //$NON-NLS-1$
          parameters, slotContents);
    } else {
      runWorkFlow(projectName, "/org/eclipse/emf/texo/generator/generate_model.oaw", //$NON-NLS-1$
          parameters, slotContents);
    }
  }

  private boolean isXtendHelios() {
    try {
      this.getClass().getClassLoader().loadClass("org.eclipse.xtend.type.impl.java.JavaMetaModel"); //$NON-NLS-1$
      return true;
    } catch (ClassNotFoundException e) {
      return false;
    }
  }

  /**
   * Run an oaw workflow which outputs its result to a project denoted by the projectName.
   * 
   * @param monitor
   *          the monitor displaying the progress
   * @param projectName
   *          the name of the project which is the output of the generated artifacts
   * @param templatePath
   *          the resource path to the template to run
   * @param parameters
   *          workflow parameters
   * @param slotContents
   *          workflow slot contents
   */
  public void runWorkFlow(final String projectName, final String templatePath,
      final Map<String, String> parameters, final Map<String, Object> slotContents) {
    try {
      final IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
      parameters.put("projectName", projectName); //$NON-NLS-1$

      // line below can be of interest when getting into classloading
      // issues
      // injecting the classloader associated with the current class
      // ResourceLoaderFactory.setCurrentThreadResourceLoader(new
      // ResourceLoaderImpl(getClass().getClassLoader()));

      // refresh upfront so that the code generation does not encounter
      // unsynchronised resources
      // pass a null progress monitor otherwise the progress gets confused
      project.refreshLocal(IResource.DEPTH_INFINITE,
          new org.eclipse.core.runtime.NullProgressMonitor());

      final Issues issues = runWorkFlow(templatePath, parameters, slotContents);

      if (issues == null) {
        throw new IllegalStateException("Could not perform requested action for unknown reason."); //$NON-NLS-1$
      } else if (issues.getErrors().length > 0) {
        final StringBuilder sb = new StringBuilder();
        for (final MWEDiagnostic mweDiagnostic : issues.getErrors()) {
          if (sb.length() > 0) {
            sb.append("\n"); //$NON-NLS-1$
          }
          sb.append(getDiagnosticMessages(mweDiagnostic, "")); //$NON-NLS-1$
        }
        throw new ArtifactGenerationException("Errors found during artifact generations\n" //$NON-NLS-1$
            + "The following error(s) where encountered:\n" //$NON-NLS-1$
            + sb.toString(), issues);
      }

      project.refreshLocal(IResource.DEPTH_INFINITE,
          new org.eclipse.core.runtime.NullProgressMonitor());
    } catch (final CoreException e) {
      throw new IllegalStateException(e);
    }
  }

  /**
   * Runs the work flow in the work flow file using the parameters and slot content.
   * 
   * @param workFlowFile
   *          the file containing the work flow definition
   * @param monitor
   *          the eclipse progressmonitor, note: maybe passed in as null
   * @param parameters
   *          work flow parameters
   * @param slotContents
   *          the slots with data
   * @return issues found
   */
  public Issues runWorkFlow(final String workFlowFile, final Map<String, String> parameters,
      final Map<String, ?> slotContents) {
    final ProgressMonitor progressMonitor = new ProgressMonitorAdapter(getMonitor());
    final WorkflowRunner wfr = new WorkflowRunner();
    if (!wfr.prepare(workFlowFile, progressMonitor, parameters)) {
      // apparently something went wrong...
      return null;
    }
    final Issues issues = new IssuesImpl();
    wfr.executeWorkflow(slotContents, issues);
    return issues;
  }

  private String getDiagnosticMessages(final Diagnostic diagnostic, final String prefix) {
    final StringBuilder sb = new StringBuilder();
    sb.append(prefix + " " + diagnostic.getSource() + //$NON-NLS-1$ 
        ": " //$NON-NLS-1$
        + diagnostic.getMessage());
    for (final Diagnostic child : diagnostic.getChildren()) {
      sb.append("\n" + prefix //$NON-NLS-1$
          + getDiagnosticMessages(child, prefix + ">")); //$NON-NLS-1$
    }
    return sb.toString();
  }

  public IProgressMonitor getMonitor() {
    return monitor;
  }

  public void setMonitor(IProgressMonitor monitor) {
    this.monitor = monitor;
  }
}
