/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: ModelEClassAnnotator.java,v 1.9 2010/03/14 00:57:33 mtaal Exp $
 */

package org.eclipse.emf.texo.modelgenerator.annotator;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.eclipse.emf.texo.annotations.annotationsmodel.EStructuralFeatureAnnotation;
import org.eclipse.emf.texo.generator.Annotator;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EClassModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EPackageModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EStructuralFeatureModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.ModelcodegeneratorPackage;
import org.eclipse.emf.texo.utils.Check;
import org.eclipse.emf.texo.utils.ModelUtils;

/**
 * Responsible for setting the values in a {@link EClassModelGenAnnotation}.
 * 
 * @author <a href="mailto:mtaal@elver.org">Martin Taal</a>
 * @version $Revision: 1.9 $
 */

public class ModelEClassAnnotator extends ModelEClassifierAnnotator implements
    Annotator<EClassModelGenAnnotation> {

  private static final String SERIALIZABLE_NAME = java.io.Serializable.class.getName();

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.texo.generator.Annotator#getAnnotationEClass()
   */
  @Override
  public EClass getAnnotationEClass() {
    return ModelcodegeneratorPackage.eINSTANCE.getEClassModelGenAnnotation();
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.emf.texo.generator.Annotator#annotate(org.eclipse.emf.texo.annotations.annotationsmodel
   * .ENamedElementAnnotation)
   */
  public void setAnnotationFeatures(EClassModelGenAnnotation annotation) {

    Check.isNotNull(annotation.getEClass(), "eClass must be set");//$NON-NLS-1$

    final EClass eClass = annotation.getEClass();

    super.annotate(annotation);

    if (annotation.getClassImplements().isEmpty()) {
      annotation.getClassImplements().addAll(getClassImplements(eClass));
    }
    final List<String> classExtends = getClassExtends(eClass);
    if (annotation.getClassExtends().isEmpty()) {
      annotation.getClassExtends().addAll(classExtends);
    }
    if (annotation.getQualifiedClassName() == null) {
      if (!ModelUtils.isEMap(eClass)) {
        annotation.setQualifiedClassName(getQualifiedClassName(eClass));
      }
    }
    if (annotation.getSimpleClassName() == null) {
      annotation.setSimpleClassName(getSimpleClassName(eClass));
    }
    if (!annotation.isSetAbstract()) {
      annotation.setAbstract(eClass.isAbstract());
    }
    if (!annotation.isSetHasSuperEClass()) {
      final EClass superEClass = getSuperClassEClass(eClass);
      annotation.setHasSuperEClass(superEClass != null);
      if (superEClass != null) {
        annotation.setSuperEClass(getEClassModelGenAnnotation(superEClass));
      }
    }

    if (!annotation.isSetHasSuperEClass()) {
      annotation.setHasSuperEClass(classExtends.size() > 0);
    }
    if (!annotation.isSetSerializable()) {
      annotation.setSerializable(isJavaSerializable(eClass));
    }

    if (!annotation.isSetGenerateCode()) {
      final boolean doGenerateCode = !ModelUtils.isEMap(eClass);
      annotation.setGenerateCode(doGenerateCode);
    }

    final EPackageModelGenAnnotation ePackageAnnotation = getEPackageModelGenAnnotation(eClass
        .getEPackage());
    annotation.setOwnerEPackageAnnotation(ePackageAnnotation);

    if (!annotation.isSetHasManyFeatures()) {
      final List<EStructuralFeature> features = new ArrayList<EStructuralFeature>();

      if (annotation.getEStructuralFeatureAnnotations().isEmpty()) {
        // fill the subfeatures
        for (final EStructuralFeature eStructuralFeature : eClass.getEStructuralFeatures()) {
          features.add(eStructuralFeature);
        }
        // also add the feature annotations of interfaces of supertypes
        for (final EStructuralFeature eStructuralFeature : getAllInterfaceFeatures(eClass)) {
          features.add(eStructuralFeature);
        }
      } else {
        for (EStructuralFeatureAnnotation featureAnnotation : annotation
            .getEStructuralFeatureAnnotations()) {
          features.add(featureAnnotation.getEStructuralFeature());
        }
      }

      for (EStructuralFeature feature : features) {
        if (!feature.isVolatile() && feature.isMany()) {
          annotation.setHasManyFeatures(true);
          break;
        }
      }
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @seeorg.eclipse.emf.texo.generator.Annotator#postAnnotating(org.eclipse.emf.texo.annotations.
   * annotationsmodel.ENamedElementAnnotation)
   */
  public void postAnnotating(EClassModelGenAnnotation annotation) {
    final EClass eClass = annotation.getEClass();
    if (annotation.getEStructuralFeatureAnnotations().isEmpty()) {
      // fill the subfeatures
      for (final EStructuralFeature eStructuralFeature : eClass.getEStructuralFeatures()) {
        annotation.getEStructuralFeatureAnnotations().add(
            getEStructuralFeatureModelGenAnnotation(eStructuralFeature));
      }
      // also add the feature annotations of interfaces of supertypes
      for (final EStructuralFeature eStructuralFeature : getAllInterfaceFeatures(eClass)) {
        annotation.getEStructuralFeatureAnnotations().add(
            getEStructuralFeatureModelGenAnnotation(eStructuralFeature));
      }

      for (EStructuralFeatureAnnotation featureAnnotation : annotation
          .getEStructuralFeatureAnnotations()) {
        if (FeatureMapUtil.isFeatureMap(featureAnnotation.getEStructuralFeature())) {
          annotation.getFeatureMapFeatures().add(
              (EStructuralFeatureModelGenAnnotation) featureAnnotation);
        }
      }
    }
  }

  private List<EStructuralFeature> getAllInterfaceFeatures(final EClass eLocalClass) {
    final List<EStructuralFeature> list = new ArrayList<EStructuralFeature>();
    for (final EClass superEClass : eLocalClass.getESuperTypes()) {
      if (superEClass.isInterface()) {
        list.addAll(superEClass.getEStructuralFeatures());
        list.addAll(getAllInterfaceFeatures(superEClass));
      }
    }
    return list;
  }

  /**
   * Creates the extends string used in the java class/interface, the returned value includes the
   * term 'extends'
   * 
   * @return the extends statement used in the java class declaration
   */
  private List<String> getClassExtends(EClass eClass) {
    final List<String> result = new ArrayList<String>();
    if (eClass.isInterface()) {
      for (final EClass superEClass : eClass.getESuperTypes()) {
        final EClassModelGenAnnotation annotation = getEClassModelGenAnnotation(superEClass);
        result.add(annotation.getQualifiedClassName());
      }
      return result;
    }

    final EClass superEClass = getSuperClassEClass(eClass);
    if (superEClass == null) {
      return result;
    }
    final EClassModelGenAnnotation annotation = getEClassModelGenAnnotation(superEClass);
    result.add(annotation.getQualifiedClassName());
    return result;
  }

  /**
   * Gets all the interfaces defined as super EClass ({@see EClass#getESuperTypes()}) and returns
   * their qualified interface names.
   * 
   * @return the list of qualified interface names
   */
  private List<String> getClassImplements(EClass eClass) {
    final List<EClass> superTypes = eClass.getESuperTypes();
    final List<String> interfaces = new ArrayList<String>();
    for (final EClass superEClass : superTypes) {
      if (superEClass.isInterface()) {
        final EClassModelGenAnnotation annotation = getEClassModelGenAnnotation(superEClass);
        interfaces.add(annotation.getQualifiedClassName());
      }
    }
    return interfaces;
  }

  /**
   * @return the first non-interface super EClass, if none there then null is returned
   */
  private EClass getSuperClassEClass(EClass eClass) {
    final List<EClass> superTypes = eClass.getESuperTypes();
    if (superTypes.isEmpty()) {
      return null;
    }
    for (final EClass superEClass : superTypes) {
      if (!superEClass.isInterface()) {
        return superEClass;
      }
    }
    return null;
  }

  /**
   * @return true if this EClass or one of its supertypes implements java.io.Serializable
   */
  private boolean isJavaSerializable(EClass eClass) {
    // if (getGenEPackage().getOptionClassImplements() != null &&
    // getGenEPackage().getOptionClassImplements().compareTo(SERIALIZABLE_NAME)
    // == 0) {
    // return true;
    // }

    // ask the own implements list
    for (final String interf : getClassImplements(eClass)) {
      if (interf.compareTo(SERIALIZABLE_NAME) == 0) {
        return true;
      }
    }

    for (final EClass superEClass : eClass.getEAllSuperTypes()) {
      final EClassModelGenAnnotation annotation = getEClassModelGenAnnotation(superEClass);
      if (annotation.isSerializable()) {
        return true;
      }
    }

    return false;
  }
}